// Google Analytics Utility Functions

// Track page views
export const trackPageView = (pagePath, pageTitle) => {
  if (typeof window !== 'undefined' && window.gtag) {
    window.gtag('event', 'page_view', {
      page_path: pagePath,
      page_title: pageTitle
    });
  }
};

// Track custom events
export const trackEvent = (eventName, eventParams = {}) => {
  if (typeof window !== 'undefined' && window.gtag) {
    window.gtag('event', eventName, eventParams);
  }
};

// Track button clicks
export const trackButtonClick = (buttonName, buttonLocation) => {
  trackEvent('button_click', {
    button_name: buttonName,
    button_location: buttonLocation
  });
};

// Track form submissions
export const trackFormSubmission = (formName, formData = {}) => {
  trackEvent('form_submission', {
    form_name: formName,
    ...formData
  });
};

// Track conversions (enrollments)
export const trackConversion = (courseName, conversionValue = 0) => {
  trackEvent('conversion', {
    course_name: courseName,
    value: conversionValue,
    currency: 'USD'
  });
  
  // Also track as purchase event for Google Ads conversion tracking
  trackEvent('purchase', {
    transaction_id: `enrollment_${Date.now()}`,
    value: conversionValue,
    currency: 'USD',
    items: [{
      item_name: courseName,
      item_category: 'course',
      price: conversionValue
    }]
  });
};

// Get UTM parameters from URL
export const getUtmParams = () => {
  if (typeof window === 'undefined') return {};
  
  const urlParams = new URLSearchParams(window.location.search);
  return {
    utm_source: urlParams.get('utm_source') || '',
    utm_medium: urlParams.get('utm_medium') || '',
    utm_campaign: urlParams.get('utm_campaign') || '',
    utm_content: urlParams.get('utm_content') || '',
    utm_term: urlParams.get('utm_term') || ''
  };
};

// Store UTM parameters in session storage
export const storeUtmParams = () => {
  if (typeof window === 'undefined') return;
  
  const utmParams = getUtmParams();
  
  // Only store if at least one UTM parameter exists
  if (Object.values(utmParams).some(value => value !== '')) {
    sessionStorage.setItem('utm_params', JSON.stringify(utmParams));
  }
};

// Get stored UTM parameters
export const getStoredUtmParams = () => {
  if (typeof window === 'undefined') return {};
  
  const stored = sessionStorage.getItem('utm_params');
  return stored ? JSON.parse(stored) : {};
};

// Track external link clicks
export const trackExternalLink = (url, linkText) => {
  trackEvent('click', {
    event_category: 'external_link',
    event_label: linkText,
    value: url
  });
};

// Track video plays (if you add videos)
export const trackVideoPlay = (videoName) => {
  trackEvent('video_start', {
    video_name: videoName
  });
};

// Track scroll depth
export const trackScrollDepth = (depth) => {
  trackEvent('scroll', {
    percent_scrolled: depth
  });
};

// Generate UTM tracking URL
export const generateTrackingUrl = (baseUrl, source, medium, campaign, content = '', term = '') => {
  const url = new URL(baseUrl);
  url.searchParams.set('utm_source', source);
  url.searchParams.set('utm_medium', medium);
  url.searchParams.set('utm_campaign', campaign);
  if (content) url.searchParams.set('utm_content', content);
  if (term) url.searchParams.set('utm_term', term);
  return url.toString();
};
