import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { 
  Calendar, Clock, Video, Users, Globe, Sparkles, TrendingUp, 
  Award, CheckCircle2, Download, ArrowRight, Zap, Target, Star,
  Play, Film, Clapperboard, Scissors, Wand2, FileText, BookOpen, Shield,
  Volume2, Subtitles, Palette, Repeat
} from 'lucide-react';
import CourseLeadForm from '../components/CourseLeadForm';
import Chatbot from '../components/Chatbot';

const WorkshopVideoEditing = () => {
  const navigate = useNavigate();
  const [timeLeft, setTimeLeft] = useState({ days: 0, hours: 0, minutes: 0, seconds: 0 });

  useEffect(() => {
    // Workshop starts December 6, 2025 at 12:00 AM EST
    const workshopDate = new Date('2025-12-06T05:00:00Z'); // 12:00 AM EST = 05:00 UTC
    const targetDate = workshopDate.getTime();

    const updateCountdown = () => {
      const now = new Date().getTime();
      const distance = targetDate - now;

      if (distance > 0) {
        setTimeLeft({
          days: Math.floor(distance / (1000 * 60 * 60 * 24)),
          hours: Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60)),
          minutes: Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60)),
          seconds: Math.floor((distance % (1000 * 60)) / 1000)
        });
      } else {
        setTimeLeft({ days: 0, hours: 0, minutes: 0, seconds: 0 });
      }
    };

    updateCountdown();
    const interval = setInterval(updateCountdown, 1000);
    return () => clearInterval(interval);
  }, []);

  const scrollToForm = () => {
    const formElement = document.getElementById('registration-form');
    if (formElement) {
      formElement.scrollIntoView({ behavior: 'smooth', block: 'start' });
    }
  };

  const scrollToPricing = () => {
    const pricingElement = document.getElementById('pricing-section');
    if (pricingElement) {
      pricingElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
    }
  };

  const modules = [
    'Understanding AI video editing workflows',
    'Cutting & trimming using auto-scene detection',
    'AI-generated transitions & templates',
    'Auto-captioning & subtitle styling',
    'AI-driven color correction & filters',
    'Professional storytelling using AI scripts',
    'Auto-resizing for reels, shorts & YouTube',
    'Adding AI voiceovers & improved audio'
  ];

  const stats = [
    { value: '82%', label: 'Of all online content is video' },
    { value: '4-6x', label: 'Faster editing with AI tools' },
    { value: '60-320%', label: 'Engagement boost from Reels/Shorts' },
    { value: '2x', label: 'Higher conversions with video ads' }
  ];

  const tools = [
    { name: 'CapCut AI', icon: '🎬' },
    { name: 'Descript', icon: '🎙️' },
    { name: 'InVideo AI', icon: '📹' },
    { name: 'VEED.io', icon: '✨' },
    { name: 'Runway ML', icon: '🚀' },
    { name: 'OpusClip', icon: '✂️' }
  ];

  const projects = [
    'Create 3 viral-style reels from raw footage',
    'Add AI subtitles, captions & animations',
    'Generate a script + video using AI',
    'Convert long video into 10 short clips',
    'Build a mini showreel portfolio'
  ];

  return (
    <div className="min-h-screen bg-white">
      {/* Navigation */}
      <nav className="fixed top-4 left-1/2 transform -translate-x-1/2 z-50 w-[95%] max-w-7xl">
        <div className="bg-black/30 backdrop-blur-lg rounded-full border border-white/20 px-8 py-3 shadow-2xl">
          <div className="flex items-center justify-between">
            <button onClick={() => navigate('/')} className="text-white font-bold text-lg hover:text-yellow-400 transition-colors">
              AI School for All
            </button>
            <div className="hidden md:flex items-center gap-6">
              <button onClick={() => navigate('/#home')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Home</button>
              <button onClick={() => navigate('/#workshops')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Workshops</button>
              <button 
                onClick={scrollToForm}
                className="bg-yellow-400 hover:bg-yellow-500 text-gray-900 px-6 py-2 rounded-full font-bold transition-all text-sm"
              >
                Register Now
              </button>
            </div>
          </div>
        </div>
      </nav>

      {/* VIBRANT VIDEO-THEMED HERO */}
      <section className="relative min-h-screen flex items-center justify-center overflow-hidden pt-20">
        {/* Dynamic Video Background */}
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1574717024653-61fd2cf4d44d?w=1920&h=1080&fit=crop&q=80"
            alt="Video Editing Workshop"
            className="w-full h-full object-cover brightness-110"
          />
          <div className="absolute inset-0 bg-gradient-to-br from-purple-600/75 via-pink-600/70 to-blue-600/75"></div>
        </div>

        {/* Animated overlay patterns */}
        <div className="absolute inset-0 opacity-10 z-10">
          <div className="absolute top-20 left-20 w-72 h-72 bg-white rounded-full blur-3xl animate-pulse"></div>
          <div className="absolute bottom-20 right-20 w-96 h-96 bg-white rounded-full blur-3xl animate-pulse delay-1000"></div>
          <div className="absolute top-1/2 left-1/2 w-64 h-64 bg-white rounded-full blur-3xl animate-pulse delay-500"></div>
        </div>

        <div className="container mx-auto px-6 relative z-20">
          <div className="max-w-5xl mx-auto text-center">
            {/* Event Badge */}
            <div className="inline-flex items-center gap-2 bg-white/20 backdrop-blur-md px-6 py-3 rounded-full mb-8 border border-white/30">
              <Film className="w-5 h-5 text-white" />
              <span className="text-white font-bold text-sm uppercase tracking-wider">5-Hour Intensive Workshop</span>
            </div>

            {/* Main Title */}
            <h1 className="text-5xl md:text-7xl font-black text-white mb-6 leading-tight drop-shadow-2xl">
              AI Video Editing &
              <br />
              <span className="text-yellow-400">Reels Creation</span>
            </h1>

            <p className="text-2xl text-white/90 mb-10 max-w-3xl mx-auto font-medium">
              Create viral reels, shorts, and professional videos in minutes using AI-powered tools
            </p>

            {/* Event Quick Info */}
            <div className="grid md:grid-cols-4 gap-4 mb-12 max-w-4xl mx-auto">
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Calendar className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">6 Dec 2025</p>
                <p className="text-white/80 text-sm">Saturday</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Clock className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">5 Hours</p>
                <p className="text-white/80 text-sm">Intensive</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Video className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">Live Online</p>
                <p className="text-white/80 text-sm">Interactive</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Zap className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">AI-Powered</p>
                <p className="text-white/80 text-sm">Fast & Easy</p>
              </div>
            </div>

            {/* CTA */}
            <button 
              onClick={scrollToForm}
              className="bg-yellow-400 hover:bg-yellow-500 text-gray-900 px-12 py-6 rounded-2xl font-black text-xl shadow-2xl hover:scale-105 transition-all inline-flex items-center gap-3"
            >
              <Play className="w-6 h-6" />
              Start Your Video Journey
              <ArrowRight className="w-6 h-6" />
            </button>

            <p className="text-white/90 mt-6 text-sm">
              <strong>Enrollment Begins Soon</strong> - Limited seats available!
            </p>
          </div>
        </div>
      </section>

      {/* COUNTDOWN TIMER */}
      <section className="py-16 px-6 bg-gradient-to-b from-purple-900 to-pink-900">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-8">
            <h3 className="text-3xl font-bold text-yellow-400 mb-2">Workshop Starts In</h3>
            <p className="text-gray-200">December 6, 2025 • 12:00 AM EST</p>
          </div>
          <div className="grid grid-cols-4 gap-4 max-w-2xl mx-auto">
            {['days', 'hours', 'minutes', 'seconds'].map((unit) => (
              <div key={unit} className="bg-gradient-to-br from-purple-700 to-pink-700 rounded-xl p-6 text-center border-2 border-yellow-400/30">
                <div className="text-5xl font-bold text-yellow-400 mb-2">{timeLeft[unit]}</div>
                <div className="text-gray-200 text-sm uppercase tracking-wide">{unit}</div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Promotional Banner - Limited Time Deal */}
      <section className="py-6 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <button
            onClick={scrollToPricing}
            className="w-full bg-gradient-to-r from-orange-500 to-red-500 hover:from-orange-600 hover:to-red-600 text-white px-8 py-5 rounded-full font-bold text-lg shadow-xl hover:shadow-2xl transition-all hover:scale-105 flex items-center justify-center gap-3"
          >
            <span className="text-2xl">🔥</span>
            <span>Limited-Time Deal — 50% OFF</span>
            <ArrowRight className="w-6 h-6" />
          </button>
        </div>
      </section>

      {/* WHY THIS WORKSHOP */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="grid md:grid-cols-2 gap-12 items-center">
            <div>
              <div className="inline-flex items-center gap-2 bg-purple-100 px-4 py-2 rounded-full mb-6">
                <Clapperboard className="w-5 h-5 text-purple-600" />
                <span className="text-purple-900 font-bold text-sm">CREATE VIRAL CONTENT</span>
              </div>
              <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-6">Why Video Rules Digital Marketing?</h2>
              <p className="text-lg text-gray-700 mb-6 leading-relaxed">
                Video content rules the digital world—reels, shorts, YouTube videos, product demos, corporate explainers, testimonials, and ads. But traditional editing is slow, technical, and time-consuming.
              </p>
              <p className="text-lg text-gray-700 mb-6 leading-relaxed">
                In this workshop, you'll learn how to create <strong className="text-purple-600">stunning videos in minutes</strong> using AI tools that automate cutting, trimming, captioning, transitions, color grading, voiceovers, and even storytelling.
              </p>
              <p className="text-lg text-gray-700 leading-relaxed">
                By the end, you'll produce <strong className="text-pink-600">polished videos with cinematic aesthetics</strong> and stay ahead in the content economy.
              </p>
            </div>
            <div className="relative">
              <div className="bg-gradient-to-br from-purple-500 to-pink-500 rounded-3xl p-8 shadow-2xl">
                <img 
                  src="https://images.unsplash.com/photo-1574717024653-61fd2cf4d44d?w=600&h=400&fit=crop&q=80"
                  alt="Video Editing"
                  className="w-full h-80 object-cover rounded-2xl shadow-xl"
                />
                <div className="absolute -bottom-6 -right-6 bg-white rounded-2xl p-6 shadow-2xl border-4 border-yellow-400">
                  <div className="text-center">
                    <div className="text-4xl font-bold text-purple-600">4-6x</div>
                    <div className="text-sm text-gray-700 font-medium">Faster Editing</div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* BEFORE/AFTER VIDEO EXAMPLES */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">See The AI Video Magic</h2>
            <p className="text-xl text-gray-600">Transform raw footage into viral content</p>
          </div>

          {/* Example 1: Raw to Polished Reel */}
          <div className="mb-16">
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">Raw Footage to Viral Reel</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BEFORE
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl bg-gray-900">
                  <img 
                    src="https://images.unsplash.com/photo-1492691527719-9d1e07e534b4?w=600&h=800&fit=crop&q=50&sat=-60"
                    alt="Raw video footage"
                    className="w-full h-96 object-cover opacity-70"
                  />
                  <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2">
                    <Play className="w-16 h-16 text-white/50" />
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Raw footage • No edits • Shaky • Boring</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  AFTER AI EDIT
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl">
                  <img 
                    src="https://images.unsplash.com/photo-1492691527719-9d1e07e534b4?w=600&h=800&fit=crop&q=90"
                    alt="Edited viral reel"
                    className="w-full h-96 object-cover"
                  />
                  <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2">
                    <div className="bg-gradient-to-r from-purple-600 to-pink-600 rounded-full p-4">
                      <Play className="w-12 h-12 text-white" />
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Smooth cuts • Auto-captions • Color graded • Viral ready!</p>
              </div>
            </div>
          </div>

          {/* Example 2: Long Video to Short Clips */}
          <div className="mb-16">
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">Long Video → Multiple Short Clips</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BEFORE
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl">
                  <div className="bg-gray-800 p-8 h-96 flex items-center justify-center">
                    <div className="text-center text-white">
                      <Film className="w-20 h-20 mx-auto mb-4 text-gray-400" />
                      <p className="text-2xl font-bold mb-2">45 Min YouTube Video</p>
                      <p className="text-gray-400">Too long for social media</p>
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Long format • Manual cutting needed • Time consuming</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  AFTER AI EDIT
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl bg-gradient-to-br from-purple-100 to-pink-100">
                  <div className="p-6 h-96">
                    <div className="grid grid-cols-3 gap-3 h-full">
                      {[1,2,3,4,5,6,7,8,9].map((i) => (
                        <div key={i} className="bg-gradient-to-br from-purple-500 to-pink-500 rounded-lg flex items-center justify-center relative">
                          <Play className="w-6 h-6 text-white" />
                          <span className="absolute bottom-1 right-1 text-xs text-white font-bold">:{i*15}s</span>
                        </div>
                      ))}
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">AI auto-cut into 9 viral clips • Perfect for Reels/Shorts!</p>
              </div>
            </div>
          </div>

          {/* Example 3: Auto-Subtitles */}
          <div>
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">AI Auto-Captioning & Styling</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BEFORE
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl">
                  <img 
                    src="https://images.unsplash.com/photo-1626814026160-2237a95fc5a0?w=600&h=600&fit=crop&q=80"
                    alt="Video without captions"
                    className="w-full h-96 object-cover"
                  />
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">No subtitles • Hard to watch on mute • Low engagement</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  AFTER AI EDIT
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl relative">
                  <img 
                    src="https://images.unsplash.com/photo-1626814026160-2237a95fc5a0?w=600&h=600&fit=crop&q=80"
                    alt="Video with AI captions"
                    className="w-full h-96 object-cover"
                  />
                  {/* Simulated captions */}
                  <div className="absolute bottom-20 left-0 right-0 text-center px-8">
                    <div className="bg-black/80 backdrop-blur-sm rounded-xl p-4 inline-block">
                      <p className="text-white text-xl font-bold">AI generates perfect captions</p>
                    </div>
                  </div>
                  <div className="absolute bottom-8 left-0 right-0 text-center">
                    <div className="bg-yellow-400 text-gray-900 rounded-lg px-6 py-2 inline-block font-bold">
                      automatically in seconds
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Auto-generated • Styled captions • 80% more engagement!</p>
              </div>
            </div>
          </div>

          {/* Results Banner */}
          <div className="mt-16 bg-gradient-to-r from-purple-600 to-pink-600 rounded-3xl p-10 text-center text-white">
            <p className="text-3xl font-bold mb-4">Create All These In Just 5 Hours!</p>
            <p className="text-xl text-white/90">No technical skills needed - AI does the work</p>
          </div>
        </div>
      </section>

      {/* INDUSTRY INSIGHTS */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Industry Insights</h2>
            <p className="text-xl text-gray-600">Why AI video editing is the future</p>
          </div>
          <div className="grid md:grid-cols-4 gap-8">
            {stats.map((stat, index) => (
              <div key={index} className="bg-gradient-to-br from-purple-50 to-pink-50 rounded-2xl p-8 shadow-lg hover:shadow-xl transition-all border-2 border-purple-100">
                <div className="text-5xl font-black text-purple-600 mb-4">{stat.value}</div>
                <p className="text-gray-700 font-medium leading-relaxed">{stat.label}</p>
              </div>
            ))}
          </div>
          <div className="mt-12 text-center">
            <p className="text-lg text-gray-600">
              <span className="text-purple-600 font-bold">Freelance video editors earn</span> <strong>₹40,000 to ₹2,00,000+ per month</strong>
            </p>
          </div>
        </div>
      </section>

      {/* WHAT YOU'LL LEARN */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">What You'll Master</h2>
            <div className="w-32 h-1 bg-gradient-to-r from-purple-500 to-pink-500 mx-auto"></div>
          </div>
          
          {/* Video Editing Gallery */}
          <div className="grid md:grid-cols-3 gap-4 mb-12">
            <img 
              src="https://images.unsplash.com/photo-1574717024653-61fd2cf4d44d?w=400&h=300&fit=crop&q=80"
              alt="Video editing workspace"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
            <img 
              src="https://images.unsplash.com/photo-1626814026160-2237a95fc5a0?w=400&h=300&fit=crop&q=80"
              alt="Content creation"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
            <img 
              src="https://images.unsplash.com/photo-1492691527719-9d1e07e534b4?w=400&h=300&fit=crop&q=80"
              alt="Video recording"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
          </div>

          <div className="grid md:grid-cols-2 gap-6">
            {modules.map((module, index) => (
              <div key={index} className="flex items-start gap-4 bg-gradient-to-r from-purple-50 to-pink-50 rounded-xl p-6 hover:shadow-lg transition-all">
                <div className="w-10 h-10 rounded-full bg-gradient-to-br from-purple-600 to-pink-600 flex items-center justify-center flex-shrink-0">
                  <CheckCircle2 className="w-6 h-6 text-white" />
                </div>
                <p className="text-gray-800 text-lg font-medium">{module}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* WHO SHOULD JOIN */}
      <section className="py-20 px-6 bg-gradient-to-br from-purple-700 to-pink-700 text-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold mb-4">Perfect For</h2>
            <p className="text-xl text-purple-100">This workshop is designed for...</p>
          </div>
          <div className="grid md:grid-cols-4 gap-6">
            {['Content Creators', 'Influencers', 'Freelancers', 'Digital Marketers', 
              'Coaches & Trainers', 'Students', 'Business Owners', 'Anyone Creating Videos'].map((role, index) => (
              <div key={index} className="bg-white/10 backdrop-blur-sm rounded-xl p-6 text-center hover:bg-white/20 transition-all border border-yellow-400/30">
                <Star className="w-8 h-8 text-yellow-400 mx-auto mb-3" />
                <p className="font-semibold text-lg">{role}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* HANDS-ON PROJECTS */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Hands-On Projects</h2>
            <p className="text-xl text-gray-600">Build your video portfolio live</p>
          </div>

          {/* Project Gallery */}
          <div className="grid md:grid-cols-4 gap-4 mb-12">
            <img 
              src="https://images.unsplash.com/photo-1611162617474-5b21e879e113?w=300&h=300&fit=crop&q=80"
              alt="Reel creation"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1626814026160-2237a95fc5a0?w=300&h=300&fit=crop&q=80"
              alt="Video editing"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1598550476439-6847785fcea6?w=300&h=300&fit=crop&q=80"
              alt="Content shooting"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1574717024653-61fd2cf4d44d?w=300&h=300&fit=crop&q=80"
              alt="Post production"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
          </div>

          <div className="grid md:grid-cols-2 gap-8">
            <div className="bg-gradient-to-br from-purple-600 to-pink-600 rounded-3xl p-10 text-white">
              <Film className="w-16 h-16 mb-6" />
              <h3 className="text-3xl font-bold mb-6">What You'll Create</h3>
              <ul className="space-y-4">
                {projects.map((project, index) => (
                  <li key={index} className="flex items-start gap-3">
                    <CheckCircle2 className="w-6 h-6 flex-shrink-0 mt-1" />
                    <span className="text-lg">{project}</span>
                  </li>
                ))}
              </ul>
            </div>
            <div className="bg-gray-50 rounded-3xl p-10 border-2 border-purple-200">
              <Video className="w-16 h-16 text-purple-600 mb-6" />
              <h3 className="text-3xl font-bold text-gray-900 mb-6">Your Deliverables</h3>
              <ul className="space-y-4">
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-purple-600 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">1</span>
                  </div>
                  <span className="text-lg text-gray-700">3+ viral-ready reels</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-purple-600 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">2</span>
                  </div>
                  <span className="text-lg text-gray-700">10 short clips from long video</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-purple-600 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">3</span>
                  </div>
                  <span className="text-lg text-gray-700">AI-generated video with script</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-purple-600 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">4</span>
                  </div>
                  <span className="text-lg text-gray-700">Professional showreel portfolio</span>
                </li>
              </ul>
            </div>
          </div>
        </div>
      </section>

      {/* TOOLS */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-100 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">AI Tools You'll Master</h2>
            <p className="text-xl text-gray-600">Industry-leading video editing platforms</p>
          </div>
          <div className="grid grid-cols-2 md:grid-cols-3 gap-6">
            {tools.map((tool, index) => (
              <div key={index} className="bg-white rounded-2xl p-8 shadow-lg hover:shadow-xl transition-all text-center border-2 border-gray-100 hover:border-purple-300 group">
                <div className="text-5xl mb-4 group-hover:scale-110 transition-transform">{tool.icon}</div>
                <p className="font-bold text-gray-900">{tool.name}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* CAREER OUTCOMES */}
      <section className="py-20 px-6 bg-gradient-to-br from-pink-600 to-purple-600">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-white mb-4">Career Paths</h2>
            <p className="text-xl text-pink-100">Launch your video editing career</p>
          </div>
          <div className="grid md:grid-cols-4 gap-6">
            {['AI Video Editor', 'Reels Creator', 'YouTube Specialist', 'Social Media Video Producer',
              'Short-Form Strategist', 'Video Automation Expert', 'Content Production Manager', 'Digital Storyteller'].map((career, index) => (
              <div key={index} className="bg-white/10 backdrop-blur-md rounded-xl p-6 text-center hover:bg-white/20 transition-all border-2 border-white/30">
                <Target className="w-10 h-10 text-yellow-400 mx-auto mb-3" />
                <p className="font-bold text-white text-lg">{career}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* TRAINERS */}
      <section className="py-20 px-6 bg-white relative overflow-hidden">
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1553877522-43269d4ea984?w=1920&h=1080&fit=crop&q=80"
            alt="Training background"
            className="w-full h-full object-cover opacity-5"
          />
        </div>
        <div className="container mx-auto max-w-6xl relative z-10">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Expert Trainers</h2>
            <div className="w-32 h-1 bg-gradient-to-r from-purple-500 to-pink-500 mx-auto"></div>
          </div>
          <div className="bg-gradient-to-br from-purple-600 to-pink-600 rounded-3xl p-12 text-white shadow-2xl">
            <div className="grid md:grid-cols-2 gap-12 items-center">
              <div>
                <div className="flex items-center gap-4 mb-6">
                  <div className="w-16 h-16 rounded-full bg-yellow-400 flex items-center justify-center">
                    <Award className="w-10 h-10 text-purple-900" />
                  </div>
                  <div>
                    <h3 className="text-2xl font-bold">Industry Veterans</h3>
                    <p className="text-purple-100">8-12 Years Experience</p>
                  </div>
                </div>
                <p className="text-lg text-purple-100 mb-6 leading-relaxed">
                  Your workshop is delivered by <strong className="text-yellow-400">WorkshopLibrary-certified media & video production experts</strong> with real experience in:
                </p>
                <ul className="space-y-3">
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>Editing for brands & influencers</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>Social media content creation</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>YouTube video production</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>AI-based automation for agencies</span>
                  </li>
                </ul>
              </div>
              <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-8 border-2 border-yellow-400/50">
                <p className="text-lg text-purple-100 italic mb-6">
                  "Real industry professionals who have created viral content and managed video teams for global brands."
                </p>
                <div className="flex items-center gap-3">
                  <Shield className="w-8 h-8 text-yellow-400" />
                  <span className="text-yellow-400 font-bold">WorkshopLibrary Certified</span>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Pricing Section */}
      <section id="pricing-section" className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-12">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Workshop Pricing</h2>
            <p className="text-xl text-gray-600">Special limited-time offer for early birds</p>
          </div>
          <div className="bg-gradient-to-br from-slate-50 to-blue-50 rounded-3xl p-10 border-2 border-slate-200 shadow-xl">
            <div className="relative">
              <div className="bg-gradient-to-br from-white via-blue-50/30 to-purple-50/30 rounded-3xl p-10 shadow-2xl border-2 border-blue-200/50 relative overflow-hidden">
                <div className="absolute inset-0 bg-gradient-to-r from-blue-400/10 to-purple-400/10 rounded-3xl blur-xl"></div>
                <div className="relative z-10">
                  <div className="flex justify-center mb-6">
                    <div className="bg-gradient-to-r from-orange-500 to-red-500 text-white px-6 py-3 rounded-full font-bold text-sm shadow-lg inline-flex items-center gap-2">
                      <span>🔥</span><span>Limited-Time Deal — 50% OFF</span>
                    </div>
                  </div>
                  <div className="text-center mb-4"><p className="text-gray-400 text-lg line-through">$100 / 365 AED</p></div>
                  <div className="text-center mb-4">
                    <div className="flex items-center justify-center gap-4 mb-3">
                      <p className="text-6xl md:text-7xl font-black text-blue-700">$50</p>
                      <span className="text-4xl text-gray-600 font-bold">/</span>
                      <p className="text-6xl md:text-7xl font-black text-blue-700">185 AED</p>
                    </div>
                    <p className="text-gray-700 font-semibold text-lg">For the Next 15 Days Only — Early Bird Offer</p>
                  </div>
                  <div className="text-center mb-8"><p className="text-gray-600 text-base">Enroll Now & Unlock Your Instant 50% Discount — Limited Seats</p></div>
                  <div className="text-center">
                    <button onClick={scrollToForm} className="w-full md:w-auto bg-gradient-to-r from-yellow-400 to-amber-500 hover:from-yellow-500 hover:to-amber-600 text-gray-900 px-12 py-5 rounded-full font-bold text-xl shadow-2xl hover:shadow-3xl transition-all hover:scale-105 inline-flex items-center justify-center gap-3">
                      Claim This Deal<ArrowRight className="w-6 h-6" />
                    </button>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <div className="mt-8">
            <h3 className="text-2xl font-bold text-gray-900 text-center mb-6">Additional Group & Corporate Discounts</h3>
            <div className="bg-white rounded-2xl p-8 shadow-lg border border-gray-200">
              <div className="flex flex-col md:flex-row items-start gap-6">
                <div className="flex-shrink-0">
                  <div className="w-16 h-16 bg-slate-100 rounded-xl flex items-center justify-center">
                    <Users className="w-8 h-8 text-slate-700" />
                  </div>
                </div>
                <div className="flex-1">
                  <h4 className="text-xl font-bold text-gray-900 mb-4">Save More When You Learn Together</h4>
                  <ul className="space-y-3 mb-6">
                    <li className="flex items-start gap-3 text-gray-700"><CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" /><span>10% extra discount for group bookings (2 or more participants)</span></li>
                    <li className="flex items-start gap-3 text-gray-700"><CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" /><span>Special pricing available for corporate teams</span></li>
                    <li className="flex items-start gap-3 text-gray-700"><CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" /><span>Dedicated support and invoicing for companies</span></li>
                  </ul>
                  <button onClick={scrollToForm} className="border-2 border-slate-700 text-slate-700 hover:bg-slate-700 hover:text-white px-8 py-3 rounded-xl font-semibold transition-all inline-flex items-center gap-2">
                    Request Corporate Pricing<ArrowRight className="w-5 h-5" />
                  </button>
                </div>
              </div>
              <div className="mt-6 pt-6 border-t border-gray-200">
                <p className="text-sm text-gray-600 mb-2">✓ Discounts apply on top of the current 15-day promotional price.</p>
                <p className="text-sm text-gray-600">✓ For larger teams, customized offers are available.</p>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* CERTIFICATION */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-4xl">
          <div className="bg-gradient-to-br from-purple-700 to-pink-700 rounded-3xl p-12 text-center text-white border-4 border-yellow-400 shadow-2xl">
            <Award className="w-24 h-24 text-yellow-400 mx-auto mb-6" />
            <h2 className="text-4xl font-bold mb-4">Get Certified</h2>
            <p className="text-2xl text-purple-100 mb-6">
              🎓 AI Video Editing & Reels Creation Certificate
            </p>
            <p className="text-lg text-purple-200 mb-8">
              Issued by <strong className="text-yellow-400">Workshop Library</strong> - globally recognized & industry trusted
            </p>
            <div className="inline-flex items-center gap-3 bg-white/10 backdrop-blur-sm px-8 py-4 rounded-full border-2 border-yellow-400/50">
              <Shield className="w-6 h-6 text-yellow-400" />
              <span className="font-bold">Globally Recognized Certification</span>
            </div>
          </div>
        </div>
      </section>

      {/* FAQ */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Frequently Asked Questions</h2>
          </div>
          <div className="space-y-6">
            {[
              { q: 'Do I need editing experience?', a: 'No, everything is beginner friendly.' },
              { q: 'Can I edit videos on my phone?', a: 'Yes! Many tools we teach are mobile-friendly.' },
              { q: 'Are recordings provided?', a: 'Yes, 30-day access is included.' },
              { q: 'Will I be able to create reels after this?', a: 'Yes, you\'ll create multiple reels during the workshop itself.' },
              { q: 'Will I get a certificate?', a: 'Yes, a recognized certificate is issued.' }
            ].map((faq, index) => (
              <div key={index} className="bg-gradient-to-r from-purple-50 to-pink-50 rounded-xl p-6 hover:shadow-lg transition-all border-2 border-purple-100 hover:border-purple-300">
                <h3 className="text-xl font-bold text-gray-900 mb-3 flex items-center gap-3">
                  <div className="w-8 h-8 rounded-full bg-purple-600 flex items-center justify-center text-white font-bold">
                    Q
                  </div>
                  {faq.q}
                </h3>
                <p className="text-lg text-gray-700 ml-11">{faq.a}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* REGISTRATION FORM */}
      <section id="registration-form" className="py-20 px-6 bg-gradient-to-br from-purple-600 to-pink-600">
        <div className="container mx-auto max-w-2xl">
          <div className="text-center mb-8">
            <h2 className="text-4xl font-bold text-white mb-4">Create Viral Content Today!</h2>
            <p className="text-xl text-purple-100">Limited seats for December 6, 2025</p>
          </div>
          <CourseLeadForm courseName="Video Editing & Reels Creation Workshop" formTitle="Register for Workshop" />
        </div>
      </section>

      {/* FOOTER */}
      <footer className="py-16 px-6 bg-gradient-to-r from-gray-900 via-gray-800 to-gray-900 text-white relative overflow-hidden">
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1451187580459-43490279c0fa?w=1920&h=1080&fit=crop&q=80"
            alt="Footer background"
            className="w-full h-full object-cover opacity-5"
          />
        </div>
        <div className="container mx-auto max-w-7xl relative z-10">
          <div className="text-center">
            <div className="flex items-center justify-center gap-3 mb-6">
              <div className="w-12 h-12 rounded-xl bg-gradient-to-br from-purple-700 to-pink-700 border-2 border-yellow-400/50 flex items-center justify-center">
                <Sparkles className="w-7 h-7 text-yellow-400" />
              </div>
              <div>
                <h3 className="font-bold text-xl">AI School for All</h3>
                <p className="text-xs text-gray-400">by Workshop Library</p>
              </div>
            </div>
            <p className="text-gray-300 mb-6">Professional AI and Creative Skills Education</p>
            <div className="flex items-center justify-center gap-3 mb-6">
              <Shield className="w-5 h-5 text-yellow-400" />
              <span className="text-sm text-gray-300 font-semibold">WorkshopLibrary Partner</span>
            </div>
            <p className="text-xs text-gray-500">
              © 2026 AI School for All. All rights reserved.
            </p>
          </div>
        </div>
      </footer>

      {/* Chatbot */}
      <Chatbot />
    </div>
  );
};

export default WorkshopVideoEditing;
