import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { 
  Calendar, Clock, Video, Users, Globe, Sparkles, TrendingUp, 
  Award, CheckCircle2, Download, ArrowRight, Zap, Target, Star,
  Share2, Instagram, ThumbsUp, BarChart3, MessageCircle, Heart,
  Hash, Repeat, Eye, Megaphone, FileText, BookOpen, Shield
} from 'lucide-react';
import CourseLeadForm from '../components/CourseLeadForm';
import Chatbot from '../components/Chatbot';

const WorkshopSocialMedia = () => {
  const navigate = useNavigate();
  const [timeLeft, setTimeLeft] = useState({ days: 0, hours: 0, minutes: 0, seconds: 0 });

  useEffect(() => {
    // Workshop starts December 7, 2025 at 12:00 AM EST
    const workshopDate = new Date('2025-12-07T05:00:00Z'); // 12:00 AM EST = 05:00 UTC
    const targetDate = workshopDate.getTime();

    const updateCountdown = () => {
      const now = new Date().getTime();
      const distance = targetDate - now;

      if (distance > 0) {
        setTimeLeft({
          days: Math.floor(distance / (1000 * 60 * 60 * 24)),
          hours: Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60)),
          minutes: Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60)),
          seconds: Math.floor((distance % (1000 * 60)) / 1000)
        });
      } else {
        setTimeLeft({ days: 0, hours: 0, minutes: 0, seconds: 0 });
      }
    };

    updateCountdown();
    const interval = setInterval(updateCountdown, 1000);
    return () => clearInterval(interval);
  }, []);

  const scrollToForm = () => {
    const formElement = document.getElementById('registration-form');
    if (formElement) {
      formElement.scrollIntoView({ behavior: 'smooth', block: 'start' });
    }
  };

  const scrollToPricing = () => {
    const pricingElement = document.getElementById('pricing-section');
    if (pricingElement) {
      pricingElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
    }
  };

  const modules = [
    'AI-based content ideas & trend forecasting',
    'Write captions & hooks for higher engagement',
    'Create aesthetic posts in minutes',
    'Design carousels, reels & short-form visuals',
    'Hashtag strategy using AI tools',
    'Build a 30-day content calendar with AI',
    'Competitor analysis & industry insights',
    'Repurpose content into multiple formats'
  ];

  const stats = [
    { value: '92%', label: 'Marketers use social media as primary tool' },
    { value: '10-14h', label: 'Weekly time saved with AI' },
    { value: '25-60%', label: 'Engagement increase with AI posts' },
    { value: '₹20K-₹1.5L', label: 'Monthly freelance earnings' }
  ];

  const tools = [
    { name: 'Canva AI', icon: '🎨' },
    { name: 'ChatGPT', icon: '🤖' },
    { name: 'OpusClip', icon: '✂️' },
    { name: 'Notion AI', icon: '📝' },
    { name: 'Buffer AI', icon: '📊' },
    { name: 'Flick AI', icon: '#️⃣' },
    { name: 'TrendTok', icon: '📈' }
  ];

  const projects = [
    'Create a 30-day content calendar using AI',
    'Generate 5 high-quality carousels',
    'Produce 3 reels with AI editing',
    'AI-based hashtag & caption generation',
    'Build your mini social media portfolio'
  ];

  return (
    <div className="min-h-screen bg-white">
      {/* Navigation */}
      <nav className="fixed top-4 left-1/2 transform -translate-x-1/2 z-50 w-[95%] max-w-7xl">
        <div className="bg-black/30 backdrop-blur-lg rounded-full border border-white/20 px-8 py-3 shadow-2xl">
          <div className="flex items-center justify-between">
            <button onClick={() => navigate('/')} className="text-white font-bold text-lg hover:text-yellow-400 transition-colors">
              AI School for All
            </button>
            <div className="hidden md:flex items-center gap-6">
              <button onClick={() => navigate('/#home')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Home</button>
              <button onClick={() => navigate('/#workshops')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Workshops</button>
              <button 
                onClick={scrollToForm}
                className="bg-yellow-400 hover:bg-yellow-500 text-gray-900 px-6 py-2 rounded-full font-bold transition-all text-sm"
              >
                Register Now
              </button>
            </div>
          </div>
        </div>
      </nav>

      {/* SOCIAL MEDIA THEMED HERO */}
      <section className="relative min-h-screen flex items-center justify-center overflow-hidden pt-20">
        {/* Dynamic Background */}
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1611162616305-c69b3fa7fbe0?w=1920&h=1080&fit=crop&q=80"
            alt="Social Media Marketing"
            className="w-full h-full object-cover brightness-110"
          />
          <div className="absolute inset-0 bg-gradient-to-br from-blue-500/80 via-cyan-500/75 to-teal-500/80"></div>
        </div>

        {/* Animated overlay patterns */}
        <div className="absolute inset-0 opacity-10 z-10">
          <div className="absolute top-20 left-20 w-72 h-72 bg-white rounded-full blur-3xl animate-pulse"></div>
          <div className="absolute bottom-20 right-20 w-96 h-96 bg-white rounded-full blur-3xl animate-pulse delay-1000"></div>
          <div className="absolute top-1/2 left-1/2 w-64 h-64 bg-white rounded-full blur-3xl animate-pulse delay-500"></div>
        </div>

        <div className="container mx-auto px-6 relative z-20">
          <div className="max-w-5xl mx-auto text-center">
            {/* Event Badge */}
            <div className="inline-flex items-center gap-2 bg-white/20 backdrop-blur-md px-6 py-3 rounded-full mb-8 border border-white/30">
              <Megaphone className="w-5 h-5 text-white" />
              <span className="text-white font-bold text-sm uppercase tracking-wider">3-Hour Power Workshop</span>
            </div>

            {/* Main Title */}
            <h1 className="text-5xl md:text-7xl font-black text-white mb-6 leading-tight drop-shadow-2xl">
              AI-Powered
              <br />
              <span className="text-yellow-400">Social Media Booster</span>
            </h1>

            <p className="text-2xl text-white/90 mb-10 max-w-3xl mx-auto font-medium">
              Supercharge your social presence with AI — Create 30 days of content in under 2 hours!
            </p>

            {/* Event Quick Info */}
            <div className="grid md:grid-cols-4 gap-4 mb-12 max-w-4xl mx-auto">
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Calendar className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">7 Dec 2025</p>
                <p className="text-white/80 text-sm">Sunday</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Clock className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">3 Hours</p>
                <p className="text-white/80 text-sm">Fast-Paced</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Video className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">Live Online</p>
                <p className="text-white/80 text-sm">Interactive</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Zap className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">AI-Powered</p>
                <p className="text-white/80 text-sm">Lightning Fast</p>
              </div>
            </div>

            {/* CTA */}
            <button 
              onClick={scrollToForm}
              className="bg-yellow-400 hover:bg-yellow-500 text-gray-900 px-12 py-6 rounded-2xl font-black text-xl shadow-2xl hover:scale-105 transition-all inline-flex items-center gap-3"
            >
              <Sparkles className="w-6 h-6" />
              Boost My Social Media Now
              <ArrowRight className="w-6 h-6" />
            </button>

            <p className="text-white/90 mt-6 text-sm">
              <strong>Enrollment Opens Soon</strong> - Limited seats available!
            </p>
          </div>
        </div>
      </section>

      {/* COUNTDOWN TIMER */}
      <section className="py-16 px-6 bg-gradient-to-b from-blue-900 to-cyan-900">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-8">
            <h3 className="text-3xl font-bold text-yellow-400 mb-2">Workshop Starts In</h3>
            <p className="text-gray-200">December 7, 2025 • 12:00 AM EST</p>
          </div>
          <div className="grid grid-cols-4 gap-4 max-w-2xl mx-auto">
            {['days', 'hours', 'minutes', 'seconds'].map((unit) => (
              <div key={unit} className="bg-gradient-to-br from-blue-700 to-cyan-700 rounded-xl p-6 text-center border-2 border-yellow-400/30">
                <div className="text-5xl font-bold text-yellow-400 mb-2">{timeLeft[unit]}</div>
                <div className="text-gray-200 text-sm uppercase tracking-wide">{unit}</div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Promotional Banner - Limited Time Deal */}
      <section className="py-6 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <button onClick={scrollToPricing} className="w-full bg-gradient-to-r from-orange-500 to-red-500 hover:from-orange-600 hover:to-red-600 text-white px-8 py-5 rounded-full font-bold text-lg shadow-xl hover:shadow-2xl transition-all hover:scale-105 flex items-center justify-center gap-3">
            <span className="text-2xl">🔥</span><span>Limited-Time Deal — 50% OFF</span><ArrowRight className="w-6 h-6" />
          </button>
        </div>
      </section>

      {/* WHY THIS WORKSHOP */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="grid md:grid-cols-2 gap-12 items-center">
            <div>
              <div className="inline-flex items-center gap-2 bg-blue-100 px-4 py-2 rounded-full mb-6">
                <Share2 className="w-5 h-5 text-blue-600" />
                <span className="text-blue-900 font-bold text-sm">GROW YOUR BRAND</span>
              </div>
              <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-6">Why Social Media Growth is Hard (Without AI)</h2>
              <p className="text-lg text-gray-700 mb-6 leading-relaxed">
                Creating consistent, high-quality social media content is exhausting. You need fresh ideas daily, eye-catching visuals, engaging captions, trending hashtags, and perfect posting schedules.
              </p>
              <p className="text-lg text-gray-700 mb-6 leading-relaxed">
                In this workshop, you'll learn how to <strong className="text-blue-600">create 30 days of content in under 2 hours</strong> using AI tools that handle ideation, design, copywriting, scheduling, and analytics.
              </p>
              <p className="text-lg text-gray-700 leading-relaxed">
                Walk away with a <strong className="text-cyan-600">complete content system</strong>, ready-to-post templates, a monthly calendar, and the confidence to grow any brand with AI-powered social media strategies.
              </p>
            </div>
            <div className="relative">
              <div className="bg-gradient-to-br from-blue-500 to-cyan-500 rounded-3xl p-8 shadow-2xl">
                <img 
                  src="https://images.unsplash.com/photo-1611926653458-09294b3142bf?w=600&h=400&fit=crop&q=80"
                  alt="Social Media Growth"
                  className="w-full h-80 object-cover rounded-2xl shadow-xl"
                />
                <div className="absolute -bottom-6 -right-6 bg-white rounded-2xl p-6 shadow-2xl border-4 border-yellow-400">
                  <div className="text-center">
                    <div className="text-4xl font-bold text-blue-600">10-14h</div>
                    <div className="text-sm text-gray-700 font-medium">Saved Per Week</div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* PLATFORM SHOWCASE - Instagram, Facebook, TikTok */}
      <section className="py-20 px-6 bg-gradient-to-b from-blue-900 to-cyan-900">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-white mb-4">Master All Major Platforms</h2>
            <p className="text-xl text-cyan-100">Create viral content for Instagram, Facebook, TikTok & more</p>
          </div>

          <div className="grid md:grid-cols-3 gap-8">
            {/* Instagram */}
            <div className="bg-gradient-to-br from-pink-500 to-purple-600 rounded-3xl p-8 text-white shadow-2xl hover:scale-105 transition-transform">
              <div className="flex items-center justify-center mb-6">
                <Instagram className="w-16 h-16" />
              </div>
              <h3 className="text-3xl font-bold text-center mb-4">Instagram</h3>
              <ul className="space-y-3 text-lg">
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 flex-shrink-0" />
                  <span>Reels with AI captions</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 flex-shrink-0" />
                  <span>Story templates</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 flex-shrink-0" />
                  <span>Carousel posts</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 flex-shrink-0" />
                  <span>Grid aesthetic planning</span>
                </li>
              </ul>
            </div>

            {/* Facebook / Meta */}
            <div className="bg-gradient-to-br from-blue-600 to-blue-800 rounded-3xl p-8 text-white shadow-2xl hover:scale-105 transition-transform">
              <div className="flex items-center justify-center mb-6">
                <Share2 className="w-16 h-16" />
              </div>
              <h3 className="text-3xl font-bold text-center mb-4">Facebook / Meta</h3>
              <ul className="space-y-3 text-lg">
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 flex-shrink-0" />
                  <span>Video ads with AI copy</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 flex-shrink-0" />
                  <span>Brand promotion posts</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 flex-shrink-0" />
                  <span>Engagement boosting</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 flex-shrink-0" />
                  <span>Group content strategy</span>
                </li>
              </ul>
            </div>

            {/* TikTok */}
            <div className="bg-gradient-to-br from-gray-900 to-gray-800 rounded-3xl p-8 text-white shadow-2xl hover:scale-105 transition-transform border-2 border-cyan-400">
              <div className="flex items-center justify-center mb-6">
                <Video className="w-16 h-16 text-cyan-400" />
              </div>
              <h3 className="text-3xl font-bold text-center mb-4">TikTok</h3>
              <ul className="space-y-3 text-lg">
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 flex-shrink-0 text-cyan-400" />
                  <span>Viral short videos</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 flex-shrink-0 text-cyan-400" />
                  <span>Trend forecasting</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 flex-shrink-0 text-cyan-400" />
                  <span>Hashtag challenges</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 flex-shrink-0 text-cyan-400" />
                  <span>Duet & stitch strategies</span>
                </li>
              </ul>
            </div>
          </div>

          {/* Additional Platforms */}
          <div className="mt-12 text-center">
            <p className="text-xl text-white mb-6">Also covers:</p>
            <div className="flex flex-wrap justify-center gap-4">
              {['LinkedIn Posts', 'YouTube Shorts', 'Twitter/X Threads', 'Pinterest Pins', 'Snapchat Stories'].map((platform, index) => (
                <div key={index} className="bg-white/10 backdrop-blur-sm px-6 py-3 rounded-full border border-white/30">
                  <span className="text-white font-semibold">{platform}</span>
                </div>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* BEFORE/AFTER SOCIAL MEDIA TRANSFORMATIONS */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">See The AI Social Media Magic</h2>
            <p className="text-xl text-gray-600">Real transformations you'll learn to create</p>
          </div>

          {/* Example 1: Instagram Feed Transformation */}
          <div className="mb-16">
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">Instagram Feed Glow-Up</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BEFORE
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl bg-gray-100">
                  <div className="p-6">
                    <div className="grid grid-cols-3 gap-2">
                      {[1,2,3,4,5,6,7,8,9].map((i) => (
                        <div key={i} className="aspect-square bg-gray-300 rounded-lg flex items-center justify-center">
                          <span className="text-gray-400 text-xs">Post {i}</span>
                        </div>
                      ))}
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Random posts • No theme • Low engagement • Inconsistent</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  AFTER AI TRANSFORMATION
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl">
                  <div className="p-6 bg-gradient-to-br from-blue-50 to-cyan-50">
                    <div className="grid grid-cols-3 gap-2">
                      <img src="https://images.unsplash.com/photo-1611162616305-c69b3fa7fbe0?w=300&h=300&fit=crop&q=80" alt="Social post 1" className="aspect-square object-cover rounded-lg" />
                      <img src="https://images.unsplash.com/photo-1611926653458-09294b3142bf?w=300&h=300&fit=crop&q=80" alt="Social post 2" className="aspect-square object-cover rounded-lg" />
                      <img src="https://images.unsplash.com/photo-1611605698335-8b1569810432?w=300&h=300&fit=crop&q=80" alt="Social post 3" className="aspect-square object-cover rounded-lg" />
                      <img src="https://images.unsplash.com/photo-1633356122544-f134324a6cee?w=300&h=300&fit=crop&q=80" alt="Social post 4" className="aspect-square object-cover rounded-lg" />
                      <img src="https://images.unsplash.com/photo-1611162616475-46b635cb6868?w=300&h=300&fit=crop&q=80" alt="Social post 5" className="aspect-square object-cover rounded-lg" />
                      <img src="https://images.unsplash.com/photo-1579869847557-1f67382cc158?w=300&h=300&fit=crop&q=80" alt="Social post 6" className="aspect-square object-cover rounded-lg" />
                      <img src="https://images.unsplash.com/photo-1563986768609-322da13575f3?w=300&h=300&fit=crop&q=80" alt="Social post 7" className="aspect-square object-cover rounded-lg" />
                      <img src="https://images.unsplash.com/photo-1611162616305-c69b3fa7fbe0?w=300&h=300&fit=crop&q=85" alt="Social post 8" className="aspect-square object-cover rounded-lg" />
                      <img src="https://images.unsplash.com/photo-1611926653458-09294b3142bf?w=300&h=300&fit=crop&q=85" alt="Social post 9" className="aspect-square object-cover rounded-lg" />
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Cohesive aesthetic • Brand colors • High engagement • AI-planned theme!</p>
              </div>
            </div>
          </div>

          {/* Example 2: Engagement Metrics */}
          <div className="mb-16">
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">Post Engagement Boost</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BEFORE
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl bg-white">
                  <div className="p-8">
                    <img 
                      src="https://images.unsplash.com/photo-1611605698335-8b1569810432?w=600&h=400&fit=crop&q=60&sat=-40"
                      alt="Generic social post"
                      className="w-full h-64 object-cover rounded-xl mb-6"
                    />
                    <div className="space-y-4">
                      <div className="flex items-center justify-between text-gray-500">
                        <div className="flex items-center gap-2">
                          <Heart className="w-5 h-5" />
                          <span className="text-lg">23 likes</span>
                        </div>
                        <div className="flex items-center gap-2">
                          <MessageCircle className="w-5 h-5" />
                          <span className="text-lg">2 comments</span>
                        </div>
                        <div className="flex items-center gap-2">
                          <Share2 className="w-5 h-5" />
                          <span className="text-lg">1 share</span>
                        </div>
                      </div>
                      <p className="text-sm text-gray-600 italic">"Check out this product! #buy #shop"</p>
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Generic caption • Wrong hashtags • Low reach • Poor engagement</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  AFTER AI OPTIMIZATION
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl bg-white">
                  <div className="p-8">
                    <img 
                      src="https://images.unsplash.com/photo-1611605698335-8b1569810432?w=600&h=400&fit=crop&q=90"
                      alt="Optimized social post"
                      className="w-full h-64 object-cover rounded-xl mb-6"
                    />
                    <div className="space-y-4">
                      <div className="flex items-center justify-between text-blue-600">
                        <div className="flex items-center gap-2">
                          <Heart className="w-5 h-5 fill-current" />
                          <span className="text-lg font-bold">1,847 likes</span>
                        </div>
                        <div className="flex items-center gap-2">
                          <MessageCircle className="w-5 h-5" />
                          <span className="text-lg font-bold">134 comments</span>
                        </div>
                        <div className="flex items-center gap-2">
                          <Share2 className="w-5 h-5" />
                          <span className="text-lg font-bold">89 shares</span>
                        </div>
                      </div>
                      <p className="text-sm text-gray-700 font-medium">"Transform your workspace with these 5 game-changing productivity hacks! Which one will you try first? #ProductivityTips #WorkspaceGoals"</p>
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">AI-crafted caption • Trending hashtags • 80x more engagement!</p>
              </div>
            </div>
          </div>

          {/* Example 3: Content Calendar */}
          <div>
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">30-Day Content Calendar Magic</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BEFORE
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl">
                  <div className="bg-white p-8 h-96 flex items-center justify-center">
                    <div className="text-center text-gray-400">
                      <Calendar className="w-20 h-20 mx-auto mb-4" />
                      <p className="text-xl font-bold mb-2">Empty Calendar</p>
                      <p className="text-sm">No plan • Daily stress • Inconsistent posting</p>
                      <p className="text-xs mt-4">Takes 20+ hours to plan manually</p>
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">No strategy • Posting randomly • Missing trends</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  AFTER AI PLANNING
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl bg-gradient-to-br from-blue-50 to-cyan-50">
                  <div className="p-6 h-96 overflow-hidden">
                    <div className="grid grid-cols-7 gap-2 text-xs">
                      <div className="font-bold text-center text-gray-700">Mon</div>
                      <div className="font-bold text-center text-gray-700">Tue</div>
                      <div className="font-bold text-center text-gray-700">Wed</div>
                      <div className="font-bold text-center text-gray-700">Thu</div>
                      <div className="font-bold text-center text-gray-700">Fri</div>
                      <div className="font-bold text-center text-gray-700">Sat</div>
                      <div className="font-bold text-center text-gray-700">Sun</div>
                      {Array.from({length: 28}, (_, i) => (
                        <div key={i} className={`aspect-square rounded-lg p-2 ${
                          i % 3 === 0 ? 'bg-blue-500' : i % 3 === 1 ? 'bg-cyan-500' : 'bg-teal-500'
                        } text-white flex flex-col items-center justify-center`}>
                          <div className="text-xs font-bold">{i + 1}</div>
                          <div className="text-[8px] mt-1">
                            {i % 3 === 0 ? '📸 Post' : i % 3 === 1 ? '🎬 Reel' : '📊 Story'}
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">30 days planned • AI-generated • Created in under 2 hours!</p>
              </div>
            </div>
          </div>

          {/* Results Banner */}
          <div className="mt-16 bg-gradient-to-r from-blue-600 to-cyan-600 rounded-3xl p-10 text-center text-white">
            <p className="text-3xl font-bold mb-4">Master All These Skills in Just 3 Hours!</p>
            <p className="text-xl text-white/90">No experience needed - AI does the heavy lifting</p>
          </div>
        </div>
      </section>

      {/* REELS & STORIES MASTERY */}
      <section className="py-20 px-6 bg-gradient-to-b from-pink-50 via-purple-50 to-blue-50">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <div className="inline-flex items-center gap-3 bg-gradient-to-r from-pink-500 to-purple-600 text-white px-6 py-3 rounded-full mb-6">
              <Video className="w-6 h-6" />
              <span className="font-bold text-lg">REELS & STORIES</span>
            </div>
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Create Viral Reels & Stories</h2>
            <p className="text-xl text-gray-600">The #1 format for social media growth</p>
          </div>

          <div className="grid md:grid-cols-3 gap-8 mb-12">
            {/* Instagram Reels */}
            <div className="bg-white rounded-3xl p-8 shadow-xl border-4 border-pink-200">
              <div className="aspect-[9/16] bg-gradient-to-br from-pink-100 to-purple-100 rounded-2xl mb-6 flex items-center justify-center relative overflow-hidden">
                <img 
                  src="https://images.unsplash.com/photo-1611162616305-c69b3fa7fbe0?w=400&h=700&fit=crop&q=80"
                  alt="Instagram Reel"
                  className="w-full h-full object-cover"
                />
                <div className="absolute bottom-4 left-4 right-4 bg-black/70 backdrop-blur-sm rounded-xl p-4">
                  <div className="flex items-center gap-2 text-white text-sm mb-2">
                    <Heart className="w-4 h-4 fill-white" />
                    <span className="font-bold">234.5K</span>
                    <MessageCircle className="w-4 h-4 ml-2" />
                    <span className="font-bold">3,891</span>
                  </div>
                  <p className="text-white text-xs">AI-generated caption & trending audio</p>
                </div>
              </div>
              <h3 className="text-xl font-bold text-gray-900 mb-2">Instagram Reels</h3>
              <p className="text-gray-600">Learn to create 15-90s reels with AI editing, captions & trending sounds</p>
            </div>

            {/* Stories */}
            <div className="bg-white rounded-3xl p-8 shadow-xl border-4 border-purple-200">
              <div className="aspect-[9/16] bg-gradient-to-br from-purple-100 to-blue-100 rounded-2xl mb-6 flex items-center justify-center relative overflow-hidden">
                <img 
                  src="https://images.unsplash.com/photo-1611926653458-09294b3142bf?w=400&h=700&fit=crop&q=80"
                  alt="Instagram Story"
                  className="w-full h-full object-cover"
                />
                <div className="absolute top-4 left-4 right-4">
                  <div className="bg-white/90 backdrop-blur-sm rounded-full px-4 py-2">
                    <p className="text-gray-900 text-sm font-bold">Daily Tips</p>
                  </div>
                </div>
                <div className="absolute bottom-4 left-4 right-4">
                  <div className="bg-gradient-to-r from-pink-500 to-purple-600 rounded-2xl p-4">
                    <p className="text-white font-bold text-center">Swipe Up 👆</p>
                  </div>
                </div>
              </div>
              <h3 className="text-xl font-bold text-gray-900 mb-2">Stories & Highlights</h3>
              <p className="text-gray-600">Design engaging 24hr stories with stickers, polls & CTAs using AI</p>
            </div>

            {/* TikTok Videos */}
            <div className="bg-white rounded-3xl p-8 shadow-xl border-4 border-cyan-200">
              <div className="aspect-[9/16] bg-gradient-to-br from-blue-100 to-cyan-100 rounded-2xl mb-6 flex items-center justify-center relative overflow-hidden">
                <img 
                  src="https://images.unsplash.com/photo-1611605698335-8b1569810432?w=400&h=700&fit=crop&q=80"
                  alt="TikTok Video"
                  className="w-full h-full object-cover"
                />
                <div className="absolute bottom-4 left-4 right-4 bg-black/70 backdrop-blur-sm rounded-xl p-4">
                  <div className="flex items-center gap-3 text-white text-sm mb-2">
                    <Heart className="w-4 h-4 fill-white" />
                    <span className="font-bold">892K</span>
                    <MessageCircle className="w-4 h-4" />
                    <span className="font-bold">12.3K</span>
                    <Share2 className="w-4 h-4" />
                    <span className="font-bold">45.2K</span>
                  </div>
                  <p className="text-white text-xs">🔥 Trending • #FYP</p>
                </div>
              </div>
              <h3 className="text-xl font-bold text-gray-900 mb-2">TikTok Shorts</h3>
              <p className="text-gray-600">Master viral TikTok format with AI scripts, effects & trending sounds</p>
            </div>
          </div>

          {/* Reels Stats */}
          <div className="bg-gradient-to-r from-pink-500 via-purple-600 to-blue-600 rounded-3xl p-10 text-white text-center">
            <h3 className="text-3xl font-bold mb-6">Why Reels & Stories Matter</h3>
            <div className="grid md:grid-cols-3 gap-8">
              <div>
                <div className="text-5xl font-black mb-2">300%</div>
                <p className="text-lg text-white/90">More reach than regular posts</p>
              </div>
              <div>
                <div className="text-5xl font-black mb-2">2.5B+</div>
                <p className="text-lg text-white/90">Daily active users on Reels</p>
              </div>
              <div>
                <div className="text-5xl font-black mb-2">6x</div>
                <p className="text-lg text-white/90">Faster follower growth</p>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* INDUSTRY INSIGHTS */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Industry Insights</h2>
            <p className="text-xl text-gray-600">Why AI social media management is essential</p>
          </div>
          <div className="grid md:grid-cols-4 gap-8">
            {stats.map((stat, index) => (
              <div key={index} className="bg-gradient-to-br from-blue-50 to-cyan-50 rounded-2xl p-8 shadow-lg hover:shadow-xl transition-all border-2 border-blue-100">
                <div className="text-5xl font-black text-blue-600 mb-4">{stat.value}</div>
                <p className="text-gray-700 font-medium leading-relaxed">{stat.label}</p>
              </div>
            ))}
          </div>
          <div className="mt-12 text-center">
            <p className="text-lg text-gray-600">
              <span className="text-blue-600 font-bold">Short-form content consumption</span> has grown <strong>6x in the last 2 years</strong>
            </p>
          </div>
        </div>
      </section>

      {/* WHAT YOU'LL LEARN */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">What You'll Master</h2>
            <div className="w-32 h-1 bg-gradient-to-r from-blue-500 to-cyan-500 mx-auto"></div>
          </div>
          
          {/* Social Media Gallery */}
          <div className="grid md:grid-cols-3 gap-4 mb-12">
            <img 
              src="https://images.unsplash.com/photo-1611162616305-c69b3fa7fbe0?w=400&h=300&fit=crop&q=80"
              alt="Instagram stories and reels"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
            <img 
              src="https://images.unsplash.com/photo-1611926653458-09294b3142bf?w=400&h=300&fit=crop&q=80"
              alt="Social media engagement"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
            <img 
              src="https://images.unsplash.com/photo-1611605698335-8b1569810432?w=400&h=300&fit=crop&q=80"
              alt="TikTok and Facebook content"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
          </div>

          <div className="grid md:grid-cols-2 gap-6">
            {modules.map((module, index) => (
              <div key={index} className="flex items-start gap-4 bg-gradient-to-r from-blue-50 to-cyan-50 rounded-xl p-6 hover:shadow-lg transition-all">
                <div className="w-10 h-10 rounded-full bg-gradient-to-br from-blue-600 to-cyan-600 flex items-center justify-center flex-shrink-0">
                  <CheckCircle2 className="w-6 h-6 text-white" />
                </div>
                <p className="text-gray-800 text-lg font-medium">{module}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* WHO SHOULD JOIN */}
      <section className="py-20 px-6 bg-gradient-to-br from-blue-700 to-cyan-700 text-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold mb-4">Perfect For</h2>
            <p className="text-xl text-blue-100">This workshop is designed for...</p>
          </div>
          <div className="grid md:grid-cols-4 gap-6">
            {['Content Creators', 'Influencers', 'Small Business Owners', 'Instagram Marketers', 
              'Freelancers', 'Students', 'Coaches & Trainers', 'Brand Builders'].map((role, index) => (
              <div key={index} className="bg-white/10 backdrop-blur-sm rounded-xl p-6 text-center hover:bg-white/20 transition-all border border-yellow-400/30">
                <Star className="w-8 h-8 text-yellow-400 mx-auto mb-3" />
                <p className="font-semibold text-lg">{role}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* HANDS-ON PROJECTS */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Hands-On Projects</h2>
            <p className="text-xl text-gray-600">Build your social media portfolio live</p>
          </div>

          {/* Project Gallery */}
          <div className="grid md:grid-cols-4 gap-4 mb-12">
            <img 
              src="https://images.unsplash.com/photo-1611162616305-c69b3fa7fbe0?w=300&h=300&fit=crop&q=80"
              alt="Instagram reels and stories"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1611926653458-09294b3142bf?w=300&h=300&fit=crop&q=80"
              alt="Social media analytics"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1611605698335-8b1569810432?w=300&h=300&fit=crop&q=80"
              alt="Facebook and TikTok posts"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1579869847557-1f67382cc158?w=300&h=300&fit=crop&q=80"
              alt="Social engagement dashboard"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
          </div>

          <div className="grid md:grid-cols-2 gap-8">
            <div className="bg-gradient-to-br from-blue-600 to-cyan-600 rounded-3xl p-10 text-white">
              <Share2 className="w-16 h-16 mb-6" />
              <h3 className="text-3xl font-bold mb-6">What You'll Create</h3>
              <ul className="space-y-4">
                {projects.map((project, index) => (
                  <li key={index} className="flex items-start gap-3">
                    <CheckCircle2 className="w-6 h-6 flex-shrink-0 mt-1" />
                    <span className="text-lg">{project}</span>
                  </li>
                ))}
              </ul>
            </div>
            <div className="bg-gray-50 rounded-3xl p-10 border-2 border-blue-200">
              <Instagram className="w-16 h-16 text-blue-600 mb-6" />
              <h3 className="text-3xl font-bold text-gray-900 mb-6">Your Deliverables</h3>
              <ul className="space-y-4">
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-blue-600 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">1</span>
                  </div>
                  <span className="text-lg text-gray-700">30-day AI-powered content calendar</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-blue-600 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">2</span>
                  </div>
                  <span className="text-lg text-gray-700">5 professional carousels</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-blue-600 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">3</span>
                  </div>
                  <span className="text-lg text-gray-700">3 viral-ready reels</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-blue-600 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">4</span>
                  </div>
                  <span className="text-lg text-gray-700">Complete social media portfolio</span>
                </li>
              </ul>
            </div>
          </div>
        </div>
      </section>

      {/* TOOLS */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-100 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">AI Tools You'll Master</h2>
            <p className="text-xl text-gray-600">Industry-leading social media platforms</p>
          </div>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
            {tools.map((tool, index) => (
              <div key={index} className="bg-white rounded-2xl p-8 shadow-lg hover:shadow-xl transition-all text-center border-2 border-gray-100 hover:border-blue-300 group">
                <div className="text-5xl mb-4 group-hover:scale-110 transition-transform">{tool.icon}</div>
                <p className="font-bold text-gray-900">{tool.name}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* CAREER OUTCOMES */}
      <section className="py-20 px-6 bg-gradient-to-br from-cyan-600 to-blue-600">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-white mb-4">Career Paths</h2>
            <p className="text-xl text-cyan-100">Launch your social media career</p>
          </div>
          <div className="grid md:grid-cols-4 gap-6">
            {['AI Social Media Strategist', 'Content Creator', 'Creative Assistant', 'Social Media Manager',
              'Reels Designer', 'Brand Engagement Expert', 'Carousel Specialist', 'Automation Expert'].map((career, index) => (
              <div key={index} className="bg-white/10 backdrop-blur-md rounded-xl p-6 text-center hover:bg-white/20 transition-all border-2 border-white/30">
                <Target className="w-10 h-10 text-yellow-400 mx-auto mb-3" />
                <p className="font-bold text-white text-lg">{career}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* TRAINERS */}
      <section className="py-20 px-6 bg-white relative overflow-hidden">
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1553877522-43269d4ea984?w=1920&h=1080&fit=crop&q=80"
            alt="Training background"
            className="w-full h-full object-cover opacity-5"
          />
        </div>
        <div className="container mx-auto max-w-6xl relative z-10">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Expert Trainers</h2>
            <div className="w-32 h-1 bg-gradient-to-r from-blue-500 to-cyan-500 mx-auto"></div>
          </div>
          <div className="bg-gradient-to-br from-blue-600 to-cyan-600 rounded-3xl p-12 text-white shadow-2xl">
            <div className="grid md:grid-cols-2 gap-12 items-center">
              <div>
                <div className="flex items-center gap-4 mb-6">
                  <div className="w-16 h-16 rounded-full bg-yellow-400 flex items-center justify-center">
                    <Award className="w-10 h-10 text-blue-900" />
                  </div>
                  <div>
                    <h3 className="text-2xl font-bold">Industry Experts</h3>
                    <p className="text-blue-100">8-10+ Years Experience</p>
                  </div>
                </div>
                <p className="text-lg text-blue-100 mb-6 leading-relaxed">
                  Your workshop is led by <strong className="text-yellow-400">WorkshopLibrary-certified social media & AI automation experts</strong> with proven experience in:
                </p>
                <ul className="space-y-3">
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>Content strategy for brands</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>Social branding & digital marketing</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>Creator growth systems</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>AI productivity tools</span>
                  </li>
                </ul>
              </div>
              <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-8 border-2 border-yellow-400/50">
                <p className="text-lg text-blue-100 italic mb-6">
                  "Real professionals who have grown influencers, managed brand campaigns, and built viral content strategies."
                </p>
                <div className="flex items-center gap-3">
                  <Shield className="w-8 h-8 text-yellow-400" />
                  <span className="text-yellow-400 font-bold">WorkshopLibrary Certified</span>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Pricing Section */}
      <section id="pricing-section" className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-12">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Workshop Pricing</h2>
            <p className="text-xl text-gray-600">Special limited-time offer for early birds</p>
          </div>
          <div className="bg-gradient-to-br from-slate-50 to-blue-50 rounded-3xl p-10 border-2 border-slate-200 shadow-xl">
            <div className="relative">
              <div className="bg-gradient-to-br from-white via-blue-50/30 to-purple-50/30 rounded-3xl p-10 shadow-2xl border-2 border-blue-200/50 relative overflow-hidden">
                <div className="absolute inset-0 bg-gradient-to-r from-blue-400/10 to-purple-400/10 rounded-3xl blur-xl"></div>
                <div className="relative z-10">
                  <div className="flex justify-center mb-6">
                    <div className="bg-gradient-to-r from-orange-500 to-red-500 text-white px-6 py-3 rounded-full font-bold text-sm shadow-lg inline-flex items-center gap-2">
                      <span>🔥</span><span>Limited-Time Deal — 50% OFF</span>
                    </div>
                  </div>
                  <div className="text-center mb-4"><p className="text-gray-400 text-lg line-through">$100 / 365 AED</p></div>
                  <div className="text-center mb-4">
                    <div className="flex items-center justify-center gap-4 mb-3">
                      <p className="text-6xl md:text-7xl font-black text-blue-700">$50</p>
                      <span className="text-4xl text-gray-600 font-bold">/</span>
                      <p className="text-6xl md:text-7xl font-black text-blue-700">185 AED</p>
                    </div>
                    <p className="text-gray-700 font-semibold text-lg">For the Next 15 Days Only — Early Bird Offer</p>
                  </div>
                  <div className="text-center mb-8"><p className="text-gray-600 text-base">Enroll Now & Unlock Your Instant 50% Discount — Limited Seats</p></div>
                  <div className="text-center">
                    <button onClick={scrollToForm} className="w-full md:w-auto bg-gradient-to-r from-yellow-400 to-amber-500 hover:from-yellow-500 hover:to-amber-600 text-gray-900 px-12 py-5 rounded-full font-bold text-xl shadow-2xl hover:shadow-3xl transition-all hover:scale-105 inline-flex items-center justify-center gap-3">
                      Claim This Deal<ArrowRight className="w-6 h-6" />
                    </button>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <div className="mt-8">
            <h3 className="text-2xl font-bold text-gray-900 text-center mb-6">Additional Group & Corporate Discounts</h3>
            <div className="bg-white rounded-2xl p-8 shadow-lg border border-gray-200">
              <div className="flex flex-col md:flex-row items-start gap-6">
                <div className="flex-shrink-0">
                  <div className="w-16 h-16 bg-slate-100 rounded-xl flex items-center justify-center">
                    <Users className="w-8 h-8 text-slate-700" />
                  </div>
                </div>
                <div className="flex-1">
                  <h4 className="text-xl font-bold text-gray-900 mb-4">Save More When You Learn Together</h4>
                  <ul className="space-y-3 mb-6">
                    <li className="flex items-start gap-3 text-gray-700"><CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" /><span>10% extra discount for group bookings (2 or more participants)</span></li>
                    <li className="flex items-start gap-3 text-gray-700"><CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" /><span>Special pricing available for corporate teams</span></li>
                    <li className="flex items-start gap-3 text-gray-700"><CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" /><span>Dedicated support and invoicing for companies</span></li>
                  </ul>
                  <button onClick={scrollToForm} className="border-2 border-slate-700 text-slate-700 hover:bg-slate-700 hover:text-white px-8 py-3 rounded-xl font-semibold transition-all inline-flex items-center gap-2">
                    Request Corporate Pricing<ArrowRight className="w-5 h-5" />
                  </button>
                </div>
              </div>
              <div className="mt-6 pt-6 border-t border-gray-200">
                <p className="text-sm text-gray-600 mb-2">✓ Discounts apply on top of the current 15-day promotional price.</p>
                <p className="text-sm text-gray-600">✓ For larger teams, customized offers are available.</p>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* CERTIFICATION */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-4xl">
          <div className="bg-gradient-to-br from-blue-700 to-cyan-700 rounded-3xl p-12 text-center text-white border-4 border-yellow-400 shadow-2xl">
            <Award className="w-24 h-24 text-yellow-400 mx-auto mb-6" />
            <h2 className="text-4xl font-bold mb-4">Get Certified</h2>
            <p className="text-2xl text-blue-100 mb-6">
              🎓 AI-Powered Social Media Booster Certificate
            </p>
            <p className="text-lg text-blue-200 mb-8">
              Issued by <strong className="text-yellow-400">Workshop Library</strong> - globally recognized & industry trusted
            </p>
            <div className="inline-flex items-center gap-3 bg-white/10 backdrop-blur-sm px-8 py-4 rounded-full border-2 border-yellow-400/50">
              <Shield className="w-6 h-6 text-yellow-400" />
              <span className="font-bold">Globally Recognized Certification</span>
            </div>
          </div>
        </div>
      </section>

      {/* FAQ */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Frequently Asked Questions</h2>
          </div>
          <div className="space-y-6">
            {[
              { q: 'Will this help me grow my Instagram?', a: 'Yes, the strategies are proven and practical.' },
              { q: 'Can beginners join?', a: 'Absolutely—no prior experience required.' },
              { q: 'Will I get templates?', a: 'Yes, you get AI-driven templates & content systems.' },
              { q: 'Are recordings provided?', a: 'Yes, 30-day replay access.' },
              { q: 'Is this helpful for businesses?', a: 'Extremely. AI saves time and boosts reach.' }
            ].map((faq, index) => (
              <div key={index} className="bg-gradient-to-r from-blue-50 to-cyan-50 rounded-xl p-6 hover:shadow-lg transition-all border-2 border-blue-100 hover:border-blue-300">
                <h3 className="text-xl font-bold text-gray-900 mb-3 flex items-center gap-3">
                  <div className="w-8 h-8 rounded-full bg-blue-600 flex items-center justify-center text-white font-bold">
                    Q
                  </div>
                  {faq.q}
                </h3>
                <p className="text-lg text-gray-700 ml-11">{faq.a}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* REGISTRATION FORM */}
      <section id="registration-form" className="py-20 px-6 bg-gradient-to-br from-blue-600 to-cyan-600">
        <div className="container mx-auto max-w-2xl">
          <div className="text-center mb-8">
            <h2 className="text-4xl font-bold text-white mb-4">Boost Your Social Media Today!</h2>
            <p className="text-xl text-blue-100">Limited seats for December 7, 2025</p>
          </div>
          <CourseLeadForm courseName="Social Media Booster Workshop" formTitle="Register for Workshop" />
        </div>
      </section>

      {/* FOOTER */}
      <footer className="py-16 px-6 bg-gradient-to-r from-gray-900 via-gray-800 to-gray-900 text-white relative overflow-hidden">
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1451187580459-43490279c0fa?w=1920&h=1080&fit=crop&q=80"
            alt="Footer background"
            className="w-full h-full object-cover opacity-5"
          />
        </div>
        <div className="container mx-auto max-w-7xl relative z-10">
          <div className="text-center">
            <div className="flex items-center justify-center gap-3 mb-6">
              <div className="w-12 h-12 rounded-xl bg-gradient-to-br from-blue-700 to-cyan-700 border-2 border-yellow-400/50 flex items-center justify-center">
                <Sparkles className="w-7 h-7 text-yellow-400" />
              </div>
              <div>
                <h3 className="font-bold text-xl">AI School for All</h3>
                <p className="text-xs text-gray-400">by Workshop Library</p>
              </div>
            </div>
            <p className="text-gray-300 mb-6">Professional AI and Creative Skills Education</p>
            <div className="flex items-center justify-center gap-3 mb-6">
              <Shield className="w-5 h-5 text-yellow-400" />
              <span className="text-sm text-gray-300 font-semibold">WorkshopLibrary Partner</span>
            </div>
            <p className="text-xs text-gray-500">
              © 2026 AI School for All. All rights reserved.
            </p>
          </div>
        </div>
      </footer>

      {/* Chatbot */}
      <Chatbot />
    </div>
  );
};

export default WorkshopSocialMedia;
