import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { 
  Calendar, 
  Clock, 
  Video,
  Users, 
  Globe,
  Sparkles,
  TrendingUp,
  CheckCircle2,
  Award,
  Download,
  ArrowRight,
  Target,
  Briefcase,
  FileText,
  Zap,
  Star,
  BookOpen,
  Shield,
  User,
  Mail,
  Linkedin,
  MessageSquare
} from 'lucide-react';
import CourseLeadForm from '../components/CourseLeadForm';
import Chatbot from '../components/Chatbot';

const WorkshopResumePortfolio = () => {
  const navigate = useNavigate();
  const [timeLeft, setTimeLeft] = useState({ days: 0, hours: 0, minutes: 0, seconds: 0 });

  useEffect(() => {
    // Workshop starts December 13, 2025 at 12:00 AM EST
    const workshopDate = new Date('2025-12-13T05:00:00Z'); // 12:00 AM EST = 05:00 UTC
    const targetDate = workshopDate.getTime();

    const updateCountdown = () => {
      const now = new Date().getTime();
      const distance = targetDate - now;

      if (distance > 0) {
        setTimeLeft({
          days: Math.floor(distance / (1000 * 60 * 60 * 24)),
          hours: Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60)),
          minutes: Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60)),
          seconds: Math.floor((distance % (1000 * 60)) / 1000)
        });
      } else {
        setTimeLeft({ days: 0, hours: 0, minutes: 0, seconds: 0 });
      }
    };

    updateCountdown();
    const interval = setInterval(updateCountdown, 1000);
    return () => clearInterval(interval);
  }, []);

  const scrollToForm = () => {
    const formElement = document.getElementById('registration-form');
    if (formElement) {
      formElement.scrollIntoView({ behavior: 'smooth', block: 'start' });
    }
  };

  const scrollToPricing = () => {
    const pricingElement = document.getElementById('pricing-section');
    if (pricingElement) {
      pricingElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
    }
  };

  const modules = [
    "Understanding modern recruitment + ATS systems",
    "Writing ultra-professional resumes using AI prompts",
    "Creating achievement-based bullet points",
    "Tailoring resumes for specific job roles",
    "Writing strong cover letters within minutes",
    "Building an online/PDF portfolio using AI tools",
    "LinkedIn optimization for career growth",
    "AI-based interview preparation (mock + coaching)"
  ];

  const stats = [
    { value: "98%", label: "Fortune 500 use ATS filters" },
    { value: "35%", label: "Higher interview rates with AI resumes" },
    { value: "6-7 sec", label: "Recruiters spend scanning resume" },
    { value: "2.6×", label: "Better hiring chances with portfolio" }
  ];

  const tools = [
    { name: 'ChatGPT', icon: '🤖' },
    { name: 'Rezi AI', icon: '📄' },
    { name: 'Resume.io', icon: '📝' },
    { name: 'Canva Templates', icon: '🎨' },
    { name: 'Notion AI', icon: '📋' },
    { name: 'LinkedIn Tools', icon: '💼' },
    { name: 'Interview AI', icon: '🎯' }
  ];

  const projects = [
    "Create 2 tailored resumes (based on different job roles)",
    "Write 1 complete cover letter using AI",
    "Build a PDF or website portfolio",
    "Rewrite LinkedIn About + Headline + Experience",
    "Practice 10 interview questions with AI",
    "Create your Job-Readiness Master Folder"
  ];

  const careerOutcomes = [
    "ATS-Ready Job Applicant",
    "Professional Resume Writer (freelance option)",
    "Portfolio-Ready Candidate",
    "LinkedIn-Optimized Professional",
    "Interview-Ready Applicant",
    "Industry-Ready Fresh Talent"
  ];

  const whoShouldJoin = [
    "Freshers & college students",
    "Job seekers in any field",
    "Professionals planning a career switch",
    "Freelancers wanting a polished profile",
    "Anyone applying for internships",
    "Working professionals wanting to upgrade their identity"
  ];

  return (
    <div className="min-h-screen bg-white">
      {/* Navigation */}
      <nav className="fixed top-4 left-1/2 transform -translate-x-1/2 z-50 w-[95%] max-w-7xl">
        <div className="bg-black/30 backdrop-blur-lg rounded-full border border-white/20 px-8 py-3 shadow-2xl">
          <div className="flex items-center justify-between">
            <button onClick={() => navigate('/')} className="text-white font-bold text-lg hover:text-yellow-400 transition-colors">
              AI School for All
            </button>
            <div className="hidden md:flex items-center gap-6">
              <button onClick={() => navigate('/#home')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Home</button>
              <button onClick={() => navigate('/#workshops')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Workshops</button>
              <button 
                onClick={scrollToForm}
                className="bg-yellow-400 hover:bg-yellow-500 text-gray-900 px-6 py-2 rounded-full font-bold transition-all text-sm"
              >
                Register Now
              </button>
            </div>
          </div>
        </div>
      </nav>

      {/* EVENT-STYLE HERO BANNER */}
      <section className="relative min-h-screen flex items-center justify-center overflow-hidden pt-20">
        {/* Hero Background Image */}
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1586281380349-632531db7ed4?w=1920&h=1080&fit=crop&q=80"
            alt="Professional resume and portfolio"
            className="w-full h-full object-cover brightness-110"
          />
          <div className="absolute inset-0 bg-gradient-to-br from-slate-700/80 via-blue-700/75 to-indigo-700/80"></div>
        </div>

        {/* Animated particles */}
        <div className="absolute inset-0 opacity-20 z-10">
          <div className="absolute top-20 left-20 w-72 h-72 bg-blue-400 rounded-full blur-3xl animate-pulse"></div>
          <div className="absolute bottom-20 right-20 w-96 h-96 bg-indigo-400 rounded-full blur-3xl animate-pulse delay-1000"></div>
          <div className="absolute top-1/2 left-1/2 w-64 h-64 bg-slate-400 rounded-full blur-3xl animate-pulse delay-500"></div>
        </div>

        <div className="container mx-auto px-6 relative z-20">
          <div className="max-w-5xl mx-auto text-center">
            {/* Event Badge */}
            <div className="inline-flex items-center gap-2 bg-white/20 backdrop-blur-md px-6 py-3 rounded-full mb-8 border border-white/30">
              <Briefcase className="w-5 h-5 text-white" />
              <span className="text-white font-bold text-sm uppercase tracking-wider">💼 Career-Transforming Workshop</span>
            </div>

            {/* Main Title */}
            <h1 className="text-5xl md:text-7xl font-black text-white mb-6 leading-tight drop-shadow-2xl">
              AI-Powered Resume, Portfolio
              <br />
              <span className="text-yellow-400">& Job-Readiness Builder</span>
            </h1>

            <p className="text-2xl text-white/90 mb-10 max-w-3xl mx-auto font-medium">
              Build ATS-ready resume · Professional portfolio · Interview confidence!
            </p>

            {/* Event Quick Info */}
            <div className="grid md:grid-cols-4 gap-4 mb-12 max-w-4xl mx-auto">
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Calendar className="w-8 h-8 text-white mx-auto mb-2" />
                <div className="text-sm text-white/80 mb-1">Next Cohort</div>
                <div className="text-lg font-bold text-white">13 Dec 2025</div>
                <div className="text-xs text-white/70">(Saturday)</div>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Clock className="w-8 h-8 text-white mx-auto mb-2" />
                <div className="text-sm text-white/80 mb-1">Duration</div>
                <div className="text-lg font-bold text-white">4 Hours</div>
                <div className="text-xs text-white/70">Career-Focused</div>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Video className="w-8 h-8 text-white mx-auto mb-2" />
                <div className="text-sm text-white/80 mb-1">Format</div>
                <div className="text-lg font-bold text-white">Live Online</div>
                <div className="text-xs text-white/70">Interactive</div>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Users className="w-8 h-8 text-white mx-auto mb-2" />
                <div className="text-sm text-white/80 mb-1">Seats Left</div>
                <div className="text-lg font-bold text-yellow-400">Limited</div>
                <div className="text-xs text-white/70">Filling Fast</div>
              </div>
            </div>

            {/* Countdown Timer */}
            <div className="mb-10">
              <div className="text-white/90 text-sm uppercase tracking-wider mb-4 font-bold">⏰ Enrollment Begins In</div>
              <div className="flex gap-4 justify-center">
                {Object.entries(timeLeft).map(([unit, value]) => (
                  <div key={unit} className="bg-white/10 backdrop-blur-md border border-white/20 rounded-2xl p-6 min-w-[100px]">
                    <div className="text-4xl font-black text-white mb-1">{value.toString().padStart(2, '0')}</div>
                    <div className="text-xs text-white/70 uppercase tracking-wider">{unit}</div>
                  </div>
                ))}
              </div>
            </div>

            {/* CTA Buttons */}
            <div className="flex flex-col sm:flex-row gap-4 justify-center items-center">
              <button 
                onClick={scrollToForm}
                className="group bg-white hover:bg-yellow-400 text-blue-700 px-10 py-5 rounded-full font-bold text-lg shadow-2xl hover:shadow-3xl transition-all hover:scale-105 flex items-center gap-2"
              >
                Secure Your Spot Now
                <ArrowRight className="w-5 h-5 group-hover:translate-x-1 transition-transform" />
              </button>
              <button 
                onClick={() => document.getElementById('modules-section')?.scrollIntoView({ behavior: 'smooth' })}
                className="bg-white/10 backdrop-blur-md border-2 border-white/30 text-white px-10 py-5 rounded-full font-bold text-lg hover:bg-white/20 transition-all flex items-center gap-2"
              >
                <Download className="w-5 h-5" />
                View Curriculum
              </button>
            </div>

            {/* Trust Badge */}
            <div className="mt-12 flex items-center justify-center gap-6 flex-wrap">
              <div className="flex items-center gap-2 text-white/90">
                <Star className="w-5 h-5 text-yellow-400 fill-yellow-400" />
                <span className="font-semibold">4.9/5 Rating</span>
              </div>
              <div className="w-1 h-1 bg-white/50 rounded-full"></div>
              <div className="flex items-center gap-2 text-white/90">
                <Users className="w-5 h-5" />
                <span className="font-semibold">Thousands Placed</span>
              </div>
              <div className="w-1 h-1 bg-white/50 rounded-full"></div>
              <div className="flex items-center gap-2 text-white/90">
                <Award className="w-5 h-5" />
                <span className="font-semibold">Certificate Included</span>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Promotional Banner - Limited Time Deal */}
      <section className="py-6 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <button
            onClick={scrollToPricing}
            className="w-full bg-gradient-to-r from-orange-500 to-red-500 hover:from-orange-600 hover:to-red-600 text-white px-8 py-5 rounded-full font-bold text-lg shadow-xl hover:shadow-2xl transition-all hover:scale-105 flex items-center justify-center gap-3"
          >
            <span className="text-2xl">🔥</span>
            <span>Limited-Time Deal — 50% OFF</span>
            <ArrowRight className="w-6 h-6" />
          </button>
        </div>
      </section>

      {/* Industry Stats Section */}
      <section className="py-20 px-6 bg-gradient-to-b from-white to-slate-50">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">
              Why AI-Powered Job Readiness Matters
            </h2>
            <p className="text-xl text-gray-600">Modern hiring requires modern preparation</p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6 mb-12">
            {stats.map((stat, index) => (
              <div key={index} className="bg-white rounded-2xl p-8 border-2 border-slate-100 hover:border-blue-300 hover:shadow-xl transition-all text-center">
                <div className="text-4xl font-bold bg-gradient-to-r from-slate-700 to-blue-700 bg-clip-text text-transparent mb-2">
                  {stat.value}
                </div>
                <div className="text-sm text-gray-600">{stat.label}</div>
              </div>
            ))}
          </div>

          <div className="bg-gradient-to-r from-slate-100 to-blue-100 rounded-3xl p-8 border-2 border-slate-200 text-center">
            <div className="flex items-center justify-center gap-3 mb-3">
              <CheckCircle2 className="w-8 h-8 text-blue-700" />
              <h3 className="text-2xl font-bold text-gray-900">LinkedIn Profile Impact</h3>
            </div>
            <p className="text-5xl font-black bg-gradient-to-r from-slate-700 to-blue-700 bg-clip-text text-transparent mb-3">
              40% More Job Views
            </p>
            <p className="text-gray-700">Optimized profiles receive significantly higher visibility from recruiters</p>
          </div>
        </div>
      </section>

      {/* What You'll Learn */}
      <section id="modules-section" className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">What You Will Learn</h2>
            <p className="text-xl text-gray-600">8 essential modules for complete job readiness</p>
          </div>

          <div className="grid md:grid-cols-2 gap-6">
            {modules.map((module, index) => (
              <div key={index} className="flex items-start gap-4 bg-gradient-to-br from-slate-50 to-white rounded-2xl p-6 border-2 border-slate-100 hover:border-blue-300 hover:shadow-lg transition-all group">
                <div className="bg-gradient-to-br from-slate-700 to-blue-700 text-white rounded-xl w-10 h-10 flex items-center justify-center font-bold flex-shrink-0 group-hover:scale-110 transition-transform">
                  {index + 1}
                </div>
                <p className="text-gray-900 font-semibold text-lg pt-1">{module}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Job-Ready Kit Visual */}
      <section className="py-20 px-6 bg-gradient-to-b from-slate-50 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Your Complete Job-Ready Kit</h2>
            <p className="text-xl text-gray-600">Walk away with everything you need to land your dream job</p>
          </div>

          <div className="bg-white rounded-3xl overflow-hidden shadow-xl border-2 border-gray-100">
            <div className="bg-gradient-to-r from-slate-700 to-blue-700 text-white px-8 py-6 text-center">
              <h3 className="text-3xl font-bold">Everything You'll Build in 4 Hours</h3>
            </div>
            
            <div className="p-10">
              <div className="grid md:grid-cols-3 gap-6">
                {[
                  { 
                    title: "Professional Documents", 
                    icon: <FileText className="w-8 h-8" />,
                    items: ["2× Tailored Resumes", "ATS-Optimized Format", "Professional Cover Letter", "Achievement-Based Bullets", "Industry Keywords", "Clean Design Template"] 
                  },
                  { 
                    title: "Online Presence", 
                    icon: <Globe className="w-8 h-8" />,
                    items: ["PDF/Website Portfolio", "LinkedIn Profile Rewrite", "Optimized Headline", "Compelling About Section", "Project Showcase", "Skills Endorsements"] 
                  },
                  { 
                    title: "Interview Prep", 
                    icon: <MessageSquare className="w-8 h-8" />,
                    items: ["10 Practice Questions", "STAR Method Answers", "AI Mock Interview", "Confidence Tips", "Body Language Guide", "Common Mistakes to Avoid"] 
                  }
                ].map((category, idx) => (
                  <div key={idx} className="bg-gradient-to-br from-slate-50 to-blue-50 rounded-2xl p-6 border-2 border-slate-200">
                    <div className="flex items-center gap-3 mb-4">
                      <div className="bg-gradient-to-br from-slate-700 to-blue-700 text-white rounded-xl p-3">
                        {category.icon}
                      </div>
                      <h4 className="text-xl font-bold text-slate-700">{category.title}</h4>
                    </div>
                    <ul className="space-y-2">
                      {category.items.map((item, i) => (
                        <li key={i} className="flex items-center gap-2 text-gray-700">
                          <CheckCircle2 className="w-4 h-4 text-blue-600 flex-shrink-0" />
                          <span>{item}</span>
                        </li>
                      ))}
                    </ul>
                  </div>
                ))}
              </div>
              
              <div className="mt-8 text-center bg-gradient-to-r from-slate-100 to-blue-100 rounded-2xl p-6 border-2 border-slate-200">
                <p className="text-2xl font-bold text-blue-700">
                  Ready to apply for jobs immediately after the workshop!
                </p>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Hands-On Projects */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Hands-On Projects</h2>
            <p className="text-xl text-gray-600">Build real job-ready assets during the workshop</p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
            {projects.map((project, index) => (
              <div key={index} className="bg-gradient-to-br from-slate-50 to-white rounded-2xl p-6 border-2 border-slate-100 hover:shadow-lg transition-all">
                <div className="flex items-start gap-3">
                  <div className="bg-slate-700 text-white rounded-full w-8 h-8 flex items-center justify-center font-bold text-sm flex-shrink-0">
                    {index + 1}
                  </div>
                  <p className="text-gray-900 font-semibold pt-1">{project}</p>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Who Should Join */}
      <section className="py-20 px-6 bg-gradient-to-b from-slate-50 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Who Should Join</h2>
            <p className="text-xl text-gray-600">Perfect for anyone building or upgrading their career</p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
            {whoShouldJoin.map((person, index) => (
              <div key={index} className="flex items-center gap-3 bg-white rounded-2xl p-6 border-2 border-slate-100 hover:border-blue-300 hover:shadow-lg transition-all">
                <CheckCircle2 className="w-6 h-6 text-blue-600 flex-shrink-0" />
                <p className="text-gray-800 font-semibold">{person}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Career Outcomes */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">You Will Become</h2>
            <p className="text-xl text-gray-600">Transform into a fully prepared job market candidate</p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
            {careerOutcomes.map((outcome, index) => (
              <div key={index} className="bg-gradient-to-br from-slate-50 to-white rounded-2xl p-6 border-2 border-slate-100 hover:border-blue-300 hover:shadow-xl transition-all group text-center">
                <div className="bg-gradient-to-br from-slate-700 to-blue-700 text-white rounded-xl w-12 h-12 flex items-center justify-center mx-auto mb-4 group-hover:scale-110 transition-transform">
                  <Target className="w-6 h-6" />
                </div>
                <h3 className="text-lg font-bold text-gray-900">{outcome}</h3>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Tools You Will Master */}
      <section className="py-20 px-6 bg-gradient-to-b from-slate-50 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Tools You Will Master</h2>
            <p className="text-xl text-gray-600">Professional-grade AI tools for job readiness</p>
          </div>

          <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-7 gap-6">
            {tools.map((tool, index) => (
              <div key={index} className="bg-white rounded-2xl p-6 border-2 border-slate-100 hover:border-blue-300 hover:shadow-lg transition-all text-center group">
                <div className="text-4xl mb-3 group-hover:scale-110 transition-transform">{tool.icon}</div>
                <p className="font-bold text-gray-900 text-sm">{tool.name}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* About Trainers */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Meet Your Trainers</h2>
            <p className="text-xl text-gray-600">Learn from certified career & communication experts</p>
          </div>

          <div className="bg-gradient-to-br from-slate-50 to-blue-50 rounded-3xl p-10 border-2 border-slate-200">
            <div className="flex flex-col md:flex-row items-center gap-8">
              <div className="flex-shrink-0">
                <img 
                  src="https://images.unsplash.com/photo-1507003211169-0a1dd7228f2d?w=300&h=300&fit=crop&q=80"
                  alt="Expert trainer"
                  className="w-48 h-48 rounded-full object-cover border-4 border-blue-300"
                />
              </div>
              <div>
                <div className="flex items-center gap-2 mb-4">
                  <Award className="w-6 h-6 text-blue-700" />
                  <span className="text-blue-700 font-bold text-lg">WorkshopLibrary-Certified Experts</span>
                </div>
                <p className="text-gray-700 text-lg mb-6 leading-relaxed">
                  Delivered by career & communication trainers with <span className="font-bold text-blue-700">10-15 years of experience</span> helping thousands of learners secure internships, jobs, and interview opportunities.
                </p>
                <div className="grid md:grid-cols-2 gap-4">
                  <div className="flex items-center gap-2 text-gray-700">
                    <CheckCircle2 className="w-5 h-5 text-blue-600" />
                    Resume writing & personal branding
                  </div>
                  <div className="flex items-center gap-2 text-gray-700">
                    <CheckCircle2 className="w-5 h-5 text-blue-600" />
                    LinkedIn optimization
                  </div>
                  <div className="flex items-center gap-2 text-gray-700">
                    <CheckCircle2 className="w-5 h-5 text-blue-600" />
                    Interview preparation
                  </div>
                  <div className="flex items-center gap-2 text-gray-700">
                    <CheckCircle2 className="w-5 h-5 text-blue-600" />
                    HR & recruitment insights
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Pricing Section */}
      <section id="pricing-section" className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-12">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Workshop Pricing</h2>
            <p className="text-xl text-gray-600">Special limited-time offer for early birds</p>
          </div>

          <div className="bg-gradient-to-br from-slate-50 to-blue-50 rounded-3xl p-10 border-2 border-slate-200 shadow-xl">
            {/* Premium DEAL BADGE */}
            <div className="relative">
              <div className="bg-gradient-to-br from-white via-blue-50/30 to-purple-50/30 rounded-3xl p-10 shadow-2xl border-2 border-blue-200/50 relative overflow-hidden">
                {/* Subtle glow effect */}
                <div className="absolute inset-0 bg-gradient-to-r from-blue-400/10 to-purple-400/10 rounded-3xl blur-xl"></div>

                {/* Content */}
                <div className="relative z-10">
                  {/* Top Badge */}
                  <div className="flex justify-center mb-6">
                    <div className="bg-gradient-to-r from-orange-500 to-red-500 text-white px-6 py-3 rounded-full font-bold text-sm shadow-lg inline-flex items-center gap-2">
                      <span>🔥</span>
                      <span>Limited-Time Deal — 50% OFF</span>
                    </div>
                  </div>

                  {/* Strikethrough Old Price */}
                  <div className="text-center mb-4">
                    <p className="text-gray-400 text-lg line-through">
                      $100 / 365 AED
                    </p>
                  </div>

                  {/* Main Price - BIG & BOLD */}
                  <div className="text-center mb-4">
                    <div className="flex items-center justify-center gap-4 mb-3">
                      <p className="text-6xl md:text-7xl font-black text-blue-700">
                        $50
                      </p>
                      <span className="text-4xl text-gray-600 font-bold">/</span>
                      <p className="text-6xl md:text-7xl font-black text-blue-700">
                        185 AED
                      </p>
                    </div>
                    <p className="text-gray-700 font-semibold text-lg">
                      For the Next 15 Days Only — Early Bird Offer
                    </p>
                  </div>

                  {/* Subtext */}
                  <div className="text-center mb-8">
                    <p className="text-gray-600 text-base">
                      Enroll Now & Unlock Your Instant 50% Discount — Limited Seats
                    </p>
                  </div>

                  {/* CTA Button */}
                  <div className="text-center">
                    <button 
                      onClick={scrollToForm}
                      className="w-full md:w-auto bg-gradient-to-r from-yellow-400 to-amber-500 hover:from-yellow-500 hover:to-amber-600 text-gray-900 px-12 py-5 rounded-full font-bold text-xl shadow-2xl hover:shadow-3xl transition-all hover:scale-105 inline-flex items-center justify-center gap-3"
                    >
                      Claim This Deal
                      <ArrowRight className="w-6 h-6" />
                    </button>
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* Additional Group & Corporate Discounts */}
          <div className="mt-8">
            <h3 className="text-2xl font-bold text-gray-900 text-center mb-6">
              Additional Group & Corporate Discounts
            </h3>

            <div className="bg-white rounded-2xl p-8 shadow-lg border border-gray-200">
              <div className="flex flex-col md:flex-row items-start gap-6">
                {/* Icon */}
                <div className="flex-shrink-0">
                  <div className="w-16 h-16 bg-slate-100 rounded-xl flex items-center justify-center">
                    <Users className="w-8 h-8 text-slate-700" />
                  </div>
                </div>

                {/* Content */}
                <div className="flex-1">
                  <h4 className="text-xl font-bold text-gray-900 mb-4">
                    Save More When You Learn Together
                  </h4>

                  <ul className="space-y-3 mb-6">
                    <li className="flex items-start gap-3 text-gray-700">
                      <CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" />
                      <span>10% extra discount for group bookings (2 or more participants)</span>
                    </li>
                    <li className="flex items-start gap-3 text-gray-700">
                      <CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" />
                      <span>Special pricing available for corporate teams</span>
                    </li>
                    <li className="flex items-start gap-3 text-gray-700">
                      <CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" />
                      <span>Dedicated support and invoicing for companies</span>
                    </li>
                  </ul>

                  <button 
                    onClick={scrollToForm}
                    className="border-2 border-slate-700 text-slate-700 hover:bg-slate-700 hover:text-white px-8 py-3 rounded-xl font-semibold transition-all inline-flex items-center gap-2"
                  >
                    Request Corporate Pricing
                    <ArrowRight className="w-5 h-5" />
                  </button>
                </div>
              </div>

              {/* Notes */}
              <div className="mt-6 pt-6 border-t border-gray-200">
                <p className="text-sm text-gray-600 mb-2">
                  ✓ Discounts apply on top of the current 15-day promotional price.
                </p>
                <p className="text-sm text-gray-600">
                  ✓ For larger teams, customized offers are available.
                </p>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Certification */}
      <section className="py-20 px-6 bg-gradient-to-b from-slate-50 to-white">
        <div className="container mx-auto max-w-4xl">
          <div className="bg-gradient-to-br from-slate-700 to-blue-700 rounded-3xl p-12 text-white text-center shadow-2xl">
            <Award className="w-20 h-20 mx-auto mb-6" />
            <h2 className="text-4xl font-bold mb-4">Certification Included</h2>
            <div className="bg-white/20 backdrop-blur-sm rounded-2xl p-6 inline-block mb-4">
              <p className="text-2xl font-bold">🎓 AI-Powered Resume & Portfolio Builder Certificate</p>
            </div>
            <p className="text-xl text-white/90">
              Issued by Workshop Library · Recognized by recruiters, colleges & training networks
            </p>
          </div>
        </div>
      </section>

      {/* FAQs */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Frequently Asked Questions</h2>
          </div>

          <div className="space-y-4">
            {[
              {
                q: "Is this workshop suitable for beginners?",
                a: "Yes, perfect for freshers & job seekers with no prior experience."
              },
              {
                q: "Will I get a resume by the end?",
                a: "Yes — you will finish with a polished resume + cover letter ready to use."
              },
              {
                q: "Can I use this portfolio for jobs?",
                a: "Absolutely — it's created using industry-standard formats."
              },
              {
                q: "Are recordings included?",
                a: "Yes, 30-day video access to all workshop content."
              },
              {
                q: "Will you help rewrite my LinkedIn profile?",
                a: "Yes — live LinkedIn optimization is included in the workshop."
              }
            ].map((faq, index) => (
              <div key={index} className="bg-gradient-to-r from-slate-50 to-white rounded-2xl p-6 border-2 border-slate-100 hover:border-blue-300 transition-all">
                <h3 className="text-xl font-bold text-gray-900 mb-3 flex items-start gap-3">
                  <span className="text-blue-700">Q.</span>
                  {faq.q}
                </h3>
                <p className="text-gray-700 text-lg pl-8">{faq.a}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Final CTA */}
      <section className="py-20 px-6 bg-gradient-to-br from-slate-700 via-blue-700 to-slate-800 text-white">
        <div className="container mx-auto max-w-4xl text-center">
          <h2 className="text-5xl font-bold mb-6">Ready to Become Job-Ready?</h2>
          <p className="text-2xl text-white/90 mb-10">
            Join thousands who've landed their dream jobs with AI-powered preparation
          </p>
          
          <button 
            onClick={scrollToForm}
            className="bg-white text-blue-700 px-12 py-6 rounded-full font-bold text-xl hover:bg-yellow-400 transition-all shadow-2xl hover:scale-105 inline-flex items-center gap-3"
          >
            Enroll Now - Limited Seats
            <ArrowRight className="w-6 h-6" />
          </button>
          
          <div className="mt-12 grid grid-cols-3 gap-6 max-w-2xl mx-auto">
            <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-6 border border-white/20">
              <p className="text-3xl font-bold mb-1">4 Hours</p>
              <p className="text-white/80 text-sm">Career-Focused</p>
            </div>
            <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-6 border border-white/20">
              <p className="text-3xl font-bold mb-1">Complete Kit</p>
              <p className="text-white/80 text-sm">Job-Ready</p>
            </div>
            <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-6 border border-white/20">
              <p className="text-3xl font-bold mb-1">6 Projects</p>
              <p className="text-white/80 text-sm">Portfolio Ready</p>
            </div>
          </div>
        </div>
      </section>

      {/* Registration Form Section */}
      <section id="registration-form" className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-2xl">
          <div className="text-center mb-12">
            <h2 className="text-4xl font-bold text-gray-900 mb-4">Secure Your Spot</h2>
            <p className="text-xl text-gray-600">Fill in your details to reserve your seat</p>
          </div>
          <CourseLeadForm />
        </div>
      </section>

      {/* Chatbot */}
      <Chatbot />
    </div>
  );
};

export default WorkshopResumePortfolio;
