import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { 
  Calendar, Clock, Video, Users, Globe, Sparkles, TrendingUp, 
  Award, CheckCircle2, Download, ArrowRight, Zap, Target, Star,
  MessageSquare, Brain, Wand2, Code, Image as ImageIcon, FileText,
  BookOpen, Shield, Lightbulb, Cpu, Layers, Command
} from 'lucide-react';
import CourseLeadForm from '../components/CourseLeadForm';
import Chatbot from '../components/Chatbot';

const WorkshopPromptEngineering = () => {
  const navigate = useNavigate();
  const [timeLeft, setTimeLeft] = useState({ days: 0, hours: 0, minutes: 0, seconds: 0 });

  useEffect(() => {
    // Workshop starts December 14, 2025 at 12:00 AM EST
    const workshopDate = new Date('2025-12-14T05:00:00Z'); // 12:00 AM EST = 05:00 UTC
    const targetDate = workshopDate.getTime();

    const updateCountdown = () => {
      const now = new Date().getTime();
      const distance = targetDate - now;

      if (distance > 0) {
        setTimeLeft({
          days: Math.floor(distance / (1000 * 60 * 60 * 24)),
          hours: Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60)),
          minutes: Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60)),
          seconds: Math.floor((distance % (1000 * 60)) / 1000)
        });
      } else {
        setTimeLeft({ days: 0, hours: 0, minutes: 0, seconds: 0 });
      }
    };

    updateCountdown();
    const interval = setInterval(updateCountdown, 1000);
    return () => clearInterval(interval);
  }, []);

  const scrollToForm = () => {
    const formElement = document.getElementById('registration-form');
    if (formElement) {
      formElement.scrollIntoView({ behavior: 'smooth', block: 'start' });
    }
  };

  const scrollToPricing = () => {
    const pricingElement = document.getElementById('pricing-section');
    if (pricingElement) {
      pricingElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
    }
  };

  const modules = [
    'Understanding how AI interprets prompts',
    'Role-based prompting for clarity',
    'Multi-step instruction frameworks',
    'Corrections & refinement loops',
    'Writing prompts for content creation',
    'Image prompts for AI design tools',
    'Prompting for productivity & business tasks',
    'Automation-friendly prompting styles'
  ];

  const stats = [
    { value: '#1', label: 'Fastest-growing AI skill globally' },
    { value: '85%', label: 'Companies use AI for core tasks' },
    { value: '20-40%', label: 'Higher salaries with AI skills' },
    { value: '3x', label: 'Productivity boost with AI' }
  ];

  const tools = [
    { name: 'ChatGPT', icon: '🤖' },
    { name: 'Gemini AI', icon: '✨' },
    { name: 'Claude AI', icon: '💬' },
    { name: 'Canva AI', icon: '🎨' },
    { name: 'Midjourney', icon: '🖼️' },
    { name: 'Notion AI', icon: '📝' },
    { name: 'Perplexity AI', icon: '🔍' }
  ];

  const projects = [
    'Write 15+ professional prompts',
    'Create 5-step chain-of-thought workflows',
    'Generate content: ad copy, blogs, reels',
    'Create AI images using prompt structures',
    'Build your own Prompt Library starter pack'
  ];

  return (
    <div className="min-h-screen bg-white">
      {/* Navigation */}
      <nav className="fixed top-4 left-1/2 transform -translate-x-1/2 z-50 w-[95%] max-w-7xl">
        <div className="bg-black/30 backdrop-blur-lg rounded-full border border-white/20 px-8 py-3 shadow-2xl">
          <div className="flex items-center justify-between">
            <button onClick={() => navigate('/')} className="text-white font-bold text-lg hover:text-yellow-400 transition-colors">
              AI School for All
            </button>
            <div className="hidden md:flex items-center gap-6">
              <button onClick={() => navigate('/#home')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Home</button>
              <button onClick={() => navigate('/#workshops')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Workshops</button>
              <button 
                onClick={scrollToForm}
                className="bg-yellow-400 hover:bg-yellow-500 text-gray-900 px-6 py-2 rounded-full font-bold transition-all text-sm"
              >
                Register Now
              </button>
            </div>
          </div>
        </div>
      </nav>

      {/* AI-POWERED HERO */}
      <section className="relative min-h-screen flex items-center justify-center overflow-hidden pt-20">
        {/* ChatGPT Prominent Background */}
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1488590528505-98d2b5aba04b?w=1920&h=1080&fit=crop&q=80"
            alt="Prompt Engineering Laptop"
            className="w-full h-full object-cover brightness-120"
          />
          <div className="absolute inset-0 bg-gradient-to-br from-emerald-600/50 via-green-500/45 to-purple-600/55"></div>
        </div>

        {/* Animated AI particles */}
        <div className="absolute inset-0 opacity-20 z-10">
          <div className="absolute top-20 left-20 w-72 h-72 bg-yellow-400 rounded-full blur-3xl animate-pulse"></div>
          <div className="absolute bottom-20 right-20 w-96 h-96 bg-purple-400 rounded-full blur-3xl animate-pulse delay-1000"></div>
          <div className="absolute top-1/2 left-1/2 w-64 h-64 bg-green-400 rounded-full blur-3xl animate-pulse delay-500"></div>
        </div>

        <div className="container mx-auto px-6 relative z-20">
          <div className="max-w-5xl mx-auto text-center">
            {/* Event Badge */}
            <div className="inline-flex items-center gap-2 bg-white/20 backdrop-blur-md px-6 py-3 rounded-full mb-8 border border-white/30">
              <Brain className="w-5 h-5 text-white" />
              <span className="text-white font-bold text-sm uppercase tracking-wider">🔥 Hottest AI Skill in 2025</span>
            </div>

            {/* Main Title */}
            <h1 className="text-5xl md:text-7xl font-black text-white mb-6 leading-tight drop-shadow-2xl">
              Prompt Engineering
              <br />
              <span className="text-yellow-400">for Beginners</span>
            </h1>

            <p className="text-2xl text-white/90 mb-10 max-w-3xl mx-auto font-medium">
              Master the #1 AI skill - Make ChatGPT, Gemini & all AI tools work like magic!
            </p>

            {/* Event Quick Info */}
            <div className="grid md:grid-cols-4 gap-4 mb-12 max-w-4xl mx-auto">
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Calendar className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">14 Dec 2025</p>
                <p className="text-white/80 text-sm">Sunday</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Clock className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">3 Hours</p>
                <p className="text-white/80 text-sm">Power-Packed</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Video className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">Live Online</p>
                <p className="text-white/80 text-sm">Interactive</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Zap className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">Hands-On</p>
                <p className="text-white/80 text-sm">15+ Prompts</p>
              </div>
            </div>

            {/* CTA */}
            <button 
              onClick={scrollToForm}
              className="bg-yellow-400 hover:bg-yellow-500 text-gray-900 px-12 py-6 rounded-2xl font-black text-xl shadow-2xl hover:scale-105 transition-all inline-flex items-center gap-3"
            >
              <Wand2 className="w-6 h-6" />
              Master AI Prompting Now
              <ArrowRight className="w-6 h-6" />
            </button>

            <p className="text-white/90 mt-6 text-sm">
              🔥 <strong>HOTTEST AI SKILL</strong> - Enrollment Opens Soon!
            </p>
          </div>
        </div>
      </section>

      {/* COUNTDOWN TIMER */}
      <section className="py-16 px-6 bg-gradient-to-b from-emerald-900 to-purple-900">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-8">
            <h3 className="text-3xl font-bold text-yellow-400 mb-2">Workshop Starts In</h3>
            <p className="text-gray-200">December 14, 2025 • 12:00 AM EST</p>
          </div>
          <div className="grid grid-cols-4 gap-4 max-w-2xl mx-auto">
            {['days', 'hours', 'minutes', 'seconds'].map((unit) => (
              <div key={unit} className="bg-gradient-to-br from-emerald-700 to-purple-700 rounded-xl p-6 text-center border-2 border-yellow-400/30">
                <div className="text-5xl font-bold text-yellow-400 mb-2">{timeLeft[unit]}</div>
                <div className="text-gray-200 text-sm uppercase tracking-wide">{unit}</div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Promotional Banner - Limited Time Deal */}
      <section className="py-6 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <button onClick={scrollToPricing} className="w-full bg-gradient-to-r from-orange-500 to-red-500 hover:from-orange-600 hover:to-red-600 text-white px-8 py-5 rounded-full font-bold text-lg shadow-xl hover:shadow-2xl transition-all hover:scale-105 flex items-center justify-center gap-3">
            <span className="text-2xl">🔥</span><span>Limited-Time Deal — 50% OFF</span><ArrowRight className="w-6 h-6" />
          </button>
        </div>
      </section>

      {/* WHY THIS WORKSHOP */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="grid md:grid-cols-2 gap-12 items-center">
            <div>
              <div className="inline-flex items-center gap-2 bg-emerald-100 px-4 py-2 rounded-full mb-6">
                <Lightbulb className="w-5 h-5 text-emerald-700" />
                <span className="text-emerald-900 font-bold text-sm">THE MOST IMPORTANT AI SKILL</span>
              </div>
              <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-6">Why Prompt Engineering is EVERYTHING</h2>
              <p className="text-lg text-gray-700 mb-6 leading-relaxed">
                Every AI tool—ChatGPT, Gemini, Midjourney, Canva AI—depends entirely on <strong className="text-emerald-600">how well you communicate with it</strong>. A bad prompt gets mediocre results. A great prompt delivers genius-level outputs.
              </p>
              <p className="text-lg text-gray-700 mb-6 leading-relaxed">
                This workshop teaches you the <strong className="text-emerald-600">science and structure</strong> behind writing powerful prompts that work consistently. You'll learn role-based prompting, step-by-step instructions, refinement loops, and multi-model workflows.
              </p>
              <p className="text-lg text-gray-700 leading-relaxed">
                Walk away with <strong className="text-purple-600">15+ professional prompts</strong>, a starter prompt library, and the ability to make AI work like your personal assistant, researcher, strategist, writer, and designer!
              </p>
            </div>
            <div className="relative">
              <div className="bg-gradient-to-br from-emerald-500 to-purple-600 rounded-3xl p-8 shadow-2xl">
                <img 
                  src="https://images.unsplash.com/photo-1519389950473-47ba0277781c?w=600&h=400&fit=crop&q=80"
                  alt="Writing AI Prompts"
                  className="w-full h-80 object-cover rounded-2xl shadow-xl"
                />
                <div className="absolute -bottom-6 -right-6 bg-white rounded-2xl p-6 shadow-2xl border-4 border-yellow-400">
                  <div className="text-center">
                    <div className="text-4xl font-bold text-emerald-600">3x</div>
                    <div className="text-sm text-gray-700 font-medium">Productivity Boost</div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* BEFORE/AFTER PROMPT TRANSFORMATIONS */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">See The Power of Great Prompts</h2>
            <p className="text-xl text-gray-600">Bad prompts vs. Perfect prompts - Mind-blowing difference!</p>
          </div>

          {/* Example 1: Content Writing Prompt */}
          <div className="mb-16">
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">Content Writing Transformation</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BAD PROMPT ❌
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl bg-white">
                  <div className="bg-gray-800 p-6">
                    <div className="flex items-start gap-3 mb-4">
                      <MessageSquare className="w-6 h-6 text-gray-400" />
                      <div className="flex-1">
                        <p className="text-gray-300 font-mono text-sm mb-3">User:</p>
                        <div className="bg-gray-700 rounded-lg p-4">
                          <p className="text-white text-sm">Write a blog about AI</p>
                        </div>
                      </div>
                    </div>
                  </div>
                  <div className="p-6">
                    <p className="text-gray-400 text-xs mb-3">AI Response:</p>
                    <p className="text-gray-600 text-sm leading-relaxed">
                      "Artificial Intelligence is changing the world. It has many applications in various industries. AI is important for the future. Companies are using AI for automation..."
                    </p>
                    <div className="mt-4 flex items-center gap-2 text-red-500 text-xs">
                      <span className="font-bold">❌ Generic</span>
                      <span>•</span>
                      <span className="font-bold">❌ No structure</span>
                      <span>•</span>
                      <span className="font-bold">❌ Boring</span>
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Vague prompt • Generic output • Unusable content</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  PERFECT PROMPT ✅
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl bg-white">
                  <div className="bg-gray-800 p-6">
                    <div className="flex items-start gap-3 mb-4">
                      <Brain className="w-6 h-6 text-emerald-400" />
                      <div className="flex-1">
                        <p className="text-gray-300 font-mono text-sm mb-3">User:</p>
                        <div className="bg-gray-700 rounded-lg p-4">
                          <p className="text-white text-sm">You are a tech blogger writing for business owners. Write a 500-word blog post about "How AI is Transforming Small Business Marketing in 2025". Include: 1) Hook with a stat, 2) 3 real use cases, 3) Actionable tip, 4) Strong CTA. Tone: conversational but professional.</p>
                        </div>
                      </div>
                    </div>
                  </div>
                  <div className="p-6 bg-gradient-to-br from-emerald-50 to-green-50">
                    <p className="text-gray-400 text-xs mb-3">AI Response:</p>
                    <p className="text-gray-900 font-semibold text-sm leading-relaxed mb-3">
                      "92% of small businesses now use AI for marketing—here's why you should too."
                    </p>
                    <p className="text-gray-700 text-sm leading-relaxed">
                      AI is no longer reserved for tech giants. Today, small businesses are leveraging AI to automate email campaigns, personalize customer experiences, and analyze data faster than ever...
                    </p>
                    <div className="mt-4 flex items-center gap-2 text-green-600 text-xs font-semibold">
                      <span>✅ Structured</span>
                      <span>•</span>
                      <span>✅ Engaging</span>
                      <span>•</span>
                      <span>✅ Ready to publish!</span>
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Structured prompt • Perfect output • Professional quality!</p>
              </div>
            </div>
          </div>

          {/* Example 2: Image Generation Prompt */}
          <div className="mb-16">
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">AI Image Generation Mastery</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BAD PROMPT ❌
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl bg-white">
                  <div className="bg-gray-800 p-6">
                    <div className="flex items-center gap-3 mb-4">
                      <ImageIcon className="w-6 h-6 text-gray-400" />
                      <p className="text-white text-sm font-mono">Generate: "a mountain"</p>
                    </div>
                  </div>
                  <div className="bg-gray-200 h-64 flex items-center justify-center">
                    <img 
                      src="https://images.unsplash.com/photo-1506905925346-21bda4d32df4?w=500&h=300&fit=crop&q=40&sat=-60"
                      alt="Generic mountain"
                      className="w-full h-full object-cover opacity-50"
                    />
                  </div>
                  <div className="p-4">
                    <p className="text-red-500 text-xs font-semibold">❌ Generic • Unclear style • Not usable</p>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Vague prompt • Random result • Unpredictable quality</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  PERFECT PROMPT ✅
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl bg-white">
                  <div className="bg-gray-800 p-6">
                    <div className="flex items-start gap-3 mb-4">
                      <Wand2 className="w-6 h-6 text-emerald-400" />
                      <p className="text-white text-sm font-mono">Generate: "Snow-capped mountain peak at sunrise, cinematic wide angle, dramatic golden lighting, misty valleys below, ultra-detailed, professional photography, 8K quality, National Geographic style"</p>
                    </div>
                  </div>
                  <div className="relative">
                    <img 
                      src="https://images.unsplash.com/photo-1506905925346-21bda4d32df4?w=500&h=300&fit=crop&q=90"
                      alt="Stunning mountain"
                      className="w-full h-64 object-cover"
                    />
                    <div className="absolute top-4 right-4 bg-green-500 text-white px-3 py-1 rounded-full text-xs font-bold">
                      Perfect!
                    </div>
                  </div>
                  <div className="p-4 bg-gradient-to-r from-emerald-50 to-green-50">
                    <p className="text-green-600 text-xs font-semibold">✅ Detailed • Specific style • Magazine-quality result!</p>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Detailed prompt • Stunning result • Professional quality!</p>
              </div>
            </div>
          </div>

          {/* Example 3: Business Task Prompt */}
          <div>
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">Business Task Automation</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BAD PROMPT ❌
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl bg-white">
                  <div className="p-8">
                    <div className="bg-gray-100 rounded-lg p-4 mb-4">
                      <p className="text-gray-700 text-sm mb-2 font-semibold">Prompt:</p>
                      <p className="text-gray-600 text-sm">"Write an email to client"</p>
                    </div>
                    <div className="bg-red-50 rounded-lg p-4">
                      <p className="text-gray-400 text-xs mb-2">Result:</p>
                      <p className="text-gray-600 text-sm">Dear Client, Thank you for your inquiry. We will get back to you soon. Best regards.</p>
                    </div>
                    <div className="mt-4 text-center">
                      <span className="text-red-500 text-xs font-bold">❌ Too generic • No context • Unprofessional</span>
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">No details • No personality • Waste of AI</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  PERFECT PROMPT ✅
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl bg-white">
                  <div className="p-8">
                    <div className="bg-gray-100 rounded-lg p-4 mb-4">
                      <p className="text-gray-700 text-sm mb-2 font-semibold">Prompt:</p>
                      <p className="text-gray-600 text-sm">You are a professional project manager. Write a follow-up email to client "ABC Corp" about their website redesign project. Context: proposal sent 3 days ago, deadline approaching. Tone: friendly but professional. Include: 1) Quick reminder, 2) Offer to schedule call, 3) Highlight 2 key benefits. Keep under 100 words.</p>
                    </div>
                    <div className="bg-gradient-to-br from-emerald-50 to-green-50 rounded-lg p-4">
                      <p className="text-gray-400 text-xs mb-2">Result:</p>
                      <p className="text-gray-800 text-sm font-medium">Hi [Name],</p>
                      <p className="text-gray-700 text-sm mt-2">Just following up on the website redesign proposal we sent on [date]. With your launch deadline approaching, I wanted to ensure we're aligned on next steps.</p>
                      <p className="text-gray-700 text-sm mt-2">Our team can deliver a mobile-optimized site with 50% faster load times—ready to schedule a quick 15-min call this week?</p>
                      <p className="text-gray-700 text-sm mt-2">Best regards, [Your Name]</p>
                    </div>
                    <div className="mt-4 text-center">
                      <span className="text-green-600 text-xs font-bold">✅ Professional • Actionable • Ready to send!</span>
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Clear context • Perfect tone • Business-ready!</p>
              </div>
            </div>
          </div>

          {/* Results Banner */}
          <div className="mt-16 bg-gradient-to-r from-emerald-600 to-purple-600 rounded-3xl p-10 text-center text-white">
            <p className="text-3xl font-bold mb-4">Learn All These Techniques in Just 3 Hours!</p>
            <p className="text-xl text-white/90">Go from beginner to prompt pro - No experience needed!</p>
          </div>
        </div>
      </section>

      {/* AI TOOLS SHOWCASE */}
      <section className="py-20 px-6 bg-gradient-to-b from-emerald-900 to-purple-900">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-white mb-4">Master Prompting for ALL AI Tools</h2>
            <p className="text-xl text-emerald-100">Learn universal prompting that works everywhere</p>
          </div>

          <div className="grid md:grid-cols-3 gap-8 mb-12">
            {/* Text AI Tools */}
            <div className="bg-white/10 backdrop-blur-md rounded-3xl p-8 border-2 border-white/20 hover:border-emerald-400 transition-all">
              <div className="flex items-center justify-center w-16 h-16 rounded-full bg-gradient-to-br from-emerald-400 to-green-500 mb-6 mx-auto">
                <MessageSquare className="w-8 h-8 text-white" />
              </div>
              <h3 className="text-2xl font-bold text-white text-center mb-4">Text AI</h3>
              <ul className="space-y-3 text-white/90">
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 text-emerald-400 flex-shrink-0" />
                  <span>ChatGPT (GPT-4)</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 text-emerald-400 flex-shrink-0" />
                  <span>Google Gemini</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 text-emerald-400 flex-shrink-0" />
                  <span>Claude AI</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 text-emerald-400 flex-shrink-0" />
                  <span>Perplexity AI</span>
                </li>
              </ul>
            </div>

            {/* Image AI Tools */}
            <div className="bg-white/10 backdrop-blur-md rounded-3xl p-8 border-2 border-white/20 hover:border-purple-400 transition-all">
              <div className="flex items-center justify-center w-16 h-16 rounded-full bg-gradient-to-br from-purple-400 to-pink-500 mb-6 mx-auto">
                <ImageIcon className="w-8 h-8 text-white" />
              </div>
              <h3 className="text-2xl font-bold text-white text-center mb-4">Image AI</h3>
              <ul className="space-y-3 text-white/90">
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 text-purple-400 flex-shrink-0" />
                  <span>Midjourney</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 text-purple-400 flex-shrink-0" />
                  <span>DALL-E</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 text-purple-400 flex-shrink-0" />
                  <span>Canva AI</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 text-purple-400 flex-shrink-0" />
                  <span>Stable Diffusion</span>
                </li>
              </ul>
            </div>

            {/* Productivity AI Tools */}
            <div className="bg-white/10 backdrop-blur-md rounded-3xl p-8 border-2 border-white/20 hover:border-yellow-400 transition-all">
              <div className="flex items-center justify-center w-16 h-16 rounded-full bg-gradient-to-br from-yellow-400 to-orange-500 mb-6 mx-auto">
                <Zap className="w-8 h-8 text-white" />
              </div>
              <h3 className="text-2xl font-bold text-white text-center mb-4">Productivity</h3>
              <ul className="space-y-3 text-white/90">
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                  <span>Notion AI</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                  <span>Jasper AI</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                  <span>Copy.ai</span>
                </li>
                <li className="flex items-center gap-2">
                  <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                  <span>Zapier AI</span>
                </li>
              </ul>
            </div>
          </div>

          <div className="text-center">
            <p className="text-xl text-white font-semibold">Learn prompting frameworks that work across ALL these tools!</p>
          </div>
        </div>
      </section>

      {/* INDUSTRY INSIGHTS */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Industry Insights</h2>
            <p className="text-xl text-gray-600">Why prompt engineering is the skill of 2025</p>
          </div>
          <div className="grid md:grid-cols-4 gap-8">
            {stats.map((stat, index) => (
              <div key={index} className="bg-gradient-to-br from-emerald-50 to-green-50 rounded-2xl p-8 shadow-lg hover:shadow-xl transition-all border-2 border-emerald-100">
                <div className="text-5xl font-black text-emerald-600 mb-4">{stat.value}</div>
                <p className="text-gray-700 font-medium leading-relaxed">{stat.label}</p>
              </div>
            ))}
          </div>
          <div className="mt-12 text-center">
            <p className="text-lg text-gray-600">
              <span className="text-emerald-600 font-bold">Global AI adoption</span> expected to reach <strong>75% by 2026</strong>
            </p>
          </div>
        </div>
      </section>

      {/* WHAT YOU'LL LEARN */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">What You'll Master</h2>
            <div className="w-32 h-1 bg-gradient-to-r from-emerald-500 to-purple-600 mx-auto"></div>
          </div>
          
          {/* AI Prompt Gallery */}
          <div className="grid md:grid-cols-3 gap-4 mb-12">
            <img 
              src="https://images.unsplash.com/photo-1488590528505-98d2b5aba04b?w=400&h=300&fit=crop&q=80"
              alt="Laptop prompt writing"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
            <img 
              src="https://images.unsplash.com/photo-1519389950473-47ba0277781c?w=400&h=300&fit=crop&q=80"
              alt="Team writing prompts"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
            <img 
              src="https://images.unsplash.com/photo-1498050108023-c5249f4df085?w=400&h=300&fit=crop&q=80"
              alt="Coding prompts"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
          </div>

          <div className="grid md:grid-cols-2 gap-6">
            {modules.map((module, index) => (
              <div key={index} className="flex items-start gap-4 bg-gradient-to-r from-emerald-50 to-green-50 rounded-xl p-6 hover:shadow-lg transition-all">
                <div className="w-10 h-10 rounded-full bg-gradient-to-br from-emerald-600 to-purple-600 flex items-center justify-center flex-shrink-0">
                  <CheckCircle2 className="w-6 h-6 text-white" />
                </div>
                <p className="text-gray-800 text-lg font-medium">{module}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* WHO SHOULD JOIN */}
      <section className="py-20 px-6 bg-gradient-to-br from-emerald-700 to-purple-700 text-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold mb-4">Perfect For</h2>
            <p className="text-xl text-emerald-100">This workshop is designed for...</p>
          </div>
          <div className="grid md:grid-cols-4 gap-6">
            {['Students', 'Freelancers', 'Digital Marketers', 'Creators', 
              'Founders', 'Social Media Managers', 'Anyone Using AI', 'Career Builders'].map((role, index) => (
              <div key={index} className="bg-white/10 backdrop-blur-sm rounded-xl p-6 text-center hover:bg-white/20 transition-all border border-yellow-400/30">
                <Star className="w-8 h-8 text-yellow-400 mx-auto mb-3" />
                <p className="font-semibold text-lg">{role}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* HANDS-ON PROJECTS */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Hands-On Projects</h2>
            <p className="text-xl text-gray-600">Build your prompt library live</p>
          </div>

          {/* Project Gallery */}
          <div className="grid md:grid-cols-4 gap-4 mb-12">
            <img 
              src="https://images.unsplash.com/photo-1488590528505-98d2b5aba04b?w=300&h=300&fit=crop&q=80"
              alt="Laptop writing"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1519389950473-47ba0277781c?w=300&h=300&fit=crop&q=80"
              alt="Collaborative work"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1517694712202-14dd9538aa97?w=300&h=300&fit=crop&q=80"
              alt="Computer work"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1498050108023-c5249f4df085?w=300&h=300&fit=crop&q=80"
              alt="Desk setup"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
          </div>

          <div className="grid md:grid-cols-2 gap-8">
            <div className="bg-gradient-to-br from-emerald-600 to-purple-600 rounded-3xl p-10 text-white">
              <Brain className="w-16 h-16 mb-6" />
              <h3 className="text-3xl font-bold mb-6">What You'll Create</h3>
              <ul className="space-y-4">
                {projects.map((project, index) => (
                  <li key={index} className="flex items-start gap-3">
                    <CheckCircle2 className="w-6 h-6 flex-shrink-0 mt-1" />
                    <span className="text-lg">{project}</span>
                  </li>
                ))}
              </ul>
            </div>
            <div className="bg-gray-50 rounded-3xl p-10 border-2 border-emerald-200">
              <Wand2 className="w-16 h-16 text-emerald-600 mb-6" />
              <h3 className="text-3xl font-bold text-gray-900 mb-6">Your Deliverables</h3>
              <ul className="space-y-4">
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-emerald-600 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">1</span>
                  </div>
                  <span className="text-lg text-gray-700">Personal prompt library (15+ prompts)</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-emerald-600 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">2</span>
                  </div>
                  <span className="text-lg text-gray-700">Chain-of-thought templates</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-emerald-600 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">3</span>
                  </div>
                  <span className="text-lg text-gray-700">AI-generated content samples</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-emerald-600 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">4</span>
                  </div>
                  <span className="text-lg text-gray-700">Image generation prompts</span>
                </li>
              </ul>
            </div>
          </div>
        </div>
      </section>

      {/* TOOLS */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-100 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">AI Tools You'll Master</h2>
            <p className="text-xl text-gray-600">Learn to prompt all major AI platforms</p>
          </div>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
            {tools.map((tool, index) => (
              <div key={index} className="bg-white rounded-2xl p-8 shadow-lg hover:shadow-xl transition-all text-center border-2 border-gray-100 hover:border-emerald-300 group">
                <div className="text-5xl mb-4 group-hover:scale-110 transition-transform">{tool.icon}</div>
                <p className="font-bold text-gray-900">{tool.name}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* CAREER OUTCOMES */}
      <section className="py-20 px-6 bg-gradient-to-br from-purple-700 to-emerald-700">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-white mb-4">Career Paths</h2>
            <p className="text-xl text-purple-100">Launch your AI career</p>
          </div>
          <div className="grid md:grid-cols-3 gap-6">
            {['Prompt Engineer', 'AI Content Specialist', 'AI Research Assistant', 
              'Automation Strategist', 'Creative AI Designer', 'AI Productivity Consultant'].map((career, index) => (
              <div key={index} className="bg-white/10 backdrop-blur-md rounded-xl p-6 text-center hover:bg-white/20 transition-all border-2 border-white/30">
                <Target className="w-10 h-10 text-yellow-400 mx-auto mb-3" />
                <p className="font-bold text-white text-lg">{career}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* TRAINERS */}
      <section className="py-20 px-6 bg-white relative overflow-hidden">
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1551434678-e076c223a692?w=1920&h=1080&fit=crop&q=80"
            alt="Training background"
            className="w-full h-full object-cover opacity-5"
          />
        </div>
        <div className="container mx-auto max-w-6xl relative z-10">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Expert Trainers</h2>
            <div className="w-32 h-1 bg-gradient-to-r from-emerald-500 to-purple-600 mx-auto"></div>
          </div>
          <div className="bg-gradient-to-br from-emerald-600 to-purple-700 rounded-3xl p-12 text-white shadow-2xl">
            <div className="grid md:grid-cols-2 gap-12 items-center">
              <div>
                <div className="flex items-center gap-4 mb-6">
                  <div className="w-16 h-16 rounded-full bg-yellow-400 flex items-center justify-center">
                    <Award className="w-10 h-10 text-emerald-900" />
                  </div>
                  <div>
                    <h3 className="text-2xl font-bold">AI Experts</h3>
                    <p className="text-emerald-100">8-12 Years Experience</p>
                  </div>
                </div>
                <p className="text-lg text-emerald-100 mb-6 leading-relaxed">
                  Your workshop is led by <strong className="text-yellow-400">WorkshopLibrary-certified AI trainers</strong> with deep experience in:
                </p>
                <ul className="space-y-3">
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>AI communication systems</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>Content automation</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>Design prompting</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>Business & productivity workflows</span>
                  </li>
                </ul>
              </div>
              <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-8 border-2 border-yellow-400/50">
                <p className="text-lg text-emerald-100 italic mb-6">
                  "Real AI experts who have trained 10,000+ learners in AI foundations and practical prompting techniques."
                </p>
                <div className="flex items-center gap-3">
                  <Shield className="w-8 h-8 text-yellow-400" />
                  <span className="text-yellow-400 font-bold">WorkshopLibrary Certified</span>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Pricing Section */}
      <section id="pricing-section" className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-12">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Workshop Pricing</h2>
            <p className="text-xl text-gray-600">Special limited-time offer for early birds</p>
          </div>
          <div className="bg-gradient-to-br from-slate-50 to-blue-50 rounded-3xl p-10 border-2 border-slate-200 shadow-xl">
            <div className="relative">
              <div className="bg-gradient-to-br from-white via-blue-50/30 to-purple-50/30 rounded-3xl p-10 shadow-2xl border-2 border-blue-200/50 relative overflow-hidden">
                <div className="absolute inset-0 bg-gradient-to-r from-blue-400/10 to-purple-400/10 rounded-3xl blur-xl"></div>
                <div className="relative z-10">
                  <div className="flex justify-center mb-6">
                    <div className="bg-gradient-to-r from-orange-500 to-red-500 text-white px-6 py-3 rounded-full font-bold text-sm shadow-lg inline-flex items-center gap-2">
                      <span>🔥</span><span>Limited-Time Deal — 50% OFF</span>
                    </div>
                  </div>
                  <div className="text-center mb-4"><p className="text-gray-400 text-lg line-through">$100 / 365 AED</p></div>
                  <div className="text-center mb-4">
                    <div className="flex items-center justify-center gap-4 mb-3">
                      <p className="text-6xl md:text-7xl font-black text-blue-700">$50</p>
                      <span className="text-4xl text-gray-600 font-bold">/</span>
                      <p className="text-6xl md:text-7xl font-black text-blue-700">185 AED</p>
                    </div>
                    <p className="text-gray-700 font-semibold text-lg">For the Next 15 Days Only — Early Bird Offer</p>
                  </div>
                  <div className="text-center mb-8"><p className="text-gray-600 text-base">Enroll Now & Unlock Your Instant 50% Discount — Limited Seats</p></div>
                  <div className="text-center">
                    <button onClick={scrollToForm} className="w-full md:w-auto bg-gradient-to-r from-yellow-400 to-amber-500 hover:from-yellow-500 hover:to-amber-600 text-gray-900 px-12 py-5 rounded-full font-bold text-xl shadow-2xl hover:shadow-3xl transition-all hover:scale-105 inline-flex items-center justify-center gap-3">
                      Claim This Deal<ArrowRight className="w-6 h-6" />
                    </button>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <div className="mt-8">
            <h3 className="text-2xl font-bold text-gray-900 text-center mb-6">Additional Group & Corporate Discounts</h3>
            <div className="bg-white rounded-2xl p-8 shadow-lg border border-gray-200">
              <div className="flex flex-col md:flex-row items-start gap-6">
                <div className="flex-shrink-0">
                  <div className="w-16 h-16 bg-slate-100 rounded-xl flex items-center justify-center">
                    <Users className="w-8 h-8 text-slate-700" />
                  </div>
                </div>
                <div className="flex-1">
                  <h4 className="text-xl font-bold text-gray-900 mb-4">Save More When You Learn Together</h4>
                  <ul className="space-y-3 mb-6">
                    <li className="flex items-start gap-3 text-gray-700"><CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" /><span>10% extra discount for group bookings (2 or more participants)</span></li>
                    <li className="flex items-start gap-3 text-gray-700"><CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" /><span>Special pricing available for corporate teams</span></li>
                    <li className="flex items-start gap-3 text-gray-700"><CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" /><span>Dedicated support and invoicing for companies</span></li>
                  </ul>
                  <button onClick={scrollToForm} className="border-2 border-slate-700 text-slate-700 hover:bg-slate-700 hover:text-white px-8 py-3 rounded-xl font-semibold transition-all inline-flex items-center gap-2">
                    Request Corporate Pricing<ArrowRight className="w-5 h-5" />
                  </button>
                </div>
              </div>
              <div className="mt-6 pt-6 border-t border-gray-200">
                <p className="text-sm text-gray-600 mb-2">✓ Discounts apply on top of the current 15-day promotional price.</p>
                <p className="text-sm text-gray-600">✓ For larger teams, customized offers are available.</p>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* CERTIFICATION */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-4xl">
          <div className="bg-gradient-to-br from-emerald-700 to-purple-700 rounded-3xl p-12 text-center text-white border-4 border-yellow-400 shadow-2xl">
            <Award className="w-24 h-24 text-yellow-400 mx-auto mb-6" />
            <h2 className="text-4xl font-bold mb-4">Get Certified</h2>
            <p className="text-2xl text-emerald-100 mb-6">
              🎓 Prompt Engineering for Beginners Certificate
            </p>
            <p className="text-lg text-emerald-200 mb-8">
              Issued by <strong className="text-yellow-400">Workshop Library</strong> - recognized across industries & upskilling networks
            </p>
            <div className="inline-flex items-center gap-3 bg-white/10 backdrop-blur-sm px-8 py-4 rounded-full border-2 border-yellow-400/50">
              <Shield className="w-6 h-6 text-yellow-400" />
              <span className="font-bold">Globally Recognized Certification</span>
            </div>
          </div>
        </div>
      </section>

      {/* FAQ */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Frequently Asked Questions</h2>
          </div>
          <div className="space-y-6">
            {[
              { q: 'Do I need AI knowledge?', a: 'No, this is a beginner-friendly workshop.' },
              { q: 'Will I learn prompts for multiple tools?', a: 'Yes — text, design, image, productivity sab cover hota hai.' },
              { q: 'Are recordings included?', a: 'Yes, 30-day access.' },
              { q: 'Will this help in my job or freelance work?', a: 'Yes — prompt engineering is now a top requested skill worldwide.' },
              { q: 'Can I use this for my business?', a: 'Absolutely! You\'ll learn prompts for business tasks, marketing, and automation.' }
            ].map((faq, index) => (
              <div key={index} className="bg-gradient-to-r from-emerald-50 to-green-50 rounded-xl p-6 hover:shadow-lg transition-all border-2 border-emerald-100 hover:border-emerald-300">
                <h3 className="text-xl font-bold text-gray-900 mb-3 flex items-center gap-3">
                  <div className="w-8 h-8 rounded-full bg-emerald-600 flex items-center justify-center text-white font-bold">
                    Q
                  </div>
                  {faq.q}
                </h3>
                <p className="text-lg text-gray-700 ml-11">{faq.a}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* REGISTRATION FORM */}
      <section id="registration-form" className="py-20 px-6 bg-gradient-to-br from-emerald-700 to-purple-700">
        <div className="container mx-auto max-w-2xl">
          <div className="text-center mb-8">
            <h2 className="text-4xl font-bold text-white mb-4">Master AI Prompting Today!</h2>
            <p className="text-xl text-emerald-100">Limited seats for December 14, 2025</p>
          </div>
          <CourseLeadForm courseName="Prompt Engineering for Beginners Workshop" formTitle="Register for Workshop" />
        </div>
      </section>

      {/* FOOTER */}
      <footer className="py-16 px-6 bg-gradient-to-r from-gray-900 via-gray-800 to-gray-900 text-white relative overflow-hidden">
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1451187580459-43490279c0fa?w=1920&h=1080&fit=crop&q=80"
            alt="Footer background"
            className="w-full h-full object-cover opacity-5"
          />
        </div>
        <div className="container mx-auto max-w-7xl relative z-10">
          <div className="text-center">
            <div className="flex items-center justify-center gap-3 mb-6">
              <div className="w-12 h-12 rounded-xl bg-gradient-to-br from-emerald-600 to-purple-700 border-2 border-yellow-400/50 flex items-center justify-center">
                <Sparkles className="w-7 h-7 text-yellow-400" />
              </div>
              <div>
                <h3 className="font-bold text-xl">AI School for All</h3>
                <p className="text-xs text-gray-400">by Workshop Library</p>
              </div>
            </div>
            <p className="text-gray-300 mb-6">Professional AI and Creative Skills Education</p>
            <div className="flex items-center justify-center gap-3 mb-6">
              <Shield className="w-5 h-5 text-yellow-400" />
              <span className="text-sm text-gray-300 font-semibold">WorkshopLibrary Partner</span>
            </div>
            <p className="text-xs text-gray-500">
              © 2026 AI School for All. All rights reserved.
            </p>
          </div>
        </div>
      </footer>

      {/* Chatbot */}
      <Chatbot />
    </div>
  );
};

export default WorkshopPromptEngineering;
