import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { 
  Calendar, Clock, Video, Users, Globe, Sparkles, TrendingUp, 
  Award, CheckCircle2, Download, ArrowRight, Zap, Target, Star,
  Camera, Image, Palette, Layers, Play, FileText, BookOpen, Shield
} from 'lucide-react';
import CourseLeadForm from '../components/CourseLeadForm';
import Chatbot from '../components/Chatbot';

const WorkshopPhotoEditing = () => {
  const navigate = useNavigate();
  const [timeLeft, setTimeLeft] = useState({ days: 0, hours: 0, minutes: 0, seconds: 0 });

  useEffect(() => {
    // Workshop starts November 30, 2025 at 12:00 AM EST
    const workshopDate = new Date('2025-11-30T05:00:00Z'); // 12:00 AM EST = 05:00 UTC
    const targetDate = workshopDate.getTime();

    const updateCountdown = () => {
      const now = new Date().getTime();
      const distance = targetDate - now;

      if (distance > 0) {
        setTimeLeft({
          days: Math.floor(distance / (1000 * 60 * 60 * 24)),
          hours: Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60)),
          minutes: Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60)),
          seconds: Math.floor((distance % (1000 * 60)) / 1000)
        });
      } else {
        // If countdown is over, show zeros
        setTimeLeft({ days: 0, hours: 0, minutes: 0, seconds: 0 });
      }
    };

    updateCountdown();
    const interval = setInterval(updateCountdown, 1000);
    return () => clearInterval(interval);
  }, []);

  const scrollToForm = () => {
    const formElement = document.getElementById('registration-form');
    if (formElement) {
      formElement.scrollIntoView({ behavior: 'smooth', block: 'start' });
    }
  };

  const scrollToPricing = () => {
    const pricingElement = document.getElementById('pricing-section');
    if (pricingElement) {
      pricingElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
    }
  };

  const modules = [
    'AI-powered enhancement & upscaling',
    'Professional portrait retouching',
    'Product photo cleaning (shadows, dust, edges)',
    'Lighting correction & color grading',
    'Removing unwanted objects & distractions',
    'Background removal & replacement',
    'Creative AI-based image variations',
    'Exporting images for social media, ads & e-commerce'
  ];

  const stats = [
    { value: '64%', label: 'Creators use AI tools for editing' },
    { value: '70%', label: 'Reduction in editing time' },
    { value: '35-40%', label: 'Higher e-commerce conversions' },
    { value: '₹30K-₹1.2L', label: 'Monthly freelance earnings' }
  ];

  const tools = [
    { name: 'Adobe Express AI', icon: '🎨' },
    { name: 'Canva AI Editor', icon: '✨' },
    { name: 'Remove.bg', icon: '🖼️' },
    { name: 'SnapEdit AI', icon: '📸' },
    { name: 'Luminar AI', icon: '🌟' },
    { name: 'Pixelcut', icon: '✂️' },
    { name: 'CleanUp Pictures', icon: '🧹' }
  ];

  const projects = [
    'Edit 5 portrait photos using 3 AI tools',
    'Create 3 product photos suitable for e-commerce',
    'Remove objects + retouch compositions',
    'Create before/after transformation portfolio',
    'Build your first AI-powered preset pack'
  ];

  return (
    <div className="min-h-screen bg-white">
      {/* Navigation - Same as Course Page */}
      <nav className="fixed top-4 left-1/2 transform -translate-x-1/2 z-50 w-[95%] max-w-7xl">
        <div className="bg-black/30 backdrop-blur-lg rounded-full border border-white/20 px-8 py-3 shadow-2xl">
          <div className="flex items-center justify-between">
            <button onClick={() => navigate('/')} className="text-white font-bold text-lg hover:text-yellow-400 transition-colors">
              AI School for All
            </button>
            <div className="hidden md:flex items-center gap-6">
              <button onClick={() => navigate('/#home')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Home</button>
              <button onClick={() => navigate('/#workshops')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Workshops</button>
              <button 
                onClick={scrollToForm}
                className="bg-yellow-400 hover:bg-yellow-500 text-gray-900 px-6 py-2 rounded-full font-bold transition-all text-sm"
              >
                Register Now
              </button>
            </div>
          </div>
        </div>
      </nav>

      {/* EVENT-STYLE HERO BANNER */}
      <section className="relative min-h-screen flex items-center justify-center overflow-hidden pt-20">
        {/* Hero Background Image */}
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1542038784456-1ea8e935640e?w=1920&h=1080&fit=crop&q=80"
            alt="Photo Editing Workshop"
            className="w-full h-full object-cover brightness-110"
          />
          <div className="absolute inset-0 bg-gradient-to-br from-orange-500/70 via-amber-500/65 to-yellow-400/70"></div>
        </div>

        {/* Animated overlay patterns */}
        <div className="absolute inset-0 opacity-10 z-10">
          <div className="absolute top-20 left-20 w-72 h-72 bg-white rounded-full blur-3xl animate-pulse"></div>
          <div className="absolute bottom-20 right-20 w-96 h-96 bg-white rounded-full blur-3xl animate-pulse delay-1000"></div>
        </div>

        <div className="container mx-auto px-6 relative z-20">
          <div className="max-w-5xl mx-auto text-center">
            {/* Event Badge */}
            <div className="inline-flex items-center gap-2 bg-white/20 backdrop-blur-md px-6 py-3 rounded-full mb-8 border border-white/30">
              <Zap className="w-5 h-5 text-white" />
              <span className="text-white font-bold text-sm uppercase tracking-wider">Live Virtual Workshop</span>
            </div>

            {/* Main Title */}
            <h1 className="text-5xl md:text-7xl font-black text-white mb-6 leading-tight drop-shadow-2xl">
              AI-Powered Photo Editing
              <br />
              <span className="text-gray-900">& Retouching</span>
            </h1>

            <p className="text-2xl text-white/90 mb-10 max-w-3xl mx-auto font-medium">
              Master next-generation AI tools to transform ordinary photos into stunning visuals in just 4 hours
            </p>

            {/* Event Quick Info */}
            <div className="grid md:grid-cols-4 gap-4 mb-12 max-w-4xl mx-auto">
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Calendar className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">30 Nov 2025</p>
                <p className="text-white/80 text-sm">Sunday</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Clock className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">4 Hours</p>
                <p className="text-white/80 text-sm">Intensive</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Video className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">Live Online</p>
                <p className="text-white/80 text-sm">Interactive</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Users className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">Limited</p>
                <p className="text-white/80 text-sm">Seats Only</p>
              </div>
            </div>

            {/* CTA */}
            <button 
              onClick={scrollToForm}
              className="bg-gray-900 hover:bg-black text-yellow-400 px-12 py-6 rounded-2xl font-black text-xl shadow-2xl hover:scale-105 transition-all inline-flex items-center gap-3"
            >
              Secure Your Spot
              <ArrowRight className="w-6 h-6" />
            </button>

            <p className="text-white/90 mt-6 text-sm">
              <strong>Enrollment Begins Soon</strong> - Be the first to register!
            </p>
          </div>
        </div>
      </section>

      {/* COUNTDOWN TIMER SECTION */}
      <section className="py-16 px-6 bg-gradient-to-b from-gray-900 to-gray-800">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-8">
            <h3 className="text-3xl font-bold text-yellow-400 mb-2">Workshop Starts In</h3>
            <p className="text-gray-400">November 30, 2025 • 12:00 AM EST</p>
          </div>
          <div className="grid grid-cols-4 gap-4 max-w-2xl mx-auto">
            <div className="bg-gradient-to-br from-gray-700 to-gray-800 rounded-xl p-6 text-center border-2 border-yellow-400/30">
              <div className="text-5xl font-bold text-yellow-400 mb-2">{timeLeft.days}</div>
              <div className="text-gray-400 text-sm uppercase tracking-wide">Days</div>
            </div>
            <div className="bg-gradient-to-br from-gray-700 to-gray-800 rounded-xl p-6 text-center border-2 border-yellow-400/30">
              <div className="text-5xl font-bold text-yellow-400 mb-2">{timeLeft.hours}</div>
              <div className="text-gray-400 text-sm uppercase tracking-wide">Hours</div>
            </div>
            <div className="bg-gradient-to-br from-gray-700 to-gray-800 rounded-xl p-6 text-center border-2 border-yellow-400/30">
              <div className="text-5xl font-bold text-yellow-400 mb-2">{timeLeft.minutes}</div>
              <div className="text-gray-400 text-sm uppercase tracking-wide">Minutes</div>
            </div>
            <div className="bg-gradient-to-br from-gray-700 to-gray-800 rounded-xl p-6 text-center border-2 border-yellow-400/30">
              <div className="text-5xl font-bold text-yellow-400 mb-2">{timeLeft.seconds}</div>
              <div className="text-gray-400 text-sm uppercase tracking-wide">Seconds</div>
            </div>
          </div>
        </div>
      </section>

      {/* Promotional Banner - Limited Time Deal */}
      <section className="py-6 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <button
            onClick={scrollToPricing}
            className="w-full bg-gradient-to-r from-orange-500 to-red-500 hover:from-orange-600 hover:to-red-600 text-white px-8 py-5 rounded-full font-bold text-lg shadow-xl hover:shadow-2xl transition-all hover:scale-105 flex items-center justify-center gap-3"
          >
            <span className="text-2xl">🔥</span>
            <span>Limited-Time Deal — 50% OFF</span>
            <ArrowRight className="w-6 h-6" />
          </button>
        </div>
      </section>

      {/* WORKSHOP OVERVIEW */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="grid md:grid-cols-2 gap-12 items-center">
            <div>
              <div className="inline-flex items-center gap-2 bg-orange-100 px-4 py-2 rounded-full mb-6">
                <Camera className="w-5 h-5 text-orange-600" />
                <span className="text-orange-900 font-bold text-sm">TRANSFORM YOUR SKILLS</span>
              </div>
              <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-6">Why This Workshop?</h2>
              <p className="text-lg text-gray-700 mb-6 leading-relaxed">
                This power-packed 4-hour workshop is designed to help beginners and professionals master the art of photo editing using next-generation AI tools.
              </p>
              <p className="text-lg text-gray-700 mb-6 leading-relaxed">
                The world today demands fast, consistent, and high-quality visual output—whether you're creating content for social media, running an e-commerce store, working with clients, or building your brand.
              </p>
              <p className="text-lg text-gray-700 leading-relaxed">
                By the end of the workshop, you'll walk away with a <strong className="text-orange-600">polished mini-portfolio</strong> of before/after transformations and the confidence to edit photos for brands, clients, agencies, and your own content creation journey.
              </p>
            </div>
            <div className="relative">
              <div className="bg-gradient-to-br from-orange-400 to-amber-500 rounded-3xl p-8 shadow-2xl">
                <img 
                  src="https://images.unsplash.com/photo-1561070791-2526d30994b5?w=600&h=400&fit=crop&q=80"
                  alt="Photo Editing Workshop"
                  className="w-full h-80 object-cover rounded-2xl shadow-xl"
                />
                <div className="absolute -bottom-6 -right-6 bg-white rounded-2xl p-6 shadow-2xl border-4 border-yellow-400">
                  <div className="text-center">
                    <div className="text-4xl font-bold text-orange-600">10x</div>
                    <div className="text-sm text-gray-700 font-medium">Faster Editing</div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* BEFORE/AFTER RETOUCHING SHOWCASE */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">See The AI Magic</h2>
            <p className="text-xl text-gray-600">Real retouching transformations you'll learn to create</p>
          </div>

          {/* Example 1: Portrait Retouching */}
          <div className="mb-16">
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">Professional Portrait Retouching</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BEFORE
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl">
                  <img 
                    src="https://images.unsplash.com/photo-1573496359142-b8d87734a5a2?w=600&h=600&fit=crop&q=50&sat=-60&con=-30&sharp=50"
                    alt="Before retouching"
                    className="w-full h-96 object-cover"
                  />
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Raw photo • Uneven lighting • Skin imperfections • Dull colors</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  AFTER AI EDIT
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl">
                  <img 
                    src="https://images.unsplash.com/photo-1573496359142-b8d87734a5a2?w=600&h=600&fit=crop&q=85&sharp=-20"
                    alt="After retouching"
                    className="w-full h-96 object-cover"
                  />
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">AI enhanced • Perfect lighting • Flawless skin • Professional studio quality</p>
              </div>
            </div>
          </div>

          {/* Example 2: Product Photo Enhancement */}
          <div className="mb-16">
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">E-Commerce Product Enhancement</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BEFORE
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl bg-gray-200">
                  <img 
                    src="https://images.unsplash.com/photo-1491553895911-0055eca6402d?w=600&h=600&fit=crop&q=40&sat=-70&con=-40"
                    alt="Before product edit"
                    className="w-full h-96 object-cover opacity-70"
                  />
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Messy background • Shadows & dust • Dull colors • Unprofessional</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  AFTER AI EDIT
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl bg-white">
                  <img 
                    src="https://images.unsplash.com/photo-1491553895911-0055eca6402d?w=600&h=600&fit=crop&q=90"
                    alt="After product edit"
                    className="w-full h-96 object-cover"
                  />
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Pure white background • No shadows • Vibrant colors • E-commerce ready</p>
              </div>
            </div>
          </div>

          {/* Example 3: Background Removal */}
          <div>
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">AI Background Removal & Replacement</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BEFORE
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl">
                  <img 
                    src="https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=600&h=600&fit=crop&q=70"
                    alt="Before background removal"
                    className="w-full h-96 object-cover"
                  />
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Distracting background • Busy scene • Unprofessional</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  AFTER AI EDIT
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl bg-gradient-to-br from-purple-100 to-pink-100">
                  <img 
                    src="https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=600&h=600&fit=crop&q=80"
                    alt="After background removal"
                    className="w-full h-96 object-cover mix-blend-multiply"
                  />
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Clean removal • Custom background • Pro studio look</p>
              </div>
            </div>
          </div>

          {/* Results Banner */}
          <div className="mt-16 bg-gradient-to-r from-orange-500 to-amber-500 rounded-3xl p-10 text-center text-white">
            <p className="text-3xl font-bold mb-4">You'll Create All These Transformations in Just 4 Hours!</p>
            <p className="text-xl text-white/90">No prior experience needed - AI does the heavy lifting</p>
          </div>
        </div>
      </section>

      {/* INDUSTRY INSIGHTS - STATS */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Industry Insights</h2>
            <p className="text-xl text-gray-600">Why AI photo editing is the future</p>
          </div>
          <div className="grid md:grid-cols-4 gap-8">
            {stats.map((stat, index) => (
              <div key={index} className="bg-white rounded-2xl p-8 shadow-lg hover:shadow-xl transition-all border-2 border-gray-100 hover:border-orange-300">
                <div className="text-5xl font-black text-orange-600 mb-4">{stat.value}</div>
                <p className="text-gray-700 font-medium leading-relaxed">{stat.label}</p>
              </div>
            ))}
          </div>
          <div className="mt-12 text-center">
            <p className="text-lg text-gray-600">
              <span className="text-orange-600 font-bold">AI image-processing market</span> growing <strong>29% YOY globally</strong>
            </p>
          </div>
        </div>
      </section>

      {/* WHAT YOU'LL LEARN - MODULES */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">What You'll Learn</h2>
            <div className="w-32 h-1 bg-gradient-to-r from-orange-400 to-amber-500 mx-auto"></div>
          </div>
          
          {/* Image Gallery */}
          <div className="grid md:grid-cols-3 gap-4 mb-12">
            <img 
              src="https://images.unsplash.com/photo-1618005198919-d3d4b5a92ead?w=400&h=300&fit=crop&q=80"
              alt="Portrait retouching"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
            <img 
              src="https://images.unsplash.com/photo-1516321497487-e288fb19713f?w=400&h=300&fit=crop&q=80"
              alt="Product photography"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
            <img 
              src="https://images.unsplash.com/photo-1626785774573-4b799315345d?w=400&h=300&fit=crop&q=80"
              alt="AI editing tools"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
          </div>

          <div className="grid md:grid-cols-2 gap-6">
            {modules.map((module, index) => (
              <div key={index} className="flex items-start gap-4 bg-gradient-to-r from-orange-50 to-amber-50 rounded-xl p-6 hover:shadow-lg transition-all">
                <div className="w-10 h-10 rounded-full bg-gradient-to-br from-orange-500 to-amber-500 flex items-center justify-center flex-shrink-0">
                  <CheckCircle2 className="w-6 h-6 text-white" />
                </div>
                <p className="text-gray-800 text-lg font-medium">{module}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* WHO SHOULD JOIN */}
      <section className="py-20 px-6 bg-gradient-to-br from-gray-700 to-gray-600 text-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold mb-4">Who Should Join?</h2>
            <p className="text-xl text-gray-200">This workshop is perfect for...</p>
          </div>
          <div className="grid md:grid-cols-4 gap-6">
            {['Creators & Influencers', 'Students & Beginners', 'Photographers', 'Digital Marketers', 
              'E-commerce Sellers', 'Freelancers', 'Small Business Owners', 'Content Creators'].map((role, index) => (
              <div key={index} className="bg-white/10 backdrop-blur-sm rounded-xl p-6 text-center hover:bg-white/20 transition-all border border-yellow-400/30">
                <Star className="w-8 h-8 text-yellow-400 mx-auto mb-3" />
                <p className="font-semibold text-lg">{role}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* HANDS-ON PROJECTS */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Hands-On Projects</h2>
            <p className="text-xl text-gray-600">Build your portfolio during the workshop</p>
          </div>

          {/* Project Images Gallery */}
          <div className="grid md:grid-cols-4 gap-4 mb-12">
            <img 
              src="https://images.unsplash.com/photo-1531746020798-e6953c6e8e04?w=300&h=300&fit=crop&q=80"
              alt="Portrait project"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1523275335684-37898b6baf30?w=300&h=300&fit=crop&q=80"
              alt="Product project"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1522202176988-66273c2fd55f?w=300&h=300&fit=crop&q=80"
              alt="Creative project"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1516321497487-e288fb19713f?w=300&h=300&fit=crop&q=80"
              alt="E-commerce project"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
          </div>

          <div className="grid md:grid-cols-2 gap-8">
            <div className="bg-gradient-to-br from-orange-500 to-amber-500 rounded-3xl p-10 text-white">
              <Layers className="w-16 h-16 mb-6" />
              <h3 className="text-3xl font-bold mb-6">What You'll Create</h3>
              <ul className="space-y-4">
                {projects.map((project, index) => (
                  <li key={index} className="flex items-start gap-3">
                    <CheckCircle2 className="w-6 h-6 flex-shrink-0 mt-1" />
                    <span className="text-lg">{project}</span>
                  </li>
                ))}
              </ul>
            </div>
            <div className="bg-gray-50 rounded-3xl p-10 border-2 border-gray-200">
              <Image className="w-16 h-16 text-orange-600 mb-6" />
              <h3 className="text-3xl font-bold text-gray-900 mb-6">Your Deliverables</h3>
              <ul className="space-y-4">
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-orange-500 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">1</span>
                  </div>
                  <span className="text-lg text-gray-700">5+ Edited professional portraits</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-orange-500 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">2</span>
                  </div>
                  <span className="text-lg text-gray-700">E-commerce ready product photos</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-orange-500 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">3</span>
                  </div>
                  <span className="text-lg text-gray-700">Before/after portfolio showcase</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-orange-500 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">4</span>
                  </div>
                  <span className="text-lg text-gray-700">Custom AI preset pack</span>
                </li>
              </ul>
            </div>
          </div>
        </div>
      </section>

      {/* TOOLS SHOWCASE */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-100 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Tools You'll Master</h2>
            <p className="text-xl text-gray-600">Industry-leading AI editing platforms</p>
          </div>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
            {tools.map((tool, index) => (
              <div key={index} className="bg-white rounded-2xl p-8 shadow-lg hover:shadow-xl transition-all text-center border-2 border-gray-100 hover:border-orange-300 group">
                <div className="text-5xl mb-4 group-hover:scale-110 transition-transform">{tool.icon}</div>
                <p className="font-bold text-gray-900">{tool.name}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* CAREER OUTCOMES */}
      <section className="py-20 px-6 bg-gradient-to-br from-amber-500 to-orange-500">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-white mb-4">Career Outcomes</h2>
            <p className="text-xl text-white/90">What you can become after this workshop</p>
          </div>
          <div className="grid md:grid-cols-4 gap-6">
            {['AI Photo Editor', 'Creative Visual Designer', 'Retouching Specialist', 'E-commerce Image Editor',
              'Social Media Creative', 'Digital Content Designer', 'Photo Enhancement Freelancer', 'Visual Strategist'].map((career, index) => (
              <div key={index} className="bg-white/20 backdrop-blur-md rounded-xl p-6 text-center hover:bg-white/30 transition-all border-2 border-white/30">
                <Target className="w-10 h-10 text-white mx-auto mb-3" />
                <p className="font-bold text-white text-lg">{career}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* TRAINER SECTION */}
      <section className="py-20 px-6 bg-white relative overflow-hidden">
        {/* Background image */}
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1531482615713-2afd69097998?w=1920&h=1080&fit=crop&q=80"
            alt="Training background"
            className="w-full h-full object-cover opacity-5"
          />
        </div>
        <div className="container mx-auto max-w-6xl relative z-10">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">About the Trainers</h2>
            <div className="w-32 h-1 bg-gradient-to-r from-orange-400 to-amber-500 mx-auto"></div>
          </div>
          <div className="bg-gradient-to-br from-gray-600 to-gray-500 rounded-3xl p-12 text-white shadow-2xl">
            <div className="grid md:grid-cols-2 gap-12 items-center">
              <div>
                <div className="flex items-center gap-4 mb-6">
                  <div className="w-16 h-16 rounded-full bg-yellow-400 flex items-center justify-center">
                    <Award className="w-10 h-10 text-gray-900" />
                  </div>
                  <div>
                    <h3 className="text-2xl font-bold">Industry Professionals</h3>
                    <p className="text-gray-300">8-12 Years Experience</p>
                  </div>
                </div>
                <p className="text-lg text-gray-200 mb-6 leading-relaxed">
                  This workshop is delivered by <strong className="text-yellow-400">WorkshopLibrary's certified visual design & AI faculty</strong>, each with extensive experience in:
                </p>
                <ul className="space-y-3">
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>Photo editing & retouching</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>Creative design</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>Social media visual strategy</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>E-commerce content production</span>
                  </li>
                </ul>
              </div>
              <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-8 border-2 border-yellow-400/50">
                <p className="text-lg text-gray-200 italic mb-6">
                  "Trainers are real industry professionals who have worked with brands, agencies, and global content teams."
                </p>
                <div className="flex items-center gap-3">
                  <Shield className="w-8 h-8 text-yellow-400" />
                  <span className="text-yellow-400 font-bold">WorkshopLibrary Certified</span>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Pricing Section */}
      <section id="pricing-section" className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-12">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Workshop Pricing</h2>
            <p className="text-xl text-gray-600">Special limited-time offer for early birds</p>
          </div>

          <div className="bg-gradient-to-br from-slate-50 to-blue-50 rounded-3xl p-10 border-2 border-slate-200 shadow-xl">
            {/* Premium DEAL BADGE */}
            <div className="relative">
              <div className="bg-gradient-to-br from-white via-blue-50/30 to-purple-50/30 rounded-3xl p-10 shadow-2xl border-2 border-blue-200/50 relative overflow-hidden">
                <div className="absolute inset-0 bg-gradient-to-r from-blue-400/10 to-purple-400/10 rounded-3xl blur-xl"></div>
                <div className="relative z-10">
                  <div className="flex justify-center mb-6">
                    <div className="bg-gradient-to-r from-orange-500 to-red-500 text-white px-6 py-3 rounded-full font-bold text-sm shadow-lg inline-flex items-center gap-2">
                      <span>🔥</span>
                      <span>Limited-Time Deal — 50% OFF</span>
                    </div>
                  </div>
                  <div className="text-center mb-4">
                    <p className="text-gray-400 text-lg line-through">$100 / 365 AED</p>
                  </div>
                  <div className="text-center mb-4">
                    <div className="flex items-center justify-center gap-4 mb-3">
                      <p className="text-6xl md:text-7xl font-black text-blue-700">$50</p>
                      <span className="text-4xl text-gray-600 font-bold">/</span>
                      <p className="text-6xl md:text-7xl font-black text-blue-700">185 AED</p>
                    </div>
                    <p className="text-gray-700 font-semibold text-lg">For the Next 15 Days Only — Early Bird Offer</p>
                  </div>
                  <div className="text-center mb-8">
                    <p className="text-gray-600 text-base">Enroll Now & Unlock Your Instant 50% Discount — Limited Seats</p>
                  </div>
                  <div className="text-center">
                    <button onClick={scrollToForm} className="w-full md:w-auto bg-gradient-to-r from-yellow-400 to-amber-500 hover:from-yellow-500 hover:to-amber-600 text-gray-900 px-12 py-5 rounded-full font-bold text-xl shadow-2xl hover:shadow-3xl transition-all hover:scale-105 inline-flex items-center justify-center gap-3">
                      Claim This Deal
                      <ArrowRight className="w-6 h-6" />
                    </button>
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* Additional Group & Corporate Discounts */}
          <div className="mt-8">
            <h3 className="text-2xl font-bold text-gray-900 text-center mb-6">Additional Group & Corporate Discounts</h3>
            <div className="bg-white rounded-2xl p-8 shadow-lg border border-gray-200">
              <div className="flex flex-col md:flex-row items-start gap-6">
                <div className="flex-shrink-0">
                  <div className="w-16 h-16 bg-slate-100 rounded-xl flex items-center justify-center">
                    <Users className="w-8 h-8 text-slate-700" />
                  </div>
                </div>
                <div className="flex-1">
                  <h4 className="text-xl font-bold text-gray-900 mb-4">Save More When You Learn Together</h4>
                  <ul className="space-y-3 mb-6">
                    <li className="flex items-start gap-3 text-gray-700">
                      <CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" />
                      <span>10% extra discount for group bookings (2 or more participants)</span>
                    </li>
                    <li className="flex items-start gap-3 text-gray-700">
                      <CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" />
                      <span>Special pricing available for corporate teams</span>
                    </li>
                    <li className="flex items-start gap-3 text-gray-700">
                      <CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" />
                      <span>Dedicated support and invoicing for companies</span>
                    </li>
                  </ul>
                  <button onClick={scrollToForm} className="border-2 border-slate-700 text-slate-700 hover:bg-slate-700 hover:text-white px-8 py-3 rounded-xl font-semibold transition-all inline-flex items-center gap-2">
                    Request Corporate Pricing
                    <ArrowRight className="w-5 h-5" />
                  </button>
                </div>
              </div>
              <div className="mt-6 pt-6 border-t border-gray-200">
                <p className="text-sm text-gray-600 mb-2">✓ Discounts apply on top of the current 15-day promotional price.</p>
                <p className="text-sm text-gray-600">✓ For larger teams, customized offers are available.</p>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* CERTIFICATION */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-4xl">
          <div className="bg-gradient-to-br from-gray-700 to-gray-600 rounded-3xl p-12 text-center text-white border-4 border-yellow-400 shadow-2xl">
            <Award className="w-24 h-24 text-yellow-400 mx-auto mb-6" />
            <h2 className="text-4xl font-bold mb-4">Get Certified</h2>
            <p className="text-2xl text-gray-200 mb-6">
              🎓 AI Photo Editing & Retouching Workshop Certificate
            </p>
            <p className="text-lg text-gray-300 mb-8">
              Issued by <strong className="text-yellow-400">Workshop Library</strong> - globally recognized training partner
            </p>
            <div className="inline-flex items-center gap-3 bg-white/10 backdrop-blur-sm px-8 py-4 rounded-full border-2 border-yellow-400/50">
              <Shield className="w-6 h-6 text-yellow-400" />
              <span className="font-bold">Globally Recognized Certification</span>
            </div>
          </div>
        </div>
      </section>

      {/* FAQ */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Frequently Asked Questions</h2>
          </div>
          <div className="space-y-6">
            {[
              { q: 'Do I need prior editing skills?', a: 'No, the workshop is beginner-friendly.' },
              { q: 'Do I need paid software?', a: 'No, we use free or trial-based AI tools.' },
              { q: 'Will I get class recordings?', a: 'Yes, 30-day access is provided.' },
              { q: 'Can I use these skills professionally?', a: 'Absolutely — many participants start earning from editing immediately.' },
              { q: 'Is the certificate valid?', a: 'Yes, recognized under Workshop Library\'s global training ecosystem.' }
            ].map((faq, index) => (
              <div key={index} className="bg-gray-50 rounded-xl p-6 hover:shadow-lg transition-all border-2 border-gray-100 hover:border-orange-300">
                <h3 className="text-xl font-bold text-gray-900 mb-3 flex items-center gap-3">
                  <div className="w-8 h-8 rounded-full bg-orange-500 flex items-center justify-center text-white font-bold">
                    Q
                  </div>
                  {faq.q}
                </h3>
                <p className="text-lg text-gray-700 ml-11">{faq.a}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* REGISTRATION FORM */}
      <section id="registration-form" className="py-20 px-6 bg-gradient-to-br from-gray-600 to-gray-500">
        <div className="container mx-auto max-w-2xl">
          <div className="text-center mb-8">
            <h2 className="text-4xl font-bold text-white mb-4">Secure Your Spot Today!</h2>
            <p className="text-xl text-gray-200">Limited seats available for November 30, 2025</p>
          </div>
          <CourseLeadForm courseName="Photo Editing with AI Workshop" formTitle="Register for Workshop" />
        </div>
      </section>

      {/* FOOTER */}
      <footer className="py-16 px-6 bg-gradient-to-r from-gray-900 via-gray-800 to-gray-900 text-white relative overflow-hidden">
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1451187580459-43490279c0fa?w=1920&h=1080&fit=crop&q=80"
            alt="Footer background"
            className="w-full h-full object-cover opacity-5"
          />
        </div>
        <div className="container mx-auto max-w-7xl relative z-10">
          <div className="text-center">
            <div className="flex items-center justify-center gap-3 mb-6">
              <div className="w-12 h-12 rounded-xl bg-gradient-to-br from-gray-700 to-gray-600 border-2 border-yellow-400/50 flex items-center justify-center">
                <Sparkles className="w-7 h-7 text-yellow-400" />
              </div>
              <div>
                <h3 className="font-bold text-xl">AI School for All</h3>
                <p className="text-xs text-gray-400">by Workshop Library</p>
              </div>
            </div>
            <p className="text-gray-300 mb-6">Professional AI and Creative Skills Education</p>
            <div className="flex items-center justify-center gap-3 mb-6">
              <Shield className="w-5 h-5 text-yellow-400" />
              <span className="text-sm text-gray-300 font-semibold">WorkshopLibrary Partner</span>
            </div>
            <p className="text-xs text-gray-500">
              © 2025 AI School for All. All rights reserved.
            </p>
          </div>
        </div>
      </footer>

      {/* Chatbot */}
      <Chatbot />
    </div>
  );
};

export default WorkshopPhotoEditing;
