import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { 
  Calendar, Clock, Video, Users, Globe, Sparkles, TrendingUp, 
  Award, CheckCircle2, Download, ArrowRight, Zap, Target, Star,
  Linkedin, Briefcase, UserCheck, MessageSquare, Eye, TrendingUp as Growth,
  FileText, BookOpen, Shield, Network, Building2, Mail, BarChart3
} from 'lucide-react';
import CourseLeadForm from '../components/CourseLeadForm';
import Chatbot from '../components/Chatbot';

const WorkshopLinkedIn = () => {
  const navigate = useNavigate();
  const [timeLeft, setTimeLeft] = useState({ days: 0, hours: 0, minutes: 0, seconds: 0 });

  useEffect(() => {
    // Workshop starts December 13, 2025 at 12:00 AM EST
    const workshopDate = new Date('2025-12-13T05:00:00Z'); // 12:00 AM EST = 05:00 UTC
    const targetDate = workshopDate.getTime();

    const updateCountdown = () => {
      const now = new Date().getTime();
      const distance = targetDate - now;

      if (distance > 0) {
        setTimeLeft({
          days: Math.floor(distance / (1000 * 60 * 60 * 24)),
          hours: Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60)),
          minutes: Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60)),
          seconds: Math.floor((distance % (1000 * 60)) / 1000)
        });
      } else {
        setTimeLeft({ days: 0, hours: 0, minutes: 0, seconds: 0 });
      }
    };

    updateCountdown();
    const interval = setInterval(updateCountdown, 1000);
    return () => clearInterval(interval);
  }, []);

  const scrollToForm = () => {
    const formElement = document.getElementById('registration-form');
    if (formElement) {
      formElement.scrollIntoView({ behavior: 'smooth', block: 'start' });
    }
  };

  const scrollToPricing = () => {
    const pricingElement = document.getElementById('pricing-section');
    if (pricingElement) {
      pricingElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
    }
  };

  const modules = [
    'Complete LinkedIn profile makeover',
    'AI-based headline & About section writing',
    'Build strong personal positioning',
    'Generate post ideas & write content using AI',
    'Automate outreach & engagement ethically',
    'Build a 30-day LinkedIn content schedule',
    'Identify your niche & target audience',
    'Grow followers & engagement organically'
  ];

  const stats = [
    { value: '1B+', label: 'LinkedIn users worldwide' },
    { value: '87%', label: 'Recruiters use LinkedIn daily' },
    { value: '5x', label: 'Faster growth with strong content' },
    { value: '35%', label: 'Higher visibility with AI profiles' }
  ];

  const tools = [
    { name: 'Taplio', icon: '📊' },
    { name: 'ChatGPT', icon: '🤖' },
    { name: 'TexAu', icon: '⚡' },
    { name: 'Canva', icon: '🎨' },
    { name: 'Notion AI', icon: '📝' },
    { name: 'Buffer', icon: '📅' },
    { name: 'LinkedIn Analytics', icon: '📈' }
  ];

  const projects = [
    'Rewrite your complete LinkedIn profile',
    'Create 10 AI-generated LinkedIn posts',
    'Design your LinkedIn banner',
    'Create engagement & posting plan',
    'Build personalized messaging templates'
  ];

  return (
    <div className="min-h-screen bg-white">
      {/* Navigation */}
      <nav className="fixed top-4 left-1/2 transform -translate-x-1/2 z-50 w-[95%] max-w-7xl">
        <div className="bg-black/30 backdrop-blur-lg rounded-full border border-white/20 px-8 py-3 shadow-2xl">
          <div className="flex items-center justify-between">
            <button onClick={() => navigate('/')} className="text-white font-bold text-lg hover:text-yellow-400 transition-colors">
              AI School for All
            </button>
            <div className="hidden md:flex items-center gap-6">
              <button onClick={() => navigate('/#home')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Home</button>
              <button onClick={() => navigate('/#workshops')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Workshops</button>
              <button 
                onClick={scrollToForm}
                className="bg-yellow-400 hover:bg-yellow-500 text-gray-900 px-6 py-2 rounded-full font-bold transition-all text-sm"
              >
                Register Now
              </button>
            </div>
          </div>
        </div>
      </nav>

      {/* PROFESSIONAL LINKEDIN THEMED HERO */}
      <section className="relative min-h-screen flex items-center justify-center overflow-hidden pt-20">
        {/* Professional Background */}
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1557804506-669a67965ba0?w=1920&h=1080&fit=crop&q=80"
            alt="LinkedIn Professional Networking"
            className="w-full h-full object-cover brightness-110"
          />
          <div className="absolute inset-0 bg-gradient-to-br from-blue-900/85 via-blue-800/80 to-gray-900/85"></div>
        </div>

        {/* Animated overlay patterns */}
        <div className="absolute inset-0 opacity-10 z-10">
          <div className="absolute top-20 left-20 w-72 h-72 bg-yellow-400 rounded-full blur-3xl animate-pulse"></div>
          <div className="absolute bottom-20 right-20 w-96 h-96 bg-yellow-400 rounded-full blur-3xl animate-pulse delay-1000"></div>
          <div className="absolute top-1/2 left-1/2 w-64 h-64 bg-blue-400 rounded-full blur-3xl animate-pulse delay-500"></div>
        </div>

        <div className="container mx-auto px-6 relative z-20">
          <div className="max-w-5xl mx-auto text-center">
            {/* Event Badge */}
            <div className="inline-flex items-center gap-2 bg-white/20 backdrop-blur-md px-6 py-3 rounded-full mb-8 border border-white/30">
              <Linkedin className="w-5 h-5 text-white" />
              <span className="text-white font-bold text-sm uppercase tracking-wider">4-Hour Career Accelerator</span>
            </div>

            {/* Main Title */}
            <h1 className="text-5xl md:text-7xl font-black text-white mb-6 leading-tight drop-shadow-2xl">
              LinkedIn Automation &
              <br />
              <span className="text-yellow-400">Personal Branding with AI</span>
            </h1>

            <p className="text-2xl text-white/90 mb-10 max-w-3xl mx-auto font-medium">
              Transform your profile into a job-offer & client magnet using AI-powered systems
            </p>

            {/* Event Quick Info */}
            <div className="grid md:grid-cols-4 gap-4 mb-12 max-w-4xl mx-auto">
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Calendar className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">13 Dec 2025</p>
                <p className="text-white/80 text-sm">Sunday</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Clock className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">4 Hours</p>
                <p className="text-white/80 text-sm">Intensive</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Video className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">Live Online</p>
                <p className="text-white/80 text-sm">Interactive</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Zap className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">AI-Powered</p>
                <p className="text-white/80 text-sm">Automated</p>
              </div>
            </div>

            {/* CTA */}
            <button 
              onClick={scrollToForm}
              className="bg-yellow-400 hover:bg-yellow-500 text-gray-900 px-12 py-6 rounded-2xl font-black text-xl shadow-2xl hover:scale-105 transition-all inline-flex items-center gap-3"
            >
              <Briefcase className="w-6 h-6" />
              Build My LinkedIn Brand
              <ArrowRight className="w-6 h-6" />
            </button>

            <p className="text-white/90 mt-6 text-sm">
              🔥 <strong>HOT SELLING WORKSHOP</strong> - Enrollment Opens Soon!
            </p>
          </div>
        </div>
      </section>

      {/* COUNTDOWN TIMER */}
      <section className="py-16 px-6 bg-gradient-to-b from-blue-900 to-gray-900">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-8">
            <h3 className="text-3xl font-bold text-yellow-400 mb-2">Workshop Starts In</h3>
            <p className="text-gray-200">December 13, 2025 • 12:00 AM EST</p>
          </div>
          <div className="grid grid-cols-4 gap-4 max-w-2xl mx-auto">
            {['days', 'hours', 'minutes', 'seconds'].map((unit) => (
              <div key={unit} className="bg-gradient-to-br from-blue-700 to-gray-800 rounded-xl p-6 text-center border-2 border-yellow-400/30">
                <div className="text-5xl font-bold text-yellow-400 mb-2">{timeLeft[unit]}</div>
                <div className="text-gray-200 text-sm uppercase tracking-wide">{unit}</div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Promotional Banner - Limited Time Deal */}
      <section className="py-6 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <button onClick={scrollToPricing} className="w-full bg-gradient-to-r from-orange-500 to-red-500 hover:from-orange-600 hover:to-red-600 text-white px-8 py-5 rounded-full font-bold text-lg shadow-xl hover:shadow-2xl transition-all hover:scale-105 flex items-center justify-center gap-3">
            <span className="text-2xl">🔥</span><span>Limited-Time Deal — 50% OFF</span><ArrowRight className="w-6 h-6" />
          </button>
        </div>
      </section>

      {/* WHY THIS WORKSHOP */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="grid md:grid-cols-2 gap-12 items-center">
            <div>
              <div className="inline-flex items-center gap-2 bg-blue-100 px-4 py-2 rounded-full mb-6">
                <Linkedin className="w-5 h-5 text-blue-700" />
                <span className="text-blue-900 font-bold text-sm">YOUR DIGITAL IDENTITY</span>
              </div>
              <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-6">Why LinkedIn is Your Career Superpower</h2>
              <p className="text-lg text-gray-700 mb-6 leading-relaxed">
                Your LinkedIn profile is your <strong className="text-blue-700">digital resume, portfolio, and networking hub</strong> combined. It's where recruiters discover talent, clients find consultants, and opportunities knock on your door.
              </p>
              <p className="text-lg text-gray-700 mb-6 leading-relaxed">
                In this workshop, you'll learn how to <strong className="text-blue-700">transform your profile using AI frameworks</strong> that attract the right people, automate engagement ethically, and build a content system that works 24/7.
              </p>
              <p className="text-lg text-gray-700 leading-relaxed">
                Walk away with a <strong className="text-yellow-600">polished, recruiter-friendly profile</strong>, a 30-day content plan, automated tools, and the system to attract job offers, clients, and collaborations.
              </p>
            </div>
            <div className="relative">
              <div className="bg-gradient-to-br from-blue-700 to-gray-800 rounded-3xl p-8 shadow-2xl">
                <img 
                  src="https://images.unsplash.com/photo-1553877522-43269d4ea984?w=600&h=400&fit=crop&q=80"
                  alt="LinkedIn Professional Success"
                  className="w-full h-80 object-cover rounded-2xl shadow-xl"
                />
                <div className="absolute -bottom-6 -right-6 bg-white rounded-2xl p-6 shadow-2xl border-4 border-yellow-400">
                  <div className="text-center">
                    <div className="text-4xl font-bold text-blue-700">87%</div>
                    <div className="text-sm text-gray-700 font-medium">Recruiters Use Daily</div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* LINKEDIN PROFILE TRANSFORMATION */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">See The AI Profile Magic</h2>
            <p className="text-xl text-gray-600">Real transformations that attract opportunities</p>
          </div>

          {/* Example 1: Profile Headline Transformation */}
          <div className="mb-16">
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">LinkedIn Headline Glow-Up</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BEFORE
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl bg-white">
                  <div className="p-8">
                    <div className="flex items-start gap-4 mb-6">
                      <div className="w-20 h-20 rounded-full bg-gray-300 flex items-center justify-center text-gray-500 text-2xl font-bold">
                        JD
                      </div>
                      <div className="flex-1">
                        <h4 className="text-xl font-bold text-gray-900 mb-1">John Doe</h4>
                        <p className="text-gray-500 text-sm">Marketing Professional</p>
                        <p className="text-gray-400 text-xs mt-1">Mumbai, India</p>
                      </div>
                    </div>
                    <div className="space-y-3">
                      <div className="flex items-center gap-2 text-gray-400">
                        <Building2 className="w-4 h-4" />
                        <span className="text-sm">ABC Company</span>
                      </div>
                      <div className="flex items-center gap-2 text-gray-400">
                        <Users className="w-4 h-4" />
                        <span className="text-sm">234 connections</span>
                      </div>
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Generic title • No positioning • Low visibility</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  AFTER AI TRANSFORMATION
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl bg-gradient-to-br from-blue-50 to-gray-50">
                  <div className="p-8">
                    <div className="flex items-start gap-4 mb-6">
                      <div className="w-20 h-20 rounded-full bg-gradient-to-br from-blue-600 to-blue-800 flex items-center justify-center text-white text-2xl font-bold border-4 border-yellow-400">
                        JD
                      </div>
                      <div className="flex-1">
                        <h4 className="text-xl font-bold text-gray-900 mb-1">John Doe</h4>
                        <p className="text-blue-700 text-sm font-semibold">AI-Powered Marketing Strategist | Helping B2B SaaS Scale 10x | Growth Hacker | 50K+ Leads Generated</p>
                        <p className="text-gray-600 text-xs mt-1">Mumbai, India • Open to Opportunities</p>
                      </div>
                    </div>
                    <div className="space-y-3">
                      <div className="flex items-center gap-2 text-gray-700">
                        <Building2 className="w-4 h-4 text-blue-600" />
                        <span className="text-sm font-medium">Growth Marketing Lead @ TechCorp</span>
                      </div>
                      <div className="flex items-center gap-2 text-gray-700">
                        <Users className="w-4 h-4 text-blue-600" />
                        <span className="text-sm font-medium">2,847 connections</span>
                      </div>
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">AI-optimized headline • Clear value prop • 12x more profile views!</p>
              </div>
            </div>
          </div>

          {/* Example 2: Post Engagement Transformation */}
          <div className="mb-16">
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">LinkedIn Post Engagement Explosion</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BEFORE
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl bg-white">
                  <div className="p-8">
                    <div className="flex items-start gap-3 mb-4">
                      <div className="w-12 h-12 rounded-full bg-gray-300"></div>
                      <div>
                        <p className="font-semibold text-gray-900">John Doe</p>
                        <p className="text-xs text-gray-500">2h ago</p>
                      </div>
                    </div>
                    <p className="text-gray-700 mb-6">
                      Had a great meeting today. Excited about the project. #work #business
                    </p>
                    <div className="flex items-center gap-6 text-gray-400 text-sm border-t pt-4">
                      <div className="flex items-center gap-1">
                        <span>👍</span>
                        <span>8 likes</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <MessageSquare className="w-4 h-4" />
                        <span>1 comment</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <Eye className="w-4 h-4" />
                        <span>47 views</span>
                      </div>
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Vague post • No hook • Low engagement</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  AFTER AI OPTIMIZATION
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl bg-white">
                  <div className="p-8">
                    <div className="flex items-start gap-3 mb-4">
                      <div className="w-12 h-12 rounded-full bg-blue-600"></div>
                      <div>
                        <p className="font-semibold text-gray-900">John Doe</p>
                        <p className="text-xs text-gray-500">2h ago</p>
                      </div>
                    </div>
                    <p className="text-gray-900 font-medium mb-2">
                      5 AI tools that saved me 20 hours this week (and why you need them):
                    </p>
                    <p className="text-gray-700 mb-4">
                      After testing 50+ AI tools, these 5 changed everything for my marketing workflow.
                      
                      Here's what each one does... 🧵
                    </p>
                    <div className="flex items-center gap-6 text-blue-600 text-sm border-t pt-4 font-semibold">
                      <div className="flex items-center gap-1">
                        <span>👍</span>
                        <span>2,847 likes</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <MessageSquare className="w-4 h-4" />
                        <span>183 comments</span>
                      </div>
                      <div className="flex items-center gap-1">
                        <Eye className="w-4 h-4" />
                        <span>18,932 views</span>
                      </div>
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">AI-crafted hook • Thread format • 400x more engagement!</p>
              </div>
            </div>
          </div>

          {/* Example 3: Connection Growth */}
          <div>
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">Automated Connection Growth</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BEFORE
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl">
                  <div className="bg-white p-8 h-96 flex items-center justify-center">
                    <div className="text-center text-gray-500">
                      <Network className="w-20 h-20 mx-auto mb-4 text-gray-300" />
                      <p className="text-3xl font-bold mb-2">234</p>
                      <p className="text-lg mb-4">connections</p>
                      <p className="text-sm text-gray-400">Manual outreach • Slow growth • Random audience</p>
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">No strategy • Limited network • Missing opportunities</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  AFTER AI AUTOMATION
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl bg-gradient-to-br from-blue-50 to-gray-50">
                  <div className="p-8 h-96">
                    <div className="text-center mb-6">
                      <div className="inline-flex items-center justify-center w-24 h-24 rounded-full bg-gradient-to-br from-blue-600 to-blue-800 text-white mb-4">
                        <Network className="w-12 h-12" />
                      </div>
                      <p className="text-5xl font-bold text-blue-700 mb-2">2,847</p>
                      <p className="text-xl text-gray-700 font-semibold mb-6">targeted connections</p>
                    </div>
                    <div className="space-y-3">
                      <div className="bg-white rounded-xl p-4 shadow-md">
                        <div className="flex items-center justify-between">
                          <span className="text-sm font-medium text-gray-700">Monthly Growth</span>
                          <span className="text-lg font-bold text-green-600">+450</span>
                        </div>
                      </div>
                      <div className="bg-white rounded-xl p-4 shadow-md">
                        <div className="flex items-center justify-between">
                          <span className="text-sm font-medium text-gray-700">Job Offers Received</span>
                          <span className="text-lg font-bold text-blue-600">12</span>
                        </div>
                      </div>
                      <div className="bg-white rounded-xl p-4 shadow-md">
                        <div className="flex items-center justify-between">
                          <span className="text-sm font-medium text-gray-700">Client Inquiries</span>
                          <span className="text-lg font-bold text-yellow-600">23</span>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">AI-targeted outreach • 12x growth in 30 days • Real opportunities!</p>
              </div>
            </div>
          </div>

          {/* Results Banner */}
          <div className="mt-16 bg-gradient-to-r from-blue-700 to-gray-900 rounded-3xl p-10 text-center text-white">
            <p className="text-3xl font-bold mb-4">Transform Your LinkedIn Presence in Just 4 Hours!</p>
            <p className="text-xl text-white/90">No experience needed - AI handles the heavy lifting</p>
          </div>
        </div>
      </section>

      {/* INDUSTRY INSIGHTS */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Industry Insights</h2>
            <p className="text-xl text-gray-600">Why LinkedIn mastery is essential for your career</p>
          </div>
          <div className="grid md:grid-cols-4 gap-8">
            {stats.map((stat, index) => (
              <div key={index} className="bg-gradient-to-br from-blue-50 to-gray-50 rounded-2xl p-8 shadow-lg hover:shadow-xl transition-all border-2 border-blue-100">
                <div className="text-5xl font-black text-blue-700 mb-4">{stat.value}</div>
                <p className="text-gray-700 font-medium leading-relaxed">{stat.label}</p>
              </div>
            ))}
          </div>
          <div className="mt-12 text-center">
            <p className="text-lg text-gray-600">
              <span className="text-blue-700 font-bold">Optimized LinkedIn leads</span> are <strong>3x more likely to convert</strong> into opportunities
            </p>
          </div>
        </div>
      </section>

      {/* WHAT YOU'LL LEARN */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">What You'll Master</h2>
            <div className="w-32 h-1 bg-gradient-to-r from-blue-700 to-gray-800 mx-auto"></div>
          </div>
          
          {/* LinkedIn Strategy Gallery */}
          <div className="grid md:grid-cols-3 gap-4 mb-12">
            <img 
              src="https://images.unsplash.com/photo-1557804506-669a67965ba0?w=400&h=300&fit=crop&q=80"
              alt="LinkedIn networking"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
            <img 
              src="https://images.unsplash.com/photo-1553877522-43269d4ea984?w=400&h=300&fit=crop&q=80"
              alt="Professional branding"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
            <img 
              src="https://images.unsplash.com/photo-1600880292203-757bb62b4baf?w=400&h=300&fit=crop&q=80"
              alt="Career growth"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
          </div>

          <div className="grid md:grid-cols-2 gap-6">
            {modules.map((module, index) => (
              <div key={index} className="flex items-start gap-4 bg-gradient-to-r from-blue-50 to-gray-50 rounded-xl p-6 hover:shadow-lg transition-all">
                <div className="w-10 h-10 rounded-full bg-gradient-to-br from-blue-700 to-gray-800 flex items-center justify-center flex-shrink-0">
                  <CheckCircle2 className="w-6 h-6 text-white" />
                </div>
                <p className="text-gray-800 text-lg font-medium">{module}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* WHO SHOULD JOIN */}
      <section className="py-20 px-6 bg-gradient-to-br from-blue-800 to-gray-900 text-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold mb-4">Perfect For</h2>
            <p className="text-xl text-blue-100">This workshop is designed for...</p>
          </div>
          <div className="grid md:grid-cols-4 gap-6">
            {['Job Seekers', 'Students', 'Entrepreneurs', 'Freelancers', 
              'Sales Professionals', 'Consultants', 'Career Changers', 'Corporate Trainers'].map((role, index) => (
              <div key={index} className="bg-white/10 backdrop-blur-sm rounded-xl p-6 text-center hover:bg-white/20 transition-all border border-yellow-400/30">
                <Star className="w-8 h-8 text-yellow-400 mx-auto mb-3" />
                <p className="font-semibold text-lg">{role}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* HANDS-ON PROJECTS */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Hands-On Projects</h2>
            <p className="text-xl text-gray-600">Build your LinkedIn presence live</p>
          </div>

          {/* Project Gallery */}
          <div className="grid md:grid-cols-4 gap-4 mb-12">
            <img 
              src="https://images.unsplash.com/photo-1557804506-669a67965ba0?w=300&h=300&fit=crop&q=80"
              alt="Profile optimization"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1542744173-8e7e53415bb0?w=300&h=300&fit=crop&q=80"
              alt="Content creation"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1553877522-43269d4ea984?w=300&h=300&fit=crop&q=80"
              alt="Networking strategy"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1600880292203-757bb62b4baf?w=300&h=300&fit=crop&q=80"
              alt="Lead generation"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
          </div>

          <div className="grid md:grid-cols-2 gap-8">
            <div className="bg-gradient-to-br from-blue-700 to-gray-900 rounded-3xl p-10 text-white">
              <Linkedin className="w-16 h-16 mb-6" />
              <h3 className="text-3xl font-bold mb-6">What You'll Create</h3>
              <ul className="space-y-4">
                {projects.map((project, index) => (
                  <li key={index} className="flex items-start gap-3">
                    <CheckCircle2 className="w-6 h-6 flex-shrink-0 mt-1" />
                    <span className="text-lg">{project}</span>
                  </li>
                ))}
              </ul>
            </div>
            <div className="bg-gray-50 rounded-3xl p-10 border-2 border-blue-200">
              <Briefcase className="w-16 h-16 text-blue-700 mb-6" />
              <h3 className="text-3xl font-bold text-gray-900 mb-6">Your Deliverables</h3>
              <ul className="space-y-4">
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-blue-700 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">1</span>
                  </div>
                  <span className="text-lg text-gray-700">Recruiter-optimized LinkedIn profile</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-blue-700 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">2</span>
                  </div>
                  <span className="text-lg text-gray-700">10 AI-generated posts</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-blue-700 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">3</span>
                  </div>
                  <span className="text-lg text-gray-700">30-day content calendar</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-blue-700 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">4</span>
                  </div>
                  <span className="text-lg text-gray-700">Automated outreach system</span>
                </li>
              </ul>
            </div>
          </div>
        </div>
      </section>

      {/* TOOLS */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-100 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">AI Tools You'll Master</h2>
            <p className="text-xl text-gray-600">Industry-leading LinkedIn automation platforms</p>
          </div>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
            {tools.map((tool, index) => (
              <div key={index} className="bg-white rounded-2xl p-8 shadow-lg hover:shadow-xl transition-all text-center border-2 border-gray-100 hover:border-blue-300 group">
                <div className="text-5xl mb-4 group-hover:scale-110 transition-transform">{tool.icon}</div>
                <p className="font-bold text-gray-900">{tool.name}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* CAREER OUTCOMES */}
      <section className="py-20 px-6 bg-gradient-to-br from-gray-900 to-blue-900">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-white mb-4">Career Paths</h2>
            <p className="text-xl text-gray-200">Launch your LinkedIn career</p>
          </div>
          <div className="grid md:grid-cols-3 gap-6">
            {['LinkedIn Personal Branding Specialist', 'AI Content Writer for LinkedIn', 'Social Selling Executive', 
              'Corporate Branding Assistant', 'Outreach & Lead Generation Specialist', 'Career Coach'].map((career, index) => (
              <div key={index} className="bg-white/10 backdrop-blur-md rounded-xl p-6 text-center hover:bg-white/20 transition-all border-2 border-white/30">
                <Target className="w-10 h-10 text-yellow-400 mx-auto mb-3" />
                <p className="font-bold text-white text-lg">{career}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* TRAINERS */}
      <section className="py-20 px-6 bg-white relative overflow-hidden">
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1542744173-8e7e53415bb0?w=1920&h=1080&fit=crop&q=80"
            alt="Training background"
            className="w-full h-full object-cover opacity-5"
          />
        </div>
        <div className="container mx-auto max-w-6xl relative z-10">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Expert Trainers</h2>
            <div className="w-32 h-1 bg-gradient-to-r from-blue-700 to-gray-800 mx-auto"></div>
          </div>
          <div className="bg-gradient-to-br from-blue-700 to-gray-900 rounded-3xl p-12 text-white shadow-2xl">
            <div className="grid md:grid-cols-2 gap-12 items-center">
              <div>
                <div className="flex items-center gap-4 mb-6">
                  <div className="w-16 h-16 rounded-full bg-yellow-400 flex items-center justify-center">
                    <Award className="w-10 h-10 text-blue-900" />
                  </div>
                  <div>
                    <h3 className="text-2xl font-bold">Industry Veterans</h3>
                    <p className="text-blue-100">9-15 Years Experience</p>
                  </div>
                </div>
                <p className="text-lg text-blue-100 mb-6 leading-relaxed">
                  Your workshop is led by <strong className="text-yellow-400">WorkshopLibrary-certified LinkedIn & personal branding experts</strong> with proven experience in:
                </p>
                <ul className="space-y-3">
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>Corporate hiring & recruitment</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>LinkedIn content strategy</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>Personal branding & positioning</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>Lead generation & automation</span>
                  </li>
                </ul>
              </div>
              <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-8 border-2 border-yellow-400/50">
                <p className="text-lg text-blue-100 italic mb-6">
                  "Real professionals who have helped 5,000+ learners enhance their LinkedIn presence and land dream opportunities."
                </p>
                <div className="flex items-center gap-3">
                  <Shield className="w-8 h-8 text-yellow-400" />
                  <span className="text-yellow-400 font-bold">WorkshopLibrary Certified</span>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Pricing Section */}
      <section id="pricing-section" className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-12">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Workshop Pricing</h2>
            <p className="text-xl text-gray-600">Special limited-time offer for early birds</p>
          </div>
          <div className="bg-gradient-to-br from-slate-50 to-blue-50 rounded-3xl p-10 border-2 border-slate-200 shadow-xl">
            <div className="relative">
              <div className="bg-gradient-to-br from-white via-blue-50/30 to-purple-50/30 rounded-3xl p-10 shadow-2xl border-2 border-blue-200/50 relative overflow-hidden">
                <div className="absolute inset-0 bg-gradient-to-r from-blue-400/10 to-purple-400/10 rounded-3xl blur-xl"></div>
                <div className="relative z-10">
                  <div className="flex justify-center mb-6">
                    <div className="bg-gradient-to-r from-orange-500 to-red-500 text-white px-6 py-3 rounded-full font-bold text-sm shadow-lg inline-flex items-center gap-2">
                      <span>🔥</span><span>Limited-Time Deal — 50% OFF</span>
                    </div>
                  </div>
                  <div className="text-center mb-4"><p className="text-gray-400 text-lg line-through">$100 / 365 AED</p></div>
                  <div className="text-center mb-4">
                    <div className="flex items-center justify-center gap-4 mb-3">
                      <p className="text-6xl md:text-7xl font-black text-blue-700">$50</p>
                      <span className="text-4xl text-gray-600 font-bold">/</span>
                      <p className="text-6xl md:text-7xl font-black text-blue-700">185 AED</p>
                    </div>
                    <p className="text-gray-700 font-semibold text-lg">For the Next 15 Days Only — Early Bird Offer</p>
                  </div>
                  <div className="text-center mb-8"><p className="text-gray-600 text-base">Enroll Now & Unlock Your Instant 50% Discount — Limited Seats</p></div>
                  <div className="text-center">
                    <button onClick={scrollToForm} className="w-full md:w-auto bg-gradient-to-r from-yellow-400 to-amber-500 hover:from-yellow-500 hover:to-amber-600 text-gray-900 px-12 py-5 rounded-full font-bold text-xl shadow-2xl hover:shadow-3xl transition-all hover:scale-105 inline-flex items-center justify-center gap-3">
                      Claim This Deal<ArrowRight className="w-6 h-6" />
                    </button>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <div className="mt-8">
            <h3 className="text-2xl font-bold text-gray-900 text-center mb-6">Additional Group & Corporate Discounts</h3>
            <div className="bg-white rounded-2xl p-8 shadow-lg border border-gray-200">
              <div className="flex flex-col md:flex-row items-start gap-6">
                <div className="flex-shrink-0">
                  <div className="w-16 h-16 bg-slate-100 rounded-xl flex items-center justify-center">
                    <Users className="w-8 h-8 text-slate-700" />
                  </div>
                </div>
                <div className="flex-1">
                  <h4 className="text-xl font-bold text-gray-900 mb-4">Save More When You Learn Together</h4>
                  <ul className="space-y-3 mb-6">
                    <li className="flex items-start gap-3 text-gray-700"><CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" /><span>10% extra discount for group bookings (2 or more participants)</span></li>
                    <li className="flex items-start gap-3 text-gray-700"><CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" /><span>Special pricing available for corporate teams</span></li>
                    <li className="flex items-start gap-3 text-gray-700"><CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" /><span>Dedicated support and invoicing for companies</span></li>
                  </ul>
                  <button onClick={scrollToForm} className="border-2 border-slate-700 text-slate-700 hover:bg-slate-700 hover:text-white px-8 py-3 rounded-xl font-semibold transition-all inline-flex items-center gap-2">
                    Request Corporate Pricing<ArrowRight className="w-5 h-5" />
                  </button>
                </div>
              </div>
              <div className="mt-6 pt-6 border-t border-gray-200">
                <p className="text-sm text-gray-600 mb-2">✓ Discounts apply on top of the current 15-day promotional price.</p>
                <p className="text-sm text-gray-600">✓ For larger teams, customized offers are available.</p>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* CERTIFICATION */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-4xl">
          <div className="bg-gradient-to-br from-blue-800 to-gray-900 rounded-3xl p-12 text-center text-white border-4 border-yellow-400 shadow-2xl">
            <Award className="w-24 h-24 text-yellow-400 mx-auto mb-6" />
            <h2 className="text-4xl font-bold mb-4">Get Certified</h2>
            <p className="text-2xl text-blue-100 mb-6">
              🎓 LinkedIn Automation & Personal Branding Certificate
            </p>
            <p className="text-lg text-blue-200 mb-8">
              Issued by <strong className="text-yellow-400">Workshop Library</strong> - trusted by students & industry professionals
            </p>
            <div className="inline-flex items-center gap-3 bg-white/10 backdrop-blur-sm px-8 py-4 rounded-full border-2 border-yellow-400/50">
              <Shield className="w-6 h-6 text-yellow-400" />
              <span className="font-bold">Globally Recognized Certification</span>
            </div>
          </div>
        </div>
      </section>

      {/* FAQ */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Frequently Asked Questions</h2>
          </div>
          <div className="space-y-6">
            {[
              { q: 'Is this helpful for job seekers?', a: 'Yes, LinkedIn is the #1 platform for career opportunities.' },
              { q: 'Is the workshop beginner-friendly?', a: 'Yes, everything is taught step-by-step.' },
              { q: 'Will I get AI tools and templates?', a: 'Yes, downloadable resources are included.' },
              { q: 'Will you help optimize my profile?', a: 'Yes, live guidance is part of the workshop.' },
              { q: 'Are recordings available?', a: 'Yes, 30-day access is provided.' }
            ].map((faq, index) => (
              <div key={index} className="bg-gradient-to-r from-blue-50 to-gray-50 rounded-xl p-6 hover:shadow-lg transition-all border-2 border-blue-100 hover:border-blue-300">
                <h3 className="text-xl font-bold text-gray-900 mb-3 flex items-center gap-3">
                  <div className="w-8 h-8 rounded-full bg-blue-700 flex items-center justify-center text-white font-bold">
                    Q
                  </div>
                  {faq.q}
                </h3>
                <p className="text-lg text-gray-700 ml-11">{faq.a}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* REGISTRATION FORM */}
      <section id="registration-form" className="py-20 px-6 bg-gradient-to-br from-blue-800 to-gray-900">
        <div className="container mx-auto max-w-2xl">
          <div className="text-center mb-8">
            <h2 className="text-4xl font-bold text-white mb-4">Build Your LinkedIn Brand Today!</h2>
            <p className="text-xl text-blue-100">Limited seats for December 13, 2025</p>
          </div>
          <CourseLeadForm courseName="LinkedIn Automation & Personal Branding Workshop" formTitle="Register for Workshop" />
        </div>
      </section>

      {/* FOOTER */}
      <footer className="py-16 px-6 bg-gradient-to-r from-gray-900 via-gray-800 to-gray-900 text-white relative overflow-hidden">
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1451187580459-43490279c0fa?w=1920&h=1080&fit=crop&q=80"
            alt="Footer background"
            className="w-full h-full object-cover opacity-5"
          />
        </div>
        <div className="container mx-auto max-w-7xl relative z-10">
          <div className="text-center">
            <div className="flex items-center justify-center gap-3 mb-6">
              <div className="w-12 h-12 rounded-xl bg-gradient-to-br from-blue-700 to-gray-800 border-2 border-yellow-400/50 flex items-center justify-center">
                <Sparkles className="w-7 h-7 text-yellow-400" />
              </div>
              <div>
                <h3 className="font-bold text-xl">AI School for All</h3>
                <p className="text-xs text-gray-400">by Workshop Library</p>
              </div>
            </div>
            <p className="text-gray-300 mb-6">Professional AI and Creative Skills Education</p>
            <div className="flex items-center justify-center gap-3 mb-6">
              <Shield className="w-5 h-5 text-yellow-400" />
              <span className="text-sm text-gray-300 font-semibold">WorkshopLibrary Partner</span>
            </div>
            <p className="text-xs text-gray-500">
              © 2026 AI School for All. All rights reserved.
            </p>
          </div>
        </div>
      </footer>

      {/* Chatbot */}
      <Chatbot />
    </div>
  );
};

export default WorkshopLinkedIn;
