import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { 
  Calendar, 
  Clock, 
  Video,
  Users, 
  Globe,
  Sparkles,
  TrendingUp,
  CheckCircle2,
  Award,
  Download,
  ArrowRight,
  Target,
  Briefcase,
  DollarSign,
  Zap,
  MessageSquare,
  FileText,
  Laptop,
  Star,
  BookOpen,
  Shield
} from 'lucide-react';
import CourseLeadForm from '../components/CourseLeadForm';
import Chatbot from '../components/Chatbot';

const WorkshopFreelancing = () => {
  const navigate = useNavigate();
  const [timeLeft, setTimeLeft] = useState({ days: 0, hours: 0, minutes: 0, seconds: 0 });

  useEffect(() => {
    // Workshop starts December 21, 2025 at 12:00 AM EST
    const workshopDate = new Date('2025-12-21T05:00:00Z'); // 12:00 AM EST = 05:00 UTC
    const targetDate = workshopDate.getTime();

    const updateCountdown = () => {
      const now = new Date().getTime();
      const distance = targetDate - now;

      if (distance > 0) {
        setTimeLeft({
          days: Math.floor(distance / (1000 * 60 * 60 * 24)),
          hours: Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60)),
          minutes: Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60)),
          seconds: Math.floor((distance % (1000 * 60)) / 1000)
        });
      } else {
        setTimeLeft({ days: 0, hours: 0, minutes: 0, seconds: 0 });
      }
    };

    updateCountdown();
    const interval = setInterval(updateCountdown, 1000);
    return () => clearInterval(interval);
  }, []);

  const scrollToForm = () => {
    const formElement = document.getElementById('registration-form');
    if (formElement) {
      formElement.scrollIntoView({ behavior: 'smooth', block: 'start' });
    }
  };

  const scrollToPricing = () => {
    const pricingElement = document.getElementById('pricing-section');
    if (pricingElement) {
      pricingElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
    }
  };

  const modules = [
    "Finding profitable freelance niches",
    "Creating multiple services using AI",
    "AI-assisted content writing & rewriting",
    "Designing social media posts for clients",
    "Resume writing & portfolio building",
    "Simple video editing & reels creation",
    "AI website building (no coding needed)",
    "Client communication using AI templates",
    "Proposal writing & pricing strategies",
    "Building a freelance portfolio in 1 day"
  ];

  const stats = [
    { value: "2nd Largest", label: "India's global freelancing rank" },
    { value: "78%", label: "Freelancers use AI tools daily" },
    { value: "200-300%", label: "AI increases delivery speed" },
    { value: "$4.2T", label: "Global freelance market by 2030" }
  ];

  const tools = [
    { name: 'ChatGPT', icon: '🤖' },
    { name: 'Canva AI', icon: '🎨' },
    { name: 'InVideo / CapCut', icon: '🎬' },
    { name: 'Framer AI', icon: '🌐' },
    { name: 'Notion AI', icon: '📝' },
    { name: 'Jasper', icon: '✍️' },
    { name: 'Resume.io', icon: '📄' }
  ];

  const projects = [
    "Create 3 client-ready social media posts",
    "Write a fully AI-crafted professional resume",
    "Create a one-page website using AI",
    "Build 2 reels for a dummy brand",
    "Create a freelance gig listing (Upwork/Fiverr style)",
    "Build your Freelance Portfolio Folder"
  ];

  const careerOutcomes = [
    "AI Content Writer",
    "AI Graphic Designer",
    "Social Media Manager",
    "Reels/Short-Form Video Editor",
    "Resume/Portfolio Writer",
    "AI Virtual Assistant",
    "Web & Landing Page Designer",
    "Freelance Automation Specialist"
  ];

  return (
    <div className="min-h-screen bg-white">
      {/* Navigation */}
      <nav className="fixed top-4 left-1/2 transform -translate-x-1/2 z-50 w-[95%] max-w-7xl">
        <div className="bg-black/30 backdrop-blur-lg rounded-full border border-white/20 px-8 py-3 shadow-2xl">
          <div className="flex items-center justify-between">
            <button onClick={() => navigate('/')} className="text-white font-bold text-lg hover:text-yellow-400 transition-colors">
              AI School for All
            </button>
            <div className="hidden md:flex items-center gap-6">
              <button onClick={() => navigate('/#home')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Home</button>
              <button onClick={() => navigate('/#workshops')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Workshops</button>
              <button 
                onClick={scrollToForm}
                className="bg-yellow-400 hover:bg-yellow-500 text-gray-900 px-6 py-2 rounded-full font-bold transition-all text-sm"
              >
                Register Now
              </button>
            </div>
          </div>
        </div>
      </nav>

      {/* EVENT-STYLE HERO BANNER */}
      <section className="relative min-h-screen flex items-center justify-center overflow-hidden pt-20">
        {/* Hero Background Image */}
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1522202176988-66273c2fd55f?w=1920&h=1080&fit=crop&q=80"
            alt="Freelancers working with AI tools"
            className="w-full h-full object-cover brightness-110"
          />
          <div className="absolute inset-0 bg-gradient-to-br from-orange-600/75 via-amber-600/70 to-red-600/75"></div>
        </div>

        {/* Animated particles */}
        <div className="absolute inset-0 opacity-20 z-10">
          <div className="absolute top-20 left-20 w-72 h-72 bg-yellow-400 rounded-full blur-3xl animate-pulse"></div>
          <div className="absolute bottom-20 right-20 w-96 h-96 bg-orange-400 rounded-full blur-3xl animate-pulse delay-1000"></div>
          <div className="absolute top-1/2 left-1/2 w-64 h-64 bg-red-400 rounded-full blur-3xl animate-pulse delay-500"></div>
        </div>

        <div className="container mx-auto px-6 relative z-20">
          <div className="max-w-5xl mx-auto text-center">
            {/* Event Badge */}
            <div className="inline-flex items-center gap-2 bg-white/20 backdrop-blur-md px-6 py-3 rounded-full mb-8 border border-white/30">
              <Briefcase className="w-5 h-5 text-white" />
              <span className="text-white font-bold text-sm uppercase tracking-wider">💼 Most Practical Workshop</span>
            </div>

            {/* Main Title */}
            <h1 className="text-5xl md:text-7xl font-black text-white mb-6 leading-tight drop-shadow-2xl">
              AI Tools for Freelancers
              <br />
              <span className="text-yellow-400">& Side-Income Skills</span>
            </h1>

            <p className="text-2xl text-white/90 mb-10 max-w-3xl mx-auto font-medium">
              Automate 70% of your workload · Build 10 services · Start earning immediately!
            </p>

            {/* Event Quick Info */}
            <div className="grid md:grid-cols-4 gap-4 mb-12 max-w-4xl mx-auto">
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Calendar className="w-8 h-8 text-white mx-auto mb-2" />
                <div className="text-sm text-white/80 mb-1">Next Cohort</div>
                <div className="text-lg font-bold text-white">21 Dec 2025</div>
                <div className="text-xs text-white/70">(Sunday)</div>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Clock className="w-8 h-8 text-white mx-auto mb-2" />
                <div className="text-sm text-white/80 mb-1">Duration</div>
                <div className="text-lg font-bold text-white">5 Hours</div>
                <div className="text-xs text-white/70">Intensive</div>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Video className="w-8 h-8 text-white mx-auto mb-2" />
                <div className="text-sm text-white/80 mb-1">Format</div>
                <div className="text-lg font-bold text-white">Live Online</div>
                <div className="text-xs text-white/70">Interactive</div>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Users className="w-8 h-8 text-white mx-auto mb-2" />
                <div className="text-sm text-white/80 mb-1">Seats Left</div>
                <div className="text-lg font-bold text-yellow-400">Limited</div>
                <div className="text-xs text-white/70">Filling Fast</div>
              </div>
            </div>

            {/* Countdown Timer */}
            <div className="mb-10">
              <div className="text-white/90 text-sm uppercase tracking-wider mb-4 font-bold">⏰ Enrollment Begins In</div>
              <div className="flex gap-4 justify-center">
                {Object.entries(timeLeft).map(([unit, value]) => (
                  <div key={unit} className="bg-white/10 backdrop-blur-md border border-white/20 rounded-2xl p-6 min-w-[100px]">
                    <div className="text-4xl font-black text-white mb-1">{value.toString().padStart(2, '0')}</div>
                    <div className="text-xs text-white/70 uppercase tracking-wider">{unit}</div>
                  </div>
                ))}
              </div>
            </div>

            {/* CTA Buttons */}
            <div className="flex flex-col sm:flex-row gap-4 justify-center items-center">
              <button 
                onClick={scrollToForm}
                className="group bg-white hover:bg-yellow-400 text-orange-600 px-10 py-5 rounded-full font-bold text-lg shadow-2xl hover:shadow-3xl transition-all hover:scale-105 flex items-center gap-2"
              >
                Secure Your Spot Now
                <ArrowRight className="w-5 h-5 group-hover:translate-x-1 transition-transform" />
              </button>
              <button 
                onClick={() => document.getElementById('modules-section')?.scrollIntoView({ behavior: 'smooth' })}
                className="bg-white/10 backdrop-blur-md border-2 border-white/30 text-white px-10 py-5 rounded-full font-bold text-lg hover:bg-white/20 transition-all flex items-center gap-2"
              >
                <Download className="w-5 h-5" />
                View Curriculum
              </button>
            </div>

            {/* Trust Badge */}
            <div className="mt-12 flex items-center justify-center gap-6 flex-wrap">
              <div className="flex items-center gap-2 text-white/90">
                <Star className="w-5 h-5 text-yellow-400 fill-yellow-400" />
                <span className="font-semibold">4.9/5 Rating</span>
              </div>
              <div className="w-1 h-1 bg-white/50 rounded-full"></div>
              <div className="flex items-center gap-2 text-white/90">
                <Users className="w-5 h-5" />
                <span className="font-semibold">5,000+ Students</span>
              </div>
              <div className="w-1 h-1 bg-white/50 rounded-full"></div>
              <div className="flex items-center gap-2 text-white/90">
                <Award className="w-5 h-5" />
                <span className="font-semibold">Certificate Included</span>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Promotional Banner - Limited Time Deal */}
      <section className="py-6 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <button
            onClick={scrollToPricing}
            className="w-full bg-gradient-to-r from-orange-500 to-red-500 hover:from-orange-600 hover:to-red-600 text-white px-8 py-5 rounded-full font-bold text-lg shadow-xl hover:shadow-2xl transition-all hover:scale-105 flex items-center justify-center gap-3"
          >
            <span className="text-2xl">🔥</span>
            <span>Limited-Time Deal — 50% OFF</span>
            <ArrowRight className="w-6 h-6" />
          </button>
        </div>
      </section>

      {/* Industry Stats Section */}
      <section className="py-20 px-6 bg-gradient-to-b from-white to-orange-50">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">
              Why Freelancing with AI is the Future
            </h2>
            <p className="text-xl text-gray-600">The market is booming — and AI gives you the competitive edge</p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6 mb-12">
            {stats.map((stat, index) => (
              <div key={index} className="bg-white rounded-2xl p-8 border-2 border-orange-100 hover:border-orange-300 hover:shadow-xl transition-all text-center">
                <div className="text-4xl font-bold bg-gradient-to-r from-orange-600 to-red-600 bg-clip-text text-transparent mb-2">
                  {stat.value}
                </div>
                <div className="text-sm text-gray-600">{stat.label}</div>
              </div>
            ))}
          </div>

          <div className="bg-gradient-to-r from-orange-100 to-red-100 rounded-3xl p-8 border-2 border-orange-200 text-center">
            <div className="flex items-center justify-center gap-3 mb-3">
              <TrendingUp className="w-8 h-8 text-orange-600" />
              <h3 className="text-2xl font-bold text-gray-900">Potential Monthly Income</h3>
            </div>
            <p className="text-5xl font-black bg-gradient-to-r from-orange-600 to-red-600 bg-clip-text text-transparent mb-3">
              ₹50,000 - ₹80,000
            </p>
            <p className="text-gray-700">By offering just 3-4 services on Upwork & Fiverr with AI automation</p>
          </div>
        </div>
      </section>

      {/* What You'll Learn */}
      <section id="modules-section" className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">What You Will Learn</h2>
            <p className="text-xl text-gray-600">10 comprehensive modules covering all aspects of AI-powered freelancing</p>
          </div>

          <div className="grid md:grid-cols-2 gap-6">
            {modules.map((module, index) => (
              <div key={index} className="flex items-start gap-4 bg-gradient-to-br from-orange-50 to-white rounded-2xl p-6 border-2 border-orange-100 hover:border-orange-300 hover:shadow-lg transition-all group">
                <div className="bg-gradient-to-br from-orange-500 to-red-500 text-white rounded-xl w-10 h-10 flex items-center justify-center font-bold flex-shrink-0 group-hover:scale-110 transition-transform">
                  {index + 1}
                </div>
                <p className="text-gray-900 font-semibold text-lg pt-1">{module}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Hands-On Projects */}
      <section className="py-20 px-6 bg-gradient-to-b from-orange-50 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Hands-On Projects</h2>
            <p className="text-xl text-gray-600">Build your complete freelance portfolio in just 5 hours</p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6 mb-12">
            {projects.map((project, index) => (
              <div key={index} className="bg-white rounded-2xl p-6 border-2 border-orange-100 hover:shadow-lg transition-all">
                <div className="flex items-start gap-3">
                  <div className="bg-orange-500 text-white rounded-full w-8 h-8 flex items-center justify-center font-bold text-sm flex-shrink-0">
                    {index + 1}
                  </div>
                  <p className="text-gray-900 font-semibold pt-1">{project}</p>
                </div>
              </div>
            ))}
          </div>

          <div className="grid md:grid-cols-3 gap-6">
            <div className="bg-gradient-to-br from-pink-50 to-orange-50 rounded-2xl p-6 border-2 border-pink-200">
              <div className="text-4xl mb-3">📱</div>
              <h3 className="text-xl font-bold text-gray-900 mb-2">Social Media Pack</h3>
              <p className="text-gray-600 mb-3">3 professional posts</p>
              <div className="text-orange-600 font-bold">Ready to sell: ₹500-800</div>
            </div>
            <div className="bg-gradient-to-br from-blue-50 to-cyan-50 rounded-2xl p-6 border-2 border-blue-200">
              <div className="text-4xl mb-3">📄</div>
              <h3 className="text-xl font-bold text-gray-900 mb-2">AI Resume</h3>
              <p className="text-gray-600 mb-3">Professional resume in 20 mins</p>
              <div className="text-orange-600 font-bold">Ready to sell: ₹300-600</div>
            </div>
            <div className="bg-gradient-to-br from-purple-50 to-pink-50 rounded-2xl p-6 border-2 border-purple-200">
              <div className="text-4xl mb-3">🌐</div>
              <h3 className="text-xl font-bold text-gray-900 mb-2">Landing Page</h3>
              <p className="text-gray-600 mb-3">One-page website (no code)</p>
              <div className="text-orange-600 font-bold">Ready to sell: ₹2000-4000</div>
            </div>
          </div>
        </div>
      </section>

      {/* Career Outcomes */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Career Outcomes</h2>
            <p className="text-xl text-gray-600">Start offering these in-demand services immediately</p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
            {careerOutcomes.map((outcome, index) => (
              <div key={index} className="bg-gradient-to-br from-orange-50 to-white rounded-2xl p-6 border-2 border-orange-100 hover:border-orange-300 hover:shadow-xl transition-all group text-center">
                <div className="bg-gradient-to-br from-orange-500 to-red-500 text-white rounded-xl w-12 h-12 flex items-center justify-center mx-auto mb-4 group-hover:scale-110 transition-transform">
                  <Briefcase className="w-6 h-6" />
                </div>
                <h3 className="text-lg font-bold text-gray-900">{outcome}</h3>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Tools You Will Master */}
      <section className="py-20 px-6 bg-gradient-to-b from-orange-50 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Tools You Will Master</h2>
            <p className="text-xl text-gray-600">Industry-standard AI tools used by top freelancers</p>
          </div>

          <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-7 gap-6">
            {tools.map((tool, index) => (
              <div key={index} className="bg-white rounded-2xl p-6 border-2 border-orange-100 hover:border-orange-300 hover:shadow-lg transition-all text-center group">
                <div className="text-4xl mb-3 group-hover:scale-110 transition-transform">{tool.icon}</div>
                <p className="font-bold text-gray-900 text-sm">{tool.name}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* About Trainers */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Meet Your Trainers</h2>
            <p className="text-xl text-gray-600">Learn from industry experts with 10+ years of experience</p>
          </div>

          <div className="bg-gradient-to-br from-orange-50 to-red-50 rounded-3xl p-10 border-2 border-orange-200">
            <div className="flex flex-col md:flex-row items-center gap-8">
              <div className="flex-shrink-0">
                <img 
                  src="https://images.unsplash.com/photo-1573496359142-b8d87734a5a2?w=300&h=300&fit=crop&q=80"
                  alt="Expert trainer"
                  className="w-48 h-48 rounded-full object-cover border-4 border-orange-300"
                />
              </div>
              <div>
                <div className="flex items-center gap-2 mb-4">
                  <Award className="w-6 h-6 text-orange-600" />
                  <span className="text-orange-600 font-bold text-lg">WorkshopLibrary-Certified Experts</span>
                </div>
                <p className="text-gray-700 text-lg mb-6 leading-relaxed">
                  Led by freelancers & AI productivity experts with <span className="font-bold text-orange-600">10+ years of experience</span> in freelance platforms, AI workflows, and helping 5,000+ learners start their careers.
                </p>
                <div className="grid md:grid-cols-2 gap-4">
                  <div className="flex items-center gap-2 text-gray-700">
                    <CheckCircle2 className="w-5 h-5 text-orange-500" />
                    Upwork, Fiverr, Freelancer.com experts
                  </div>
                  <div className="flex items-center gap-2 text-gray-700">
                    <CheckCircle2 className="w-5 h-5 text-orange-500" />
                    AI-assisted content & creative workflows
                  </div>
                  <div className="flex items-center gap-2 text-gray-700">
                    <CheckCircle2 className="w-5 h-5 text-orange-500" />
                    Client communication systems
                  </div>
                  <div className="flex items-center gap-2 text-gray-700">
                    <CheckCircle2 className="w-5 h-5 text-orange-500" />
                    Portfolio building & gig creation
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Certification */}
      <section className="py-20 px-6 bg-gradient-to-b from-orange-50 to-white">
        <div className="container mx-auto max-w-4xl">
          <div className="bg-gradient-to-br from-orange-600 to-red-600 rounded-3xl p-12 text-white text-center shadow-2xl">
            <Award className="w-20 h-20 mx-auto mb-6" />
            <h2 className="text-4xl font-bold mb-4">Certification Included</h2>
            <div className="bg-white/20 backdrop-blur-sm rounded-2xl p-6 inline-block mb-4">
              <p className="text-2xl font-bold">🎓 AI Tools for Freelancers & Side-Income Skills Certificate</p>
            </div>
            <p className="text-xl text-white/90">
              Issued by Workshop Library · Recognized by agencies & clients
            </p>
          </div>
        </div>
      </section>

      {/* FAQs */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Frequently Asked Questions</h2>
          </div>

          <div className="space-y-4">
            {[
              {
                q: "Can I start earning after this workshop?",
                a: "Yes — you'll build real samples you can immediately offer as services."
              },
              {
                q: "Do I need prior freelancing experience?",
                a: "No, we start from scratch and cover everything you need to know."
              },
              {
                q: "Will I learn proposals & pricing?",
                a: "Yes, complete templates and pricing strategies are included."
              },
              {
                q: "Will you teach real platforms?",
                a: "Yes — Upwork, Fiverr, and other marketplaces are covered in detail."
              },
              {
                q: "Are recordings included?",
                a: "Yes, you get 30-day access to all workshop recordings."
              }
            ].map((faq, index) => (
              <div key={index} className="bg-gradient-to-r from-orange-50 to-white rounded-2xl p-6 border-2 border-orange-100 hover:border-orange-300 transition-all">
                <h3 className="text-xl font-bold text-gray-900 mb-3 flex items-start gap-3">
                  <span className="text-orange-500">Q.</span>
                  {faq.q}
                </h3>
                <p className="text-gray-700 text-lg pl-8">{faq.a}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Workshop Pricing Section */}
      <section id="pricing-section" className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-12">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Workshop Pricing</h2>
            <p className="text-xl text-gray-600">Special limited-time offer for early birds</p>
          </div>

          <div className="bg-gradient-to-br from-slate-50 to-blue-50 rounded-3xl p-10 border-2 border-slate-200 shadow-xl">
            {/* Premium DEAL BADGE */}
            <div className="relative">
              <div className="bg-gradient-to-br from-white via-blue-50/30 to-purple-50/30 rounded-3xl p-10 shadow-2xl border-2 border-blue-200/50 relative overflow-hidden">
                {/* Subtle glow effect */}
                <div className="absolute inset-0 bg-gradient-to-r from-blue-400/10 to-purple-400/10 rounded-3xl blur-xl"></div>

                {/* Content */}
                <div className="relative z-10">
                  {/* Top Badge */}
                  <div className="flex justify-center mb-6">
                    <div className="bg-gradient-to-r from-orange-500 to-red-500 text-white px-6 py-3 rounded-full font-bold text-sm shadow-lg inline-flex items-center gap-2">
                      <span>🔥</span>
                      <span>Limited-Time Deal — 50% OFF</span>
                    </div>
                  </div>

                  {/* Strikethrough Old Price */}
                  <div className="text-center mb-4">
                    <p className="text-gray-400 text-lg line-through">
                      $100 / 365 AED
                    </p>
                  </div>

                  {/* Main Price - BIG & BOLD */}
                  <div className="text-center mb-4">
                    <div className="flex items-center justify-center gap-4 mb-3">
                      <p className="text-6xl md:text-7xl font-black text-blue-700">
                        $50
                      </p>
                      <span className="text-4xl text-gray-600 font-bold">/</span>
                      <p className="text-6xl md:text-7xl font-black text-blue-700">
                        185 AED
                      </p>
                    </div>
                    <p className="text-gray-700 font-semibold text-lg">
                      For the Next 15 Days Only — Early Bird Offer
                    </p>
                  </div>

                  {/* Subtext */}
                  <div className="text-center mb-8">
                    <p className="text-gray-600 text-base">
                      Enroll Now & Unlock Your Instant 50% Discount — Limited Seats
                    </p>
                  </div>

                  {/* CTA Button */}
                  <div className="text-center">
                    <button 
                      onClick={scrollToForm}
                      className="w-full md:w-auto bg-gradient-to-r from-yellow-400 to-amber-500 hover:from-yellow-500 hover:to-amber-600 text-gray-900 px-12 py-5 rounded-full font-bold text-xl shadow-2xl hover:shadow-3xl transition-all hover:scale-105 inline-flex items-center justify-center gap-3"
                    >
                      Claim This Deal
                      <ArrowRight className="w-6 h-6" />
                    </button>
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* Additional Group & Corporate Discounts */}
          <div className="mt-8">
            <h3 className="text-2xl font-bold text-gray-900 text-center mb-6">
              Additional Group & Corporate Discounts
            </h3>

            <div className="bg-white rounded-2xl p-8 shadow-lg border border-gray-200">
              <div className="flex flex-col md:flex-row items-start gap-6">
                {/* Icon */}
                <div className="flex-shrink-0">
                  <div className="w-16 h-16 bg-slate-100 rounded-xl flex items-center justify-center">
                    <Users className="w-8 h-8 text-slate-700" />
                  </div>
                </div>

                {/* Content */}
                <div className="flex-1">
                  <h4 className="text-xl font-bold text-gray-900 mb-4">
                    Save More When You Learn Together
                  </h4>

                  <ul className="space-y-3 mb-6">
                    <li className="flex items-start gap-3 text-gray-700">
                      <CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" />
                      <span>10% extra discount for group bookings (2 or more participants)</span>
                    </li>
                    <li className="flex items-start gap-3 text-gray-700">
                      <CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" />
                      <span>Special pricing available for corporate teams</span>
                    </li>
                    <li className="flex items-start gap-3 text-gray-700">
                      <CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" />
                      <span>Flexible payment terms for organizations</span>
                    </li>
                  </ul>

                  <button
                    onClick={scrollToForm}
                    className="bg-slate-700 hover:bg-slate-800 text-white px-6 py-3 rounded-xl font-bold transition-all inline-flex items-center gap-2"
                  >
                    Inquire About Group Pricing
                    <ArrowRight className="w-5 h-5" />
                  </button>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Final CTA */}
      <section className="py-20 px-6 bg-gradient-to-br from-orange-600 via-red-600 to-orange-700 text-white">
        <div className="container mx-auto max-w-4xl text-center">
          <h2 className="text-5xl font-bold mb-6">Ready to Start Your Freelancing Journey?</h2>
          <p className="text-2xl text-white/90 mb-10">
            Join 5,000+ learners who've transformed their careers with AI-powered freelancing
          </p>
          
          <button 
            onClick={scrollToForm}
            className="bg-white text-orange-600 px-12 py-6 rounded-full font-bold text-xl hover:bg-yellow-400 transition-all shadow-2xl hover:scale-105 inline-flex items-center gap-3"
          >
            Enroll Now - Limited Seats
            <ArrowRight className="w-6 h-6" />
          </button>
          
          <div className="mt-12 grid grid-cols-3 gap-6 max-w-2xl mx-auto">
            <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-6 border border-white/20">
              <p className="text-3xl font-bold mb-1">5 Hours</p>
              <p className="text-white/80 text-sm">Intensive</p>
            </div>
            <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-6 border border-white/20">
              <p className="text-3xl font-bold mb-1">10 Services</p>
              <p className="text-white/80 text-sm">To Offer</p>
            </div>
            <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-6 border border-white/20">
              <p className="text-3xl font-bold mb-1">6 Projects</p>
              <p className="text-white/80 text-sm">Portfolio Ready</p>
            </div>
          </div>
        </div>
      </section>

      {/* Registration Form Section */}
      <section id="registration-form" className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-2xl">
          <div className="text-center mb-12">
            <h2 className="text-4xl font-bold text-gray-900 mb-4">Secure Your Spot</h2>
            <p className="text-xl text-gray-600">Fill in your details to reserve your seat</p>
          </div>
          <CourseLeadForm />
        </div>
      </section>

      {/* Chatbot */}
      <Chatbot />
    </div>
  );
};

export default WorkshopFreelancing;
