import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import { 
  Calendar, Clock, Video, Users, Globe, Sparkles, TrendingUp, 
  Award, CheckCircle2, Download, ArrowRight, Zap, Target, Star,
  Palette, Layout, Image as ImageIcon, Wand2, Layers, Type,
  BookOpen, Shield, Eye, Maximize2, FileText, Briefcase
} from 'lucide-react';
import CourseLeadForm from '../components/CourseLeadForm';
import Chatbot from '../components/Chatbot';

const WorkshopCanvaDesign = () => {
  const navigate = useNavigate();
  const [timeLeft, setTimeLeft] = useState({ days: 0, hours: 0, minutes: 0, seconds: 0 });

  useEffect(() => {
    // Workshop starts December 20, 2025 at 12:00 AM EST
    const workshopDate = new Date('2025-12-20T05:00:00Z'); // 12:00 AM EST = 05:00 UTC
    const targetDate = workshopDate.getTime();

    const updateCountdown = () => {
      const now = new Date().getTime();
      const distance = targetDate - now;

      if (distance > 0) {
        setTimeLeft({
          days: Math.floor(distance / (1000 * 60 * 60 * 24)),
          hours: Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60)),
          minutes: Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60)),
          seconds: Math.floor((distance % (1000 * 60)) / 1000)
        });
      } else {
        setTimeLeft({ days: 0, hours: 0, minutes: 0, seconds: 0 });
      }
    };

    updateCountdown();
    const interval = setInterval(updateCountdown, 1000);
    return () => clearInterval(interval);
  }, []);

  const scrollToForm = () => {
    const formElement = document.getElementById('registration-form');
    if (formElement) {
      formElement.scrollIntoView({ behavior: 'smooth', block: 'start' });
    }
  };

  const scrollToPricing = () => {
    const pricingElement = document.getElementById('pricing-section');
    if (pricingElement) {
      pricingElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
    }
  };

  const modules = [
    'Canva AI essentials & interface mastery',
    'Magic Design workflow (auto-layouts)',
    'Creating aesthetic social media posts',
    'Using AI for text, image & video enhancements',
    'Designing carousels & reels in Canva',
    'Brand Kit creation (colors, fonts & identity)',
    'Image editing: cutouts, cleanup, color grading',
    'Auto-resizing assets for multi-platform use'
  ];

  const stats = [
    { value: '90%', label: 'Small businesses use Canva globally' },
    { value: '70%', label: 'Time saved with AI design tools' },
    { value: '4x', label: 'Faster task completion with Canva AI' },
    { value: '120%', label: 'Engagement boost with visual content' }
  ];

  const tools = [
    { name: 'Canva AI Suite', icon: '🎨' },
    { name: 'Magic Design', icon: '✨' },
    { name: 'Magic Write', icon: '✍️' },
    { name: 'Magic Eraser', icon: '🪄' },
    { name: 'Brand Kit', icon: '🎯' },
    { name: 'Video Editor', icon: '🎬' },
    { name: 'AI Image Gen', icon: '🖼️' }
  ];

  const projects = [
    'Create a full Brand Kit for yourself',
    'Design 5 social media creatives',
    'Create a 6-slide carousel',
    'Build a professional poster/ad creative',
    'Create a reel using Canva AI features',
    'Export complete Design Portfolio Folder'
  ];

  return (
    <div className="min-h-screen bg-white">
      {/* Navigation */}
      <nav className="fixed top-4 left-1/2 transform -translate-x-1/2 z-50 w-[95%] max-w-7xl">
        <div className="bg-black/30 backdrop-blur-lg rounded-full border border-white/20 px-8 py-3 shadow-2xl">
          <div className="flex items-center justify-between">
            <button onClick={() => navigate('/')} className="text-white font-bold text-lg hover:text-yellow-400 transition-colors">
              AI School for All
            </button>
            <div className="hidden md:flex items-center gap-6">
              <button onClick={() => navigate('/#home')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Home</button>
              <button onClick={() => navigate('/#workshops')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Workshops</button>
              <button 
                onClick={scrollToForm}
                className="bg-yellow-400 hover:bg-yellow-500 text-gray-900 px-6 py-2 rounded-full font-bold transition-all text-sm"
              >
                Register Now
              </button>
            </div>
          </div>
        </div>
      </nav>

      {/* CREATIVE DESIGN HERO */}
      <section className="relative min-h-screen flex items-center justify-center overflow-hidden pt-20">
        {/* Creative Design Background */}
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1561070791-2526d30994b5?w=1920&h=1080&fit=crop&q=80"
            alt="Creative Design Work"
            className="w-full h-full object-cover brightness-110"
          />
          <div className="absolute inset-0 bg-gradient-to-br from-pink-500/70 via-rose-500/65 to-orange-500/70"></div>
        </div>

        {/* Animated creative particles */}
        <div className="absolute inset-0 opacity-20 z-10">
          <div className="absolute top-20 left-20 w-72 h-72 bg-yellow-400 rounded-full blur-3xl animate-pulse"></div>
          <div className="absolute bottom-20 right-20 w-96 h-96 bg-pink-400 rounded-full blur-3xl animate-pulse delay-1000"></div>
          <div className="absolute top-1/2 left-1/2 w-64 h-64 bg-orange-400 rounded-full blur-3xl animate-pulse delay-500"></div>
        </div>

        <div className="container mx-auto px-6 relative z-20">
          <div className="max-w-5xl mx-auto text-center">
            {/* Event Badge */}
            <div className="inline-flex items-center gap-2 bg-white/20 backdrop-blur-md px-6 py-3 rounded-full mb-8 border border-white/30">
              <Palette className="w-5 h-5 text-white" />
              <span className="text-white font-bold text-sm uppercase tracking-wider">🎨 Unleash Your Creativity</span>
            </div>

            {/* Main Title */}
            <h1 className="text-5xl md:text-7xl font-black text-white mb-6 leading-tight drop-shadow-2xl">
              Canva + AI
              <br />
              <span className="text-yellow-400">Design Secrets</span>
            </h1>

            <p className="text-2xl text-white/90 mb-10 max-w-3xl mx-auto font-medium">
              Create stunning designs 4x faster - Zero experience needed!
            </p>

            {/* Event Quick Info */}
            <div className="grid md:grid-cols-4 gap-4 mb-12 max-w-4xl mx-auto">
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Calendar className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">20 Dec 2025</p>
                <p className="text-white/80 text-sm">Saturday</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Clock className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">4 Hours</p>
                <p className="text-white/80 text-sm">Creative</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Video className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">Live Online</p>
                <p className="text-white/80 text-sm">Interactive</p>
              </div>
              <div className="bg-white/10 backdrop-blur-md rounded-2xl p-6 border border-white/20">
                <Zap className="w-8 h-8 text-white mx-auto mb-2" />
                <p className="text-white font-bold text-lg">AI-Powered</p>
                <p className="text-white/80 text-sm">Magic Tools</p>
              </div>
            </div>

            {/* CTA */}
            <button 
              onClick={scrollToForm}
              className="bg-yellow-400 hover:bg-yellow-500 text-gray-900 px-12 py-6 rounded-2xl font-black text-xl shadow-2xl hover:scale-105 transition-all inline-flex items-center gap-3"
            >
              <Wand2 className="w-6 h-6" />
              Start Designing Like a Pro
              <ArrowRight className="w-6 h-6" />
            </button>

            <p className="text-white/90 mt-6 text-sm">
              <strong>100% Beginner-Friendly</strong> - No experience needed!
            </p>
          </div>
        </div>
      </section>

      {/* COUNTDOWN TIMER */}
      <section className="py-16 px-6 bg-gradient-to-b from-pink-900 to-orange-900">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-8">
            <h3 className="text-3xl font-bold text-yellow-400 mb-2">Workshop Starts In</h3>
            <p className="text-gray-200">December 20, 2025 • 12:00 AM EST</p>
          </div>
          <div className="grid grid-cols-4 gap-4 max-w-2xl mx-auto">
            {['days', 'hours', 'minutes', 'seconds'].map((unit) => (
              <div key={unit} className="bg-gradient-to-br from-pink-700 to-orange-700 rounded-xl p-6 text-center border-2 border-yellow-400/30">
                <div className="text-5xl font-bold text-yellow-400 mb-2">{timeLeft[unit]}</div>
                <div className="text-gray-200 text-sm uppercase tracking-wide">{unit}</div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Promotional Banner - Limited Time Deal */}
      <section className="py-6 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <button onClick={scrollToPricing} className="w-full bg-gradient-to-r from-orange-500 to-red-500 hover:from-orange-600 hover:to-red-600 text-white px-8 py-5 rounded-full font-bold text-lg shadow-xl hover:shadow-2xl transition-all hover:scale-105 flex items-center justify-center gap-3">
            <span className="text-2xl">🔥</span><span>Limited-Time Deal — 50% OFF</span><ArrowRight className="w-6 h-6" />
          </button>
        </div>
      </section>

      {/* WHY THIS WORKSHOP */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="grid md:grid-cols-2 gap-12 items-center">
            <div>
              <div className="inline-flex items-center gap-2 bg-pink-100 px-4 py-2 rounded-full mb-6">
                <Palette className="w-5 h-5 text-pink-700" />
                <span className="text-pink-900 font-bold text-sm">DESIGN MADE EASY</span>
              </div>
              <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-6">Why Everyone Needs Canva AI Skills</h2>
              <p className="text-lg text-gray-700 mb-6 leading-relaxed">
                Design is no longer just for designers. Today, <strong className="text-pink-600">everyone needs to create stunning visuals</strong>—social media posts, presentations, ads, thumbnails, brand kits, and more.
              </p>
              <p className="text-lg text-gray-700 mb-6 leading-relaxed">
                This workshop takes you from <strong className="text-pink-600">zero to professional designer</strong> using Canva's AI-powered Magic Design, Magic Write, background cleanup, auto-animation, and brand intelligence.
              </p>
              <p className="text-lg text-gray-700 leading-relaxed">
                Walk away with your own <strong className="text-orange-600">Brand Kit, Template Pack, and Portfolio</strong> showcasing your creative skills powered by AI!
              </p>
            </div>
            <div className="relative">
              <div className="bg-gradient-to-br from-pink-500 to-orange-600 rounded-3xl p-8 shadow-2xl">
                <img 
                  src="https://images.unsplash.com/photo-1626785774573-4b799315345d?w=600&h=400&fit=crop&q=80"
                  alt="Canva Design Work"
                  className="w-full h-80 object-cover rounded-2xl shadow-xl"
                />
                <div className="absolute -bottom-6 -right-6 bg-white rounded-2xl p-6 shadow-2xl border-4 border-yellow-400">
                  <div className="text-center">
                    <div className="text-4xl font-bold text-pink-600">4x</div>
                    <div className="text-sm text-gray-700 font-medium">Faster Design</div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* BEFORE/AFTER DESIGN TRANSFORMATIONS */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">See The Design Magic</h2>
            <p className="text-xl text-gray-600">Bad design vs. Canva AI magic!</p>
          </div>

          {/* Example 1: Social Post with Image */}
          <div className="mb-16">
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">Marketing Creative Transformation</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BEFORE ❌
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl bg-white">
                  <div className="aspect-square relative">
                    {/* Boring stock photo background */}
                    <img 
                      src="https://images.unsplash.com/photo-1441986300917-64674bd600d8?w=600&h=600&fit=crop&q=50&sat=-100"
                      alt="Boring store"
                      className="absolute inset-0 w-full h-full object-cover opacity-40"
                    />
                    <div className="relative bg-white/80 backdrop-blur-sm h-full p-8 flex flex-col justify-center">
                      <div className="bg-white border-2 border-gray-300 rounded p-6 text-center">
                        <p className="text-black text-2xl mb-3 font-bold" style={{fontFamily: 'Times New Roman'}}>SALE</p>
                        <p className="text-gray-700 text-base mb-3">50% OFF</p>
                        <p className="text-gray-600 text-sm mb-2">All Items</p>
                        <p className="text-gray-500 text-xs">Limited Time Only</p>
                        <div className="mt-4 pt-4 border-t border-gray-300">
                          <p className="text-gray-500 text-xs">www.myshop.com</p>
                          <p className="text-gray-500 text-xs">Call: 123-456-7890</p>
                        </div>
                      </div>
                    </div>
                  </div>
                  <div className="p-4 bg-red-50">
                    <p className="text-red-600 text-xs font-semibold">❌ Boring stock photo • Ugly font • No brand • Looks like spam</p>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Microsoft Word vibes • Gets ignored • 0 engagement</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  AFTER CANVA MAGIC ✅
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl">
                  <div className="aspect-square relative overflow-hidden">
                    {/* HOT creative with stunning image */}
                    <img 
                      src="https://images.unsplash.com/photo-1441986300917-64674bd600d8?w=600&h=600&fit=crop&q=90"
                      alt="Shopping"
                      className="absolute inset-0 w-full h-full object-cover brightness-75"
                    />
                    <div className="absolute inset-0 bg-gradient-to-br from-pink-600/80 via-rose-500/70 to-orange-600/80"></div>
                    
                    <div className="relative h-full p-8 flex flex-col justify-between">
                      {/* Top badge */}
                      <div className="flex justify-between items-start">
                        <div className="bg-yellow-400 text-gray-900 px-5 py-2 rounded-full font-black text-sm shadow-xl animate-pulse">
                          🔥 HOT DEAL
                        </div>
                        <div className="bg-white/20 backdrop-blur-md text-white px-4 py-2 rounded-full font-bold text-xs border-2 border-white/50">
                          LIMITED STOCK
                        </div>
                      </div>
                      
                      {/* Main content */}
                      <div className="text-center">
                        <div className="bg-white/10 backdrop-blur-xl rounded-3xl p-8 border-2 border-white/40 shadow-2xl">
                          <h3 className="text-6xl font-black text-white mb-2 drop-shadow-2xl leading-tight">50%</h3>
                          <p className="text-2xl font-black text-yellow-300 mb-4 drop-shadow-xl">OFF</p>
                          <div className="w-20 h-1 bg-yellow-400 mx-auto mb-4 rounded-full"></div>
                          <p className="text-xl font-bold text-white mb-2">SUMMER BLOWOUT</p>
                          <p className="text-white/90 text-sm font-semibold">Shop All Collections</p>
                        </div>
                      </div>
                      
                      {/* Bottom CTA */}
                      <div className="text-center">
                        <div className="inline-block bg-yellow-400 text-gray-900 px-8 py-3 rounded-full font-black text-lg shadow-2xl">
                          SHOP NOW →
                        </div>
                      </div>
                    </div>
                  </div>
                  <div className="p-4 bg-gradient-to-r from-green-50 to-emerald-50">
                    <p className="text-green-600 text-xs font-semibold">✅ HOT creative • Stunning images • Professional branding • STOPS THE SCROLL!</p>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Canva Magic Design • Viral-worthy • Gets 1000s of clicks!</p>
              </div>
            </div>
          </div>

          {/* Example 2: Podcast Audio Visualization */}
          <div className="mb-16">
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">Audio Waveform Design</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BEFORE ❌
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl bg-white">
                  <div className="aspect-video bg-gray-800 p-8 flex flex-col items-center justify-center">
                    <div className="text-center mb-6">
                      <p className="text-gray-400 text-sm mb-2">Podcast Episode 12</p>
                      <p className="text-white text-xs">audio_file_final_v2.mp3</p>
                    </div>
                    <div className="flex items-end gap-1 h-16">
                      {[3,5,2,4,3,5,2,4,3,5,2,4].map((h, i) => (
                        <div key={i} className="w-2 bg-gray-600" style={{height: `${h*10}px`}}></div>
                      ))}
                    </div>
                  </div>
                  <div className="p-4 bg-red-50">
                    <p className="text-red-600 text-xs font-semibold">❌ Plain waveform • No branding • Boring • Won't share</p>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Default audio player • Zero visual appeal</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  AFTER CANVA AUDIO ✅
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl">
                  <div className="aspect-video bg-gradient-to-br from-pink-600 via-rose-500 to-orange-500 p-8 flex flex-col items-center justify-center relative overflow-hidden">
                    <div className="absolute top-0 left-0 right-0 h-1 bg-yellow-400"></div>
                    <div className="absolute -top-10 -right-10 w-40 h-40 bg-yellow-300 rounded-full opacity-20"></div>
                    <div className="text-center mb-6 relative z-10">
                      <div className="inline-block bg-yellow-400 text-gray-900 px-4 py-1 rounded-full text-xs font-black mb-3">
                        🎙️ EPISODE 12
                      </div>
                      <h3 className="text-3xl font-black text-white mb-2 drop-shadow-2xl">The Growth Mindset</h3>
                      <p className="text-white/90 text-sm font-semibold">with John Doe</p>
                    </div>
                    <div className="flex items-end gap-1 h-20 relative z-10">
                      {[4,7,3,8,5,9,4,7,5,8,4,7,5,8,4,7].map((h, i) => (
                        <div key={i} className="w-3 bg-gradient-to-t from-yellow-300 to-white rounded-t-full" style={{height: `${h*8}px`}}></div>
                      ))}
                    </div>
                  </div>
                  <div className="p-4 bg-gradient-to-r from-green-50 to-emerald-50">
                    <p className="text-green-600 text-xs font-semibold">✅ Branded • Eye-catching • Shareable • Professional!</p>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Canva Audio Visualization • Social-ready • Gets shared!</p>
              </div>
            </div>
          </div>

          {/* Example 3: Branding with Images */}
          <div>
            <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">Brand Identity with Images</h3>
            <div className="grid md:grid-cols-2 gap-8">
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-red-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  BEFORE ❌
                </div>
                <div className="border-4 border-red-200 rounded-2xl overflow-hidden shadow-xl bg-white">
                  <div className="p-6 h-96">
                    <div className="bg-gray-100 rounded-lg h-full p-4 flex flex-col gap-3">
                      <div className="bg-white border border-gray-300 rounded p-3 flex items-center gap-3">
                        <div className="w-12 h-12 bg-gray-400 rounded"></div>
                        <div className="flex-1">
                          <div className="h-3 bg-gray-300 rounded w-3/4 mb-2"></div>
                          <div className="h-2 bg-gray-200 rounded w-1/2"></div>
                        </div>
                      </div>
                      <div className="bg-white border border-gray-300 rounded p-3 flex items-center gap-3">
                        <div className="w-12 h-12 bg-blue-300 rounded"></div>
                        <div className="flex-1">
                          <div className="h-3 bg-gray-300 rounded w-3/4 mb-2"></div>
                          <div className="h-2 bg-gray-200 rounded w-1/2"></div>
                        </div>
                      </div>
                      <div className="bg-white border border-gray-300 rounded p-3 flex items-center gap-3">
                        <div className="w-12 h-12 bg-red-300 rounded"></div>
                        <div className="flex-1">
                          <div className="h-3 bg-gray-300 rounded w-3/4 mb-2"></div>
                          <div className="h-2 bg-gray-200 rounded w-1/2"></div>
                        </div>
                      </div>
                    </div>
                  </div>
                  <div className="p-4 bg-red-50">
                    <p className="text-red-600 text-xs font-semibold">❌ Random images • No filter • Inconsistent • Looks messy</p>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">No brand colors • Random stock photos • Chaotic</p>
              </div>
              <div className="relative group">
                <div className="absolute -top-3 -left-3 bg-green-500 text-white px-6 py-2 rounded-full font-bold text-sm z-10 shadow-lg">
                  AFTER CANVA BRANDING ✅
                </div>
                <div className="border-4 border-green-200 rounded-2xl overflow-hidden shadow-xl">
                  <div className="p-6 h-96 bg-gradient-to-br from-pink-50 to-orange-50">
                    <div className="h-full flex flex-col gap-3">
                      <div className="bg-white rounded-2xl p-4 shadow-lg border-l-4 border-pink-500 flex items-center gap-4">
                        <div className="w-16 h-16 bg-gradient-to-br from-pink-500 to-rose-500 rounded-2xl flex items-center justify-center text-2xl">
                        </div>
                        <div className="flex-1">
                          <h4 className="text-sm font-black text-pink-600 mb-1">Product Launch</h4>
                          <p className="text-xs text-gray-600">New collection drops Friday</p>
                        </div>
                      </div>
                      <div className="bg-white rounded-2xl p-4 shadow-lg border-l-4 border-orange-500 flex items-center gap-4">
                        <div className="w-16 h-16 bg-gradient-to-br from-orange-500 to-yellow-400 rounded-2xl flex items-center justify-center text-2xl">
                        </div>
                        <div className="flex-1">
                          <h4 className="text-sm font-black text-orange-600 mb-1">Brand Colors</h4>
                          <p className="text-xs text-gray-600">Consistent visual identity</p>
                        </div>
                      </div>
                      <div className="bg-white rounded-2xl p-4 shadow-lg border-l-4 border-yellow-400 flex items-center gap-4">
                        <div className="w-16 h-16 bg-gradient-to-br from-yellow-400 to-orange-400 rounded-2xl flex items-center justify-center text-2xl">
                          🔥
                        </div>
                        <div className="flex-1">
                          <h4 className="text-sm font-black text-yellow-600 mb-1">Hot Deals</h4>
                          <p className="text-xs text-gray-600">Limited time offers</p>
                        </div>
                      </div>
                    </div>
                  </div>
                  <div className="p-4 bg-gradient-to-r from-green-50 to-emerald-50">
                    <p className="text-green-600 text-xs font-semibold">✅ Unified colors • Branded icons • Cohesive • Professional!</p>
                  </div>
                </div>
                <p className="mt-4 text-sm text-gray-600 text-center italic">Canva Brand Kit • Consistent visuals • Instant recognition!</p>
              </div>
            </div>
          </div>

          {/* Results Banner */}
          <div className="mt-16 bg-gradient-to-r from-pink-600 to-orange-600 rounded-3xl p-10 text-center text-white">
            <p className="text-3xl font-bold mb-4">Master All These Skills in Just 4 Hours!</p>
            <p className="text-xl text-white/90">From zero to professional designer - 100% beginner-friendly!</p>
          </div>
        </div>
      </section>

      {/* INDUSTRY INSIGHTS */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Industry Insights</h2>
            <p className="text-xl text-gray-600">Why Canva AI skills are essential</p>
          </div>
          <div className="grid md:grid-cols-4 gap-8">
            {stats.map((stat, index) => (
              <div key={index} className="bg-gradient-to-br from-pink-50 to-orange-50 rounded-2xl p-8 shadow-lg hover:shadow-xl transition-all border-2 border-pink-100">
                <div className="text-5xl font-black text-pink-600 mb-4">{stat.value}</div>
                <p className="text-gray-700 font-medium leading-relaxed">{stat.label}</p>
              </div>
            ))}
          </div>
          <div className="mt-12 text-center">
            <p className="text-lg text-gray-600">
              <span className="text-pink-600 font-bold">Freelance designers</span> earn <strong>₹25K-₹1.5L per month</strong>
            </p>
          </div>
        </div>
      </section>

      {/* WHAT YOU'LL LEARN */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">What You'll Master</h2>
            <div className="w-32 h-1 bg-gradient-to-r from-pink-500 to-orange-600 mx-auto"></div>
          </div>
          
          {/* Design Gallery */}
          <div className="grid md:grid-cols-3 gap-4 mb-12">
            <img 
              src="https://images.unsplash.com/photo-1561070791-2526d30994b5?w=400&h=300&fit=crop&q=80"
              alt="Graphic design"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
            <img 
              src="https://images.unsplash.com/photo-1626785774573-4b799315345d?w=400&h=300&fit=crop&q=80"
              alt="Creative work"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
            <img 
              src="https://images.unsplash.com/photo-1558655146-364adaf1fcc9?w=400&h=300&fit=crop&q=80"
              alt="Design workspace"
              className="w-full h-48 object-cover rounded-2xl shadow-lg"
            />
          </div>

          <div className="grid md:grid-cols-2 gap-6">
            {modules.map((module, index) => (
              <div key={index} className="flex items-start gap-4 bg-gradient-to-r from-pink-50 to-orange-50 rounded-xl p-6 hover:shadow-lg transition-all">
                <div className="w-10 h-10 rounded-full bg-gradient-to-br from-pink-600 to-orange-600 flex items-center justify-center flex-shrink-0">
                  <CheckCircle2 className="w-6 h-6 text-white" />
                </div>
                <p className="text-gray-800 text-lg font-medium">{module}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* WHO SHOULD JOIN */}
      <section className="py-20 px-6 bg-gradient-to-br from-pink-700 to-orange-700 text-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold mb-4">Perfect For</h2>
            <p className="text-xl text-pink-100">This workshop is designed for...</p>
          </div>
          <div className="grid md:grid-cols-4 gap-6">
            {['Students', 'Freelancers', 'Social Media Managers', 'Business Owners', 
              'Digital Marketers', 'Influencers', 'Educators', 'Anyone Creative!'].map((role, index) => (
              <div key={index} className="bg-white/10 backdrop-blur-sm rounded-xl p-6 text-center hover:bg-white/20 transition-all border border-yellow-400/30">
                <Star className="w-8 h-8 text-yellow-400 mx-auto mb-3" />
                <p className="font-semibold text-lg">{role}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* HANDS-ON PROJECTS */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Hands-On Projects</h2>
            <p className="text-xl text-gray-600">Build your design portfolio live</p>
          </div>

          {/* Project Gallery */}
          <div className="grid md:grid-cols-4 gap-4 mb-12">
            <img 
              src="https://images.unsplash.com/photo-1561070791-2526d30994b5?w=300&h=300&fit=crop&q=80"
              alt="Brand kit"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1626785774573-4b799315345d?w=300&h=300&fit=crop&q=80"
              alt="Social posts"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1558655146-364adaf1fcc9?w=300&h=300&fit=crop&q=80"
              alt="Carousel design"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
            <img 
              src="https://images.unsplash.com/photo-1559028012-481c04fa702d?w=300&h=300&fit=crop&q=80"
              alt="Portfolio"
              className="w-full h-48 object-cover rounded-xl shadow-lg hover:scale-105 transition-transform"
            />
          </div>

          <div className="grid md:grid-cols-2 gap-8">
            <div className="bg-gradient-to-br from-pink-600 to-orange-600 rounded-3xl p-10 text-white">
              <Palette className="w-16 h-16 mb-6" />
              <h3 className="text-3xl font-bold mb-6">What You'll Create</h3>
              <ul className="space-y-4">
                {projects.map((project, index) => (
                  <li key={index} className="flex items-start gap-3">
                    <CheckCircle2 className="w-6 h-6 flex-shrink-0 mt-1" />
                    <span className="text-lg">{project}</span>
                  </li>
                ))}
              </ul>
            </div>
            <div className="bg-gray-50 rounded-3xl p-10 border-2 border-pink-200">
              <Wand2 className="w-16 h-16 text-pink-600 mb-6" />
              <h3 className="text-3xl font-bold text-gray-900 mb-6">Your Deliverables</h3>
              <ul className="space-y-4">
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-pink-600 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">1</span>
                  </div>
                  <span className="text-lg text-gray-700">Complete Brand Kit (colors, fonts, logo)</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-pink-600 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">2</span>
                  </div>
                  <span className="text-lg text-gray-700">5 social media creatives</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-pink-600 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">3</span>
                  </div>
                  <span className="text-lg text-gray-700">6-slide professional carousel</span>
                </li>
                <li className="flex items-start gap-3">
                  <div className="w-6 h-6 rounded-full bg-pink-600 flex items-center justify-center flex-shrink-0 mt-1">
                    <span className="text-white text-xs font-bold">4</span>
                  </div>
                  <span className="text-lg text-gray-700">Design Portfolio Folder</span>
                </li>
              </ul>
            </div>
          </div>
        </div>
      </section>

      {/* TOOLS */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-100 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Canva AI Tools You'll Master</h2>
            <p className="text-xl text-gray-600">Industry-leading design platform</p>
          </div>
          <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
            {tools.map((tool, index) => (
              <div key={index} className="bg-white rounded-2xl p-8 shadow-lg hover:shadow-xl transition-all text-center border-2 border-gray-100 hover:border-pink-300 group">
                <div className="text-5xl mb-4 group-hover:scale-110 transition-transform">{tool.icon}</div>
                <p className="font-bold text-gray-900">{tool.name}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* CAREER OUTCOMES */}
      <section className="py-20 px-6 bg-gradient-to-br from-orange-700 to-pink-700">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-white mb-4">Career Paths</h2>
            <p className="text-xl text-orange-100">Launch your design career</p>
          </div>
          <div className="grid md:grid-cols-3 gap-6">
            {['Canva Design Specialist', 'Social Media Creative Designer', 'Reels & Carousel Designer', 
              'Presentation Designer', 'Brand Identity Assistant', 'Creative Freelancer', 'AI-Powered Content Designer'].map((career, index) => (
              <div key={index} className="bg-white/10 backdrop-blur-md rounded-xl p-6 text-center hover:bg-white/20 transition-all border-2 border-white/30">
                <Target className="w-10 h-10 text-yellow-400 mx-auto mb-3" />
                <p className="font-bold text-white text-lg">{career}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* TRAINERS */}
      <section className="py-20 px-6 bg-white relative overflow-hidden">
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1558655146-9f40138edfeb?w=1920&h=1080&fit=crop&q=80"
            alt="Training background"
            className="w-full h-full object-cover opacity-5"
          />
        </div>
        <div className="container mx-auto max-w-6xl relative z-10">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Expert Trainers</h2>
            <div className="w-32 h-1 bg-gradient-to-r from-pink-500 to-orange-600 mx-auto"></div>
          </div>
          <div className="bg-gradient-to-br from-pink-600 to-orange-700 rounded-3xl p-12 text-white shadow-2xl">
            <div className="grid md:grid-cols-2 gap-12 items-center">
              <div>
                <div className="flex items-center gap-4 mb-6">
                  <div className="w-16 h-16 rounded-full bg-yellow-400 flex items-center justify-center">
                    <Award className="w-10 h-10 text-pink-900" />
                  </div>
                  <div>
                    <h3 className="text-2xl font-bold">Creative Experts</h3>
                    <p className="text-pink-100">8-10+ Years Experience</p>
                  </div>
                </div>
                <p className="text-lg text-pink-100 mb-6 leading-relaxed">
                  Your workshop is led by <strong className="text-yellow-400">WorkshopLibrary-certified design experts</strong> with proven experience in:
                </p>
                <ul className="space-y-3">
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>Graphic design & branding</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>Creative direction</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>Social media design systems</span>
                  </li>
                  <li className="flex items-center gap-3">
                    <CheckCircle2 className="w-5 h-5 text-yellow-400 flex-shrink-0" />
                    <span>AI-powered creative workflows</span>
                  </li>
                </ul>
              </div>
              <div className="bg-white/10 backdrop-blur-sm rounded-2xl p-8 border-2 border-yellow-400/50">
                <p className="text-lg text-pink-100 italic mb-6">
                  "Real designers who have created visuals for startups, influencers, agencies & global campaigns."
                </p>
                <div className="flex items-center gap-3">
                  <Shield className="w-8 h-8 text-yellow-400" />
                  <span className="text-yellow-400 font-bold">WorkshopLibrary Certified</span>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Pricing Section */}
      <section id="pricing-section" className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-12">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Workshop Pricing</h2>
            <p className="text-xl text-gray-600">Special limited-time offer for early birds</p>
          </div>
          <div className="bg-gradient-to-br from-slate-50 to-blue-50 rounded-3xl p-10 border-2 border-slate-200 shadow-xl">
            <div className="relative">
              <div className="bg-gradient-to-br from-white via-blue-50/30 to-purple-50/30 rounded-3xl p-10 shadow-2xl border-2 border-blue-200/50 relative overflow-hidden">
                <div className="absolute inset-0 bg-gradient-to-r from-blue-400/10 to-purple-400/10 rounded-3xl blur-xl"></div>
                <div className="relative z-10">
                  <div className="flex justify-center mb-6">
                    <div className="bg-gradient-to-r from-orange-500 to-red-500 text-white px-6 py-3 rounded-full font-bold text-sm shadow-lg inline-flex items-center gap-2">
                      <span>🔥</span><span>Limited-Time Deal — 50% OFF</span>
                    </div>
                  </div>
                  <div className="text-center mb-4"><p className="text-gray-400 text-lg line-through">$100 / 365 AED</p></div>
                  <div className="text-center mb-4">
                    <div className="flex items-center justify-center gap-4 mb-3">
                      <p className="text-6xl md:text-7xl font-black text-blue-700">$50</p>
                      <span className="text-4xl text-gray-600 font-bold">/</span>
                      <p className="text-6xl md:text-7xl font-black text-blue-700">185 AED</p>
                    </div>
                    <p className="text-gray-700 font-semibold text-lg">For the Next 15 Days Only — Early Bird Offer</p>
                  </div>
                  <div className="text-center mb-8"><p className="text-gray-600 text-base">Enroll Now & Unlock Your Instant 50% Discount — Limited Seats</p></div>
                  <div className="text-center">
                    <button onClick={scrollToForm} className="w-full md:w-auto bg-gradient-to-r from-yellow-400 to-amber-500 hover:from-yellow-500 hover:to-amber-600 text-gray-900 px-12 py-5 rounded-full font-bold text-xl shadow-2xl hover:shadow-3xl transition-all hover:scale-105 inline-flex items-center justify-center gap-3">
                      Claim This Deal<ArrowRight className="w-6 h-6" />
                    </button>
                  </div>
                </div>
              </div>
            </div>
          </div>
          <div className="mt-8">
            <h3 className="text-2xl font-bold text-gray-900 text-center mb-6">Additional Group & Corporate Discounts</h3>
            <div className="bg-white rounded-2xl p-8 shadow-lg border border-gray-200">
              <div className="flex flex-col md:flex-row items-start gap-6">
                <div className="flex-shrink-0">
                  <div className="w-16 h-16 bg-slate-100 rounded-xl flex items-center justify-center">
                    <Users className="w-8 h-8 text-slate-700" />
                  </div>
                </div>
                <div className="flex-1">
                  <h4 className="text-xl font-bold text-gray-900 mb-4">Save More When You Learn Together</h4>
                  <ul className="space-y-3 mb-6">
                    <li className="flex items-start gap-3 text-gray-700"><CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" /><span>10% extra discount for group bookings (2 or more participants)</span></li>
                    <li className="flex items-start gap-3 text-gray-700"><CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" /><span>Special pricing available for corporate teams</span></li>
                    <li className="flex items-start gap-3 text-gray-700"><CheckCircle2 className="w-5 h-5 text-slate-600 flex-shrink-0 mt-0.5" /><span>Dedicated support and invoicing for companies</span></li>
                  </ul>
                  <button onClick={scrollToForm} className="border-2 border-slate-700 text-slate-700 hover:bg-slate-700 hover:text-white px-8 py-3 rounded-xl font-semibold transition-all inline-flex items-center gap-2">
                    Request Corporate Pricing<ArrowRight className="w-5 h-5" />
                  </button>
                </div>
              </div>
              <div className="mt-6 pt-6 border-t border-gray-200">
                <p className="text-sm text-gray-600 mb-2">✓ Discounts apply on top of the current 15-day promotional price.</p>
                <p className="text-sm text-gray-600">✓ For larger teams, customized offers are available.</p>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* CERTIFICATION */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-4xl">
          <div className="bg-gradient-to-br from-pink-700 to-orange-700 rounded-3xl p-12 text-center text-white border-4 border-yellow-400 shadow-2xl">
            <Award className="w-24 h-24 text-yellow-400 mx-auto mb-6" />
            <h2 className="text-4xl font-bold mb-4">Get Certified</h2>
            <p className="text-2xl text-pink-100 mb-6">
              🎓 Canva + AI Design Secrets Certificate
            </p>
            <p className="text-lg text-pink-200 mb-8">
              Issued by <strong className="text-yellow-400">Workshop Library</strong> - trusted by creators & professionals
            </p>
            <div className="inline-flex items-center gap-3 bg-white/10 backdrop-blur-sm px-8 py-4 rounded-full border-2 border-yellow-400/50">
              <Shield className="w-6 h-6 text-yellow-400" />
              <span className="font-bold">Globally Recognized Certification</span>
            </div>
          </div>
        </div>
      </section>

      {/* FAQ */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-16">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Frequently Asked Questions</h2>
          </div>
          <div className="space-y-6">
            {[
              { q: 'Do I need design experience?', a: 'No, the workshop is beginner-friendly.' },
              { q: 'Is Canva free?', a: 'Yes, everything we teach works in the free version.' },
              { q: 'Will I get templates?', a: 'Yes, a full template pack is provided.' },
              { q: 'Will this help me as a freelancer?', a: 'Absolutely — Canva is one of the fastest-growing freelance skills.' },
              { q: 'Are recordings available?', a: 'Yes, 30-day access included.' }
            ].map((faq, index) => (
              <div key={index} className="bg-gradient-to-r from-pink-50 to-orange-50 rounded-xl p-6 hover:shadow-lg transition-all border-2 border-pink-100 hover:border-pink-300">
                <h3 className="text-xl font-bold text-gray-900 mb-3 flex items-center gap-3">
                  <div className="w-8 h-8 rounded-full bg-pink-600 flex items-center justify-center text-white font-bold">
                    Q
                  </div>
                  {faq.q}
                </h3>
                <p className="text-lg text-gray-700 ml-11">{faq.a}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* REGISTRATION FORM */}
      <section id="registration-form" className="py-20 px-6 bg-gradient-to-br from-pink-700 to-orange-700">
        <div className="container mx-auto max-w-2xl">
          <div className="text-center mb-8">
            <h2 className="text-4xl font-bold text-white mb-4">Start Your Design Journey Today!</h2>
            <p className="text-xl text-pink-100">Limited seats for December 20, 2025</p>
          </div>
          <CourseLeadForm courseName="Canva + AI Design Secrets Workshop" formTitle="Register for Workshop" />
        </div>
      </section>

      {/* FOOTER */}
      <footer className="py-16 px-6 bg-gradient-to-r from-gray-900 via-gray-800 to-gray-900 text-white relative overflow-hidden">
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1451187580459-43490279c0fa?w=1920&h=1080&fit=crop&q=80"
            alt="Footer background"
            className="w-full h-full object-cover opacity-5"
          />
        </div>
        <div className="container mx-auto max-w-7xl relative z-10">
          <div className="text-center">
            <div className="flex items-center justify-center gap-3 mb-6">
              <div className="w-12 h-12 rounded-xl bg-gradient-to-br from-pink-600 to-orange-700 border-2 border-yellow-400/50 flex items-center justify-center">
                <Sparkles className="w-7 h-7 text-yellow-400" />
              </div>
              <div>
                <h3 className="font-bold text-xl">AI School for All</h3>
                <p className="text-xs text-gray-400">by Workshop Library</p>
              </div>
            </div>
            <p className="text-gray-300 mb-6">Professional AI and Creative Skills Education</p>
            <div className="flex items-center justify-center gap-3 mb-6">
              <Shield className="w-5 h-5 text-yellow-400" />
              <span className="text-sm text-gray-300 font-semibold">WorkshopLibrary Partner</span>
            </div>
            <p className="text-xs text-gray-500">
              © 2026 AI School for All. All rights reserved.
            </p>
          </div>
        </div>
      </footer>

      {/* Chatbot */}
      <Chatbot />
    </div>
  );
};

export default WorkshopCanvaDesign;
