import React from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '../components/ui/card';
import { Button } from '../components/ui/button';
import { Badge } from '../components/ui/badge';
import { Accordion, AccordionContent, AccordionItem, AccordionTrigger } from '../components/ui/accordion';
import { 
  CheckCircle2, Award, Users, BookOpen, Sparkles, TrendingUp, BarChart3, 
  Target, Globe, Shield, Star, Briefcase, Clock, Zap, Heart, ArrowRight,
  Linkedin, Twitter, Facebook, Instagram, Mail, MapPin, Phone, ExternalLink,
  ChevronRight, TrendingUp as Growth, Code, Database, Cpu, Palette, Video,
  Megaphone, FileText, Settings, Layers, Monitor, PenTool, Calendar
} from 'lucide-react';
import EnrollmentForm from '../components/EnrollmentForm';
import Chatbot from '../components/Chatbot';
import { useNavigate } from 'react-router-dom';

const LandingPage = () => {
  const navigate = useNavigate();

  const courses = [
    {
      id: 7,
      title: 'Certified Specialist in Digital Growth & Performance Marketing (CSDGPM)',
      description: 'Exemplar Global accredited specialist program. Master performance marketing, analytics, growth frameworks, and digital strategy with 100% industry expert-led training.',
      icon: <Target className="w-8 h-8 text-blue-600" />,
      duration: '6 Months',
      level: 'Specialist Level',
      image: 'https://images.unsplash.com/photo-1552664730-d307ca884978?w=800&h=500&fit=crop&q=80',
      isLive: true,
      badge: 'Now Open • Exemplar Global Certified'
    },
    {
      id: 1,
      title: 'Certified Professional in Applied Machine Learning (CPAML)',
      description: 'Master machine learning fundamentals with hands-on projects and real-world applications.',
      icon: <Sparkles className="w-8 h-8 text-gray-400" />,
      duration: '3 Months',
      level: 'Professional Certificate',
      image: 'https://images.unsplash.com/photo-1557562645-4eee56b29bc1?crop=entropy&cs=srgb&fm=jpg&ixid=M3w3NDk1ODF8MHwxfHNlYXJjaHwzfHxtYWNoaW5lJTIwbGVhcm5pbmd8ZW58MHx8fHwxNzYzMjA3MzE0fDA&ixlib=rb-4.1.0&q=85&w=800&h=500&fit=crop',
      isLive: false
    },
    {
      id: 2,
      title: 'Certified Specialist in Machine Learning & Automation (CSMLA)',
      description: 'Advanced ML automation techniques, MLOps, and deployment strategies.',
      icon: <TrendingUp className="w-8 h-8 text-gray-400" />,
      duration: '6 Months',
      level: 'Specialist Level',
      image: 'https://images.unsplash.com/photo-1761195696590-3490ea770aa1?crop=entropy&cs=srgb&fm=jpg&ixid=M3w3NDQ2NDJ8MHwxfHNlYXJjaHwxfHxhdXRvbWF0aW9uJTIwdGVjaG5vbG9neXxlbnwwfHx8fDE3NjMyMDczMTh8MA&ixlib=rb-4.1.0&q=85&w=800&h=500&fit=crop',
      isLive: false
    },
    {
      id: 3,
      title: 'Professional Certificate in Business & Data Analytics (PCBDA)',
      description: 'Transform business data into actionable insights using advanced analytics.',
      icon: <BarChart3 className="w-8 h-8 text-gray-400" />,
      duration: '3 Months',
      level: 'Professional Certificate',
      image: 'https://images.unsplash.com/photo-1608222351212-18fe0ec7b13b?crop=entropy&cs=srgb&fm=jpg&ixid=M3w3NDk1Nzl8MHwxfHNlYXJjaHwxfHxidXNpbmVzcyUyMGFuYWx5dGljc3xlbnwwfHx8fDE3NjMyMDczMjN8MA&ixlib=rb-4.1.0&q=85&w=800&h=500&fit=crop',
      isLive: false
    },
    {
      id: 4,
      title: 'Advanced Diploma in Applied Data Science & Predictive Modeling (ADADSPM)',
      description: 'Deep dive into data science, statistical modeling, and predictive analytics.',
      icon: <Target className="w-8 h-8 text-gray-400" />,
      duration: '6 Months',
      level: 'Advanced Diploma',
      image: 'https://images.unsplash.com/photo-1666875753105-c63a6f3bdc86?crop=entropy&cs=srgb&fm=jpg&ixid=M3w3NDQ2NDJ8MHwxfHNlYXJjaHwxfHxkYXRhJTIwc2NpZW5jZXxlbnwwfHx8fDE3NjMyMDczMjd8MA&ixlib=rb-4.1.0&q=85&w=800&h=500&fit=crop',
      isLive: false
    },
    {
      id: 5,
      title: 'Certified Cloud Computing Specialist (CCCS)',
      description: 'Master cloud platforms (AWS, Azure, GCP) and cloud-native development.',
      icon: <Globe className="w-8 h-8 text-gray-400" />,
      duration: '3 Months',
      level: 'Professional Certificate',
      image: 'https://images.pexels.com/photos/1181354/pexels-photo-1181354.jpeg?auto=compress&cs=tinysrgb&w=800&h=500&fit=crop',
      isLive: false
    },
    {
      id: 6,
      title: 'Professional Certificate in Digital Marketing Strategies (PCDMS)',
      description: 'Learn comprehensive digital marketing including SEO, social media, and analytics.',
      icon: <TrendingUp className="w-8 h-8 text-gray-400" />,
      duration: '3 Months',
      level: 'Professional Certificate',
      image: 'https://workshoplibrary.com/public/storage/freedemo/17485526331118228328.jpg',
      isLive: false
    }
  ];

  const workshops = [
    {
      id: 1,
      title: 'Photo Editing with AI',
      description: 'Master AI-powered photo editing tools and techniques',
      duration: '4 hours',
      icon: <Palette className="w-8 h-8" />,
      image: 'https://images.unsplash.com/photo-1561070791-2526d30994b5?w=400&h=300&fit=crop&q=80',
      color: 'from-pink-400 to-rose-500'
    },
    {
      id: 2,
      title: 'Video Production & Editing with AI',
      description: 'Create professional videos using AI tools',
      duration: '5 hours',
      icon: <Video className="w-8 h-8" />,
      image: 'https://images.unsplash.com/photo-1574717024653-61fd2cf4d44d?w=400&h=300&fit=crop&q=80',
      color: 'from-purple-400 to-indigo-500'
    },
    {
      id: 3,
      title: 'AI-Powered Social Media Booster',
      description: 'Amplify your social presence with AI automation',
      duration: '3 hours',
      icon: <Megaphone className="w-8 h-8" />,
      image: 'https://images.unsplash.com/photo-1683721003111-070bcc053d8b?w=400&h=300&fit=crop&q=80',
      color: 'from-blue-400 to-cyan-500'
    },
    {
      id: 4,
      title: 'LinkedIn Automation using AI',
      description: 'Automate LinkedIn outreach and engagement',
      duration: '4 hours',
      icon: <Linkedin className="w-8 h-8" />,
      image: 'https://images.unsplash.com/photo-1611944212129-29977ae1398c?w=400&h=300&fit=crop&q=80',
      color: 'from-blue-500 to-blue-700'
    },
    {
      id: 5,
      title: 'Prompt Engineering for Beginners',
      description: 'Learn to write effective AI prompts',
      duration: '3 hours',
      icon: <FileText className="w-8 h-8" />,
      image: 'https://images.unsplash.com/photo-1677442136019-21780ecad995?w=400&h=300&fit=crop&q=80',
      color: 'from-green-400 to-emerald-500'
    },
    {
      id: 6,
      title: 'Canva + AI Design Secrets',
      description: 'Design like a pro with Canva AI features',
      duration: '4 hours',
      icon: <Layers className="w-8 h-8" />,
      image: 'https://images.unsplash.com/photo-1626785774573-4b799315345d?w=400&h=300&fit=crop&q=80',
      color: 'from-violet-400 to-purple-500'
    },
    {
      id: 7,
      title: 'AI Tools for Freelancers',
      description: 'Boost productivity with AI automation',
      duration: '5 hours',
      icon: <Briefcase className="w-8 h-8" />,
      image: 'https://images.unsplash.com/photo-1553877522-43269d4ea984?w=400&h=300&fit=crop&q=80',
      color: 'from-orange-400 to-red-500'
    },
    {
      id: 8,
      title: 'Digital Ads Auto-Optimization with AI',
      description: 'Optimize campaigns with AI-powered insights',
      duration: '6 hours',
      icon: <Target className="w-8 h-8" />,
      image: 'https://images.unsplash.com/photo-1460925895917-afdab827c52f?w=400&h=300&fit=crop&q=80',
      color: 'from-teal-400 to-cyan-500'
    },
    {
      id: 9,
      title: 'Content Creation Mastery using AI',
      description: 'Generate high-quality content at scale',
      duration: '5 hours',
      icon: <PenTool className="w-8 h-8" />,
      image: 'https://images.unsplash.com/photo-1542435503-956c469947f6?w=400&h=300&fit=crop&q=80',
      color: 'from-amber-400 to-yellow-500'
    },
    {
      id: 10,
      title: 'AI-Powered Resume + Portfolio Builder',
      description: 'Build stunning portfolios with AI assistance',
      duration: '4 hours',
      icon: <Monitor className="w-8 h-8" />,
      image: 'https://images.unsplash.com/photo-1586281380349-632531db7ed4?w=400&h=300&fit=crop&q=80',
      color: 'from-indigo-400 to-blue-500'
    }
  ];

  const testimonials = [
    {
      name: 'Sarah Johnson',
      role: 'Digital Marketing Manager',
      company: 'Tech Startup',
      rating: 5,
      quote: 'The CSDGPM program transformed my career. The practical approach and industry insights were invaluable. Highly recommend!'
    },
    {
      name: 'Michael Chen',
      role: 'Performance Marketing Lead',
      company: 'E-commerce Brand',
      rating: 5,
      quote: 'Best investment in my professional development. The curriculum is comprehensive and immediately applicable to real-world scenarios.'
    },
    {
      name: 'Priya Sharma',
      role: 'Growth Marketing Specialist',
      company: 'SaaS Company',
      rating: 5,
      quote: 'Exemplar Global certification opened doors to senior roles. The quality of training and support exceeded my expectations.'
    }
  ];

  const industryFacts = [
    {
      icon: <BarChart3 className="w-12 h-12 text-blue-600" />,
      stat: '75%',
      title: 'Job Transformation',
      description: 'Professionals transition to higher-paying roles within 12 months',
      bgColor: 'bg-blue-50'
    },
    {
      icon: <TrendingUp className="w-12 h-12 text-green-600" />,
      stat: '$45K+',
      title: 'Average Salary Increase',
      description: 'Average annual income boost after certification completion',
      bgColor: 'bg-green-50'
    },
    {
      icon: <Target className="w-12 h-12 text-purple-600" />,
      stat: '93%',
      title: 'Career Advancement',
      description: 'Alumni achieve promotions or career pivots successfully',
      bgColor: 'bg-purple-50'
    },
    {
      icon: <Globe className="w-12 h-12 text-orange-600" />,
      stat: '150+',
      title: 'Global Hiring Partners',
      description: 'Companies actively hiring our certified professionals',
      bgColor: 'bg-orange-50'
    }
  ];

  const faqs = [
    {
      icon: <CheckCircle2 className="w-5 h-5 text-blue-600" />,
      question: 'What makes AI School for All different from other training providers?',
      answer: 'We offer Exemplar Global accredited programs with 100% industry expert-led training. Our courses combine live sessions, hands-on labs, and real-world projects with lifetime access to recorded content.'
    },
    {
      icon: <Award className="w-5 h-5 text-blue-600" />,
      question: 'Are the certifications globally recognized?',
      answer: 'Yes! Our flagship CSDGPM program is accredited by Exemplar Global, making it internationally recognized. We also prepare you for additional certifications like Google Ads and Meta Blueprint.'
    },
    {
      icon: <Clock className="w-5 h-5 text-blue-600" />,
      question: 'What is the time commitment required?',
      answer: 'Programs range from 3-6 months with 6-8 hours per week commitment. All live sessions are recorded, so you can learn at your own pace while still benefiting from real-time interactions.'
    },
    {
      icon: <Users className="w-5 h-5 text-blue-600" />,
      question: 'Do you provide career support?',
      answer: 'Yes! We provide career guidance, portfolio building support, resume reviews, and access to our network of 150+ hiring partners. Many of our alumni secure roles through our placement assistance.'
    },
    {
      icon: <Briefcase className="w-5 h-5 text-blue-600" />,
      question: 'Who should enroll in these programs?',
      answer: 'Our programs are designed for working professionals, fresh graduates, career switchers, and business owners looking to upskill in AI, data science, marketing, or cloud technologies.'
    },
    {
      icon: <Shield className="w-5 h-5 text-blue-600" />,
      question: 'What is your refund policy?',
      answer: 'We offer a 7-day money-back guarantee from the program start date. If you\'re not satisfied with the quality of training, we\'ll provide a full refund, no questions asked.'
    }
  ];

  return (
    <div className="min-h-screen bg-white">
      {/* Special Offer Banner - SUPER PROMINENT - Fixed at Very Top */}
      <div 
        onClick={() => navigate('/csdgpm-landing')}
        className="fixed top-0 left-0 right-0 bg-gradient-to-r from-red-600 via-orange-500 to-red-600 text-white py-4 px-4 text-center font-black cursor-pointer hover:scale-[1.01] transition-all shadow-2xl z-[100] border-b-4 border-yellow-400"
        style={{ animation: 'pulse 2s cubic-bezier(0.4, 0, 0.6, 1) infinite' }}
      >
        <div className="container mx-auto">
          <div className="flex flex-col sm:flex-row items-center justify-center gap-2 sm:gap-4">
            <span className="text-lg sm:text-xl font-black tracking-tight">
              🔥 MASSIVE 50% OFF - DIGITAL MARKETING COURSE 🔥
            </span>
            <span className="bg-white text-red-600 px-5 py-2 rounded-full font-black text-sm sm:text-base shadow-xl animate-bounce">
              👉 CLICK HERE NOW! 👈
            </span>
          </div>
          <p className="text-xs mt-1 font-semibold opacity-90">
            Limited Seats • Offer Expires Soon • Book Your Slot Now!
          </p>
        </div>
      </div>

      {/* Navigation Menu Strip - Transparent with Rounded Corners - Below Banner */}
      <nav className="fixed top-[120px] left-1/2 transform -translate-x-1/2 z-50 w-[95%] max-w-7xl">
        <div className="bg-black/30 backdrop-blur-lg rounded-full border border-white/20 px-8 py-3 shadow-2xl">
          <div className="flex items-center justify-between">
            {/* Logo/Brand */}
            <div className="text-white font-bold text-lg">AI School for All</div>
            
            {/* Navigation Links */}
            <div className="hidden md:flex items-center gap-6">
              <a href="#home" className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Home</a>
              <a href="#about" className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">About</a>
              <a href="#courses" className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Programs</a>
              <a href="#workshops" className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Workshops</a>
              <a href="#testimonials" className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Testimonials</a>
              <a href="#faq" className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">FAQ</a>
              <button 
                onClick={() => document.getElementById('enrollment-form')?.scrollIntoView({ behavior: 'smooth' })}
                className="bg-yellow-400 hover:bg-yellow-500 text-gray-900 px-6 py-2 rounded-full font-bold transition-all text-sm"
              >
                Enroll Now
              </button>
            </div>
          </div>
        </div>
      </nav>

      {/* Floating Side Banner - Extra Visibility */}
      <div 
        onClick={() => navigate('/csdgpm-landing')}
        className="fixed right-0 top-1/3 bg-gradient-to-b from-red-600 to-orange-600 text-white px-4 py-6 rounded-l-2xl shadow-2xl cursor-pointer hover:scale-110 transition-all z-[90] hidden lg:block"
        style={{ writingMode: 'vertical-rl', animation: 'pulse 2s cubic-bezier(0.4, 0, 0.6, 1) infinite' }}
      >
        <span className="font-black text-lg tracking-wider">🔥 50% OFF 🔥</span>
      </div>

      {/* SECTION 1: HERO WITH ROBOT BACKGROUND */}
      <section id="home" className="relative min-h-screen flex items-center overflow-hidden bg-black pt-48">
        {/* Full-Width Robot Background - Second Image - NO TINT, NO BLUR */}
        <div className="absolute inset-0">
          <img 
            src="https://customer-assets.emergentagent.com/job_academy-revamp-2/artifacts/75drn3yc_Untitled%20design%20%284%29.jpg"
            alt="AI Robot"
            className="w-full h-full object-cover object-right"
          />
          {/* Subtle dark gradient on LEFT SIDE ONLY for text readability */}
          <div className="absolute inset-0 bg-gradient-to-r from-black/90 via-black/60 to-transparent"></div>
        </div>

        <div className="container mx-auto px-8 py-20 relative z-10">
          <div className="max-w-2xl">
            {/* Main Heading */}
            <h1 className="text-6xl md:text-7xl font-black text-white mb-8 leading-tight">
              Transform Your Career with
              <span className="block mt-2">
                AI School for All
              </span>
            </h1>
            
            {/* Powered by Workshop Library - Text + Logo (No Box) */}
            <div className="flex items-center gap-4 mb-8">
              <span className="text-lg font-semibold text-white">Powered by Workshop Library</span>
              <span className="text-white text-2xl">—</span>
              <img 
                src="https://customer-assets.emergentagent.com/job_eduai-portal-6/artifacts/zrncz5al_Untitled%20%281000%20x%20500%20px%29%20%281%29.jpg"
                alt="Workshop Library"
                className="h-10 object-contain"
              />
            </div>
            
            {/* Sub-heading */}
            <p className="text-xl md:text-2xl text-white mb-10 leading-relaxed font-medium">
              Globally Recognized Certifications in AI, Data Science, Marketing & Cloud Computing
            </p>

            {/* CTA Buttons */}
            <div className="flex flex-col sm:flex-row gap-4 mb-12">
              <button 
                onClick={() => navigate('/csdgpm-landing')}
                className="px-8 py-4 rounded-xl font-bold text-lg shadow-xl hover:scale-105 transition-all inline-flex items-center justify-center gap-2 animate-pulse"
                style={{ backgroundColor: '#FFC727', color: '#000000' }}
              >
                🔥 Special Offer Landing Page - 50% OFF
                <ArrowRight className="w-5 h-5" />
              </button>
              <button 
                onClick={() => navigate('/courses/csdgpm')}
                className="bg-transparent border-2 border-white text-white px-8 py-4 rounded-xl font-bold text-lg hover:bg-white/10 transition-all inline-flex items-center justify-center gap-2"
              >
                View Course Details
                <ArrowRight className="w-5 h-5" />
              </button>
            </div>

            {/* Trust Badges Row */}
            <div className="flex flex-wrap items-center gap-4">
              <div className="flex items-center gap-2 px-5 py-3 rounded-xl border border-white/30 bg-white/5 backdrop-blur-sm">
                <Shield className="w-5 h-5 text-white" />
                <span className="text-white font-semibold text-sm">Exemplar Global Accredited</span>
              </div>
              <div className="flex items-center gap-2 px-5 py-3 rounded-xl border border-white/30 bg-white/5 backdrop-blur-sm">
                <Award className="w-5 h-5 text-white" />
                <span className="text-white font-semibold text-sm">Since 2011</span>
              </div>
              <div className="flex items-center gap-2 px-5 py-3 rounded-xl border border-white/30 bg-white/5 backdrop-blur-sm">
                <Users className="w-5 h-5 text-white" />
                <span className="text-white font-semibold text-sm">30,000+ Students Trained</span>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* SECTION 2: UPCOMING FLAGSHIP COHORT - LEADERSHIP TRAINING STYLE */}
      <section className="relative py-16 px-6" style={{ backgroundColor: '#FFFFFF' }}>
        {/* Subtle Decorative Circular Lines */}
        <div className="absolute inset-0 opacity-3 pointer-events-none">
          <div className="absolute top-1/3 left-1/4 w-64 h-64 rounded-full border border-purple-500/20"></div>
          <div className="absolute bottom-1/3 right-1/4 w-48 h-48 rounded-full border border-blue-500/20"></div>
        </div>

        <div className="container mx-auto max-w-6xl relative z-10">
          {/* Central Light Card */}
          <div 
            className="rounded-[32px] p-8 lg:p-10 relative"
            style={{ 
              backgroundColor: '#F8F9FA',
              boxShadow: '0 10px 40px rgba(0, 0, 0, 0.08)'
            }}
          >
            {/* Big Bold Heading at Top - Black NO UNDERLINE */}
            <div className="text-center mb-10">
              <h2 className="text-5xl lg:text-6xl font-black text-gray-900">
                Our Upcoming Legacy Cohort
              </h2>
            </div>

            <div className="grid lg:grid-cols-2 gap-8 items-start">
              {/* LEFT CONTENT COLUMN */}
              <div className="flex flex-col justify-center">
                {/* Program Title */}
                <h3 className="text-3xl lg:text-4xl font-black text-gray-900 mb-5 leading-tight">
                  Certified Specialist in Digital Growth & Performance Marketing (CSDGPM)
                </h3>

                {/* Subhead */}
                <p className="text-base text-gray-600 mb-5 font-medium">
                  Specialist-level program — Industry expert-led · Live classes + Labs + Projects
                </p>

                {/* Five Info Boxes - Bigger Size */}
                <div className="grid grid-cols-2 gap-4 mb-6">
                  <div 
                    className="flex items-center gap-3 px-5 py-4 rounded-xl"
                    style={{
                      backgroundColor: 'rgba(0, 0, 0, 0.03)',
                      border: '1px solid rgba(26, 77, 255, 0.3)'
                    }}
                  >
                    <Calendar className="w-6 h-6" style={{ color: '#1A4DFF' }} />
                    <div>
                      <p className="text-xs text-gray-500 mb-1">Next Cohort</p>
                      <p className="text-sm font-bold text-gray-900">January 2025</p>
                    </div>
                  </div>

                  <div 
                    className="flex items-center gap-3 px-5 py-4 rounded-xl relative"
                    style={{
                      backgroundColor: 'rgba(0, 0, 0, 0.03)',
                      border: '1px solid rgba(26, 77, 255, 0.3)'
                    }}
                  >
                    <div className="absolute top-2 right-2 w-2 h-2 bg-red-500 rounded-full animate-pulse"></div>
                    <Users className="w-6 h-6" style={{ color: '#1A4DFF' }} />
                    <div>
                      <p className="text-xs text-gray-500 mb-1">Seats Left</p>
                      <p className="text-sm font-bold text-gray-900">40 Only</p>
                    </div>
                  </div>

                  <div 
                    className="flex items-center gap-3 px-5 py-4 rounded-xl"
                    style={{
                      backgroundColor: 'rgba(0, 0, 0, 0.03)',
                      border: '1px solid rgba(26, 77, 255, 0.3)'
                    }}
                  >
                    <Clock className="w-6 h-6" style={{ color: '#1A4DFF' }} />
                    <div>
                      <p className="text-xs text-gray-500 mb-1">Duration</p>
                      <p className="text-sm font-bold text-gray-900">6 Months</p>
                    </div>
                  </div>

                  <div 
                    className="flex items-center gap-3 px-5 py-4 rounded-xl"
                    style={{
                      backgroundColor: 'rgba(0, 0, 0, 0.03)',
                      border: '1px solid rgba(26, 77, 255, 0.3)'
                    }}
                  >
                    <BookOpen className="w-6 h-6" style={{ color: '#1A4DFF' }} />
                    <div>
                      <p className="text-xs text-gray-500 mb-1">Training</p>
                      <p className="text-sm font-bold text-gray-900">150 Hours</p>
                    </div>
                  </div>

                  {/* NEW - Last Date of Admission - Highlighted */}
                  <div 
                    className="col-span-2 flex items-center justify-center gap-3 px-5 py-4 rounded-xl"
                    style={{
                      backgroundColor: 'rgba(255, 199, 39, 0.15)',
                      border: '2px solid #FFC727',
                      boxShadow: '0 0 15px rgba(255, 199, 39, 0.3)'
                    }}
                  >
                    <Star className="w-6 h-6" style={{ color: '#FFC727' }} />
                    <div className="text-center">
                      <p className="text-xs text-gray-600 mb-1">Last Date of Admission</p>
                      <p className="text-base font-black text-gray-900">15th Dec</p>
                    </div>
                  </div>
                </div>

                {/* CTA Buttons */}
                <div className="flex flex-col sm:flex-row gap-3">
                  <button 
                    onClick={() => navigate('/courses/csdgpm')}
                    className="px-6 py-3 rounded-xl font-bold text-base transition-all hover:scale-105 hover:shadow-lg inline-flex items-center justify-center gap-2"
                    style={{ backgroundColor: '#FFC727', color: '#000000' }}
                  >
                    View Course Details
                    <ArrowRight className="w-4 h-4" />
                  </button>
                  <button 
                    onClick={() => document.getElementById('enrollment-form')?.scrollIntoView({ behavior: 'smooth' })}
                    className="px-6 py-3 rounded-xl font-bold text-base transition-all hover:bg-gray-800"
                    style={{ backgroundColor: 'rgba(0, 0, 0, 0.8)', color: '#FFFFFF', border: '1px solid rgba(0, 0, 0, 0.9)' }}
                  >
                    Enroll Now
                  </button>
                </div>
              </div>

              {/* RIGHT CONTENT COLUMN - Big Image Without Border or Shape */}
              <div className="hidden lg:block h-full">
                <img 
                  src="https://customer-assets.emergentagent.com/job_academy-revamp-2/artifacts/eib8maoq_Untitled%20design%20%285%29.jpg"
                  alt="Digital Marketing Professional"
                  className="w-full h-full object-cover rounded-2xl"
                  style={{ minHeight: '550px', maxHeight: '650px' }}
                />
              </div>
            </div>

            {/* Powered by Workshop Library + Badges - Center Bottom in One Long Line */}
            <div className="text-center mt-8 pt-6 border-t border-gray-200">
              <div className="flex flex-wrap items-center justify-center gap-4">
                {/* Powered by Workshop Library */}
                <a 
                  href="https://workshoplibrary.com" 
                  target="_blank" 
                  rel="noopener noreferrer"
                  className="inline-flex items-center gap-3 hover:opacity-70 transition-opacity"
                >
                  <span className="text-sm font-bold text-gray-600">Powered by</span>
                  <img 
                    src="https://customer-assets.emergentagent.com/job_eduai-portal-6/artifacts/zrncz5al_Untitled%20%281000%20x%20500%20px%29%20%281%29.jpg"
                    alt="Workshop Library"
                    className="h-6 object-contain"
                  />
                </a>

                {/* Separator */}
                <div className="w-px h-8 bg-gray-300"></div>

                {/* Exemplar Global Badge */}
                <div 
                  className="flex items-center gap-2 px-5 py-2 rounded-full"
                  style={{
                    backgroundColor: 'rgba(107, 91, 255, 0.1)',
                    border: '2px solid #6B5BFF',
                    boxShadow: '0 0 10px rgba(107, 91, 255, 0.2)'
                  }}
                >
                  <img 
                    src="https://customer-assets.emergentagent.com/job_eduai-portal-6/artifacts/qvi4oewl_exempler%20logo.webp"
                    alt="Exemplar Global"
                    className="h-5 object-contain"
                  />
                  <span className="text-gray-900 font-bold text-xs">Exemplar Global Certified</span>
                </div>

                {/* Google Ads Badge */}
                <div 
                  className="flex items-center gap-2 px-5 py-2 rounded-full"
                  style={{
                    backgroundColor: 'rgba(26, 77, 255, 0.1)',
                    border: '2px solid #1A4DFF',
                    boxShadow: '0 0 10px rgba(26, 77, 255, 0.2)'
                  }}
                >
                  <div className="w-5 h-5 rounded-full bg-blue-600 flex items-center justify-center">
                    <span className="text-white font-bold text-[10px]">G</span>
                  </div>
                  <span className="text-gray-900 font-bold text-xs">Google Ads Certification Ready</span>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>
      {/* SECTION 3: OUR PROGRAMS (LIVE vs COMING SOON) */}
      <section id="courses" className="py-32 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-7xl">
          <div className="text-center mb-20">
            <h2 className="text-5xl md:text-6xl font-bold text-gray-900 mb-6">Our Programs</h2>
            <p className="text-2xl text-gray-600 max-w-3xl mx-auto">Professional certifications designed to accelerate your career in high-demand fields</p>
            <button 
              onClick={() => document.getElementById('enrollment-form')?.scrollIntoView({ behavior: 'smooth' })}
              className="mt-6 bg-gradient-to-r from-blue-600 to-blue-700 hover:from-blue-700 hover:to-blue-800 text-white px-8 py-3 rounded-xl font-bold text-lg shadow-lg hover:scale-105 transition-all inline-flex items-center gap-2"
            >
              Enroll in a Program
              <ArrowRight className="w-5 h-5" />
            </button>
          </div>

          {/* GLOBAL HEADER - Exemplar Global Authorization */}
          <div className="mb-12 py-6 px-10 bg-gradient-to-r from-blue-50 to-indigo-50 rounded-2xl border-2 border-blue-200 shadow-lg max-w-5xl mx-auto">
            <div className="flex items-center justify-center gap-4 flex-wrap">
              <h3 className="text-2xl font-bold text-gray-900">Authorised Training Certifications through</h3>
              <div className="flex items-center">
                <img 
                  src="https://customer-assets.emergentagent.com/job_ai-workshops/artifacts/sp9brvsf_exempler%20logo.webp" 
                  alt="Exemplar Global" 
                  className="h-16 object-contain"
                />
              </div>
            </div>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {courses.map((course) => (
              <div 
                key={course.id}
                className={`group relative rounded-3xl overflow-hidden transition-all duration-300 hover:scale-105 hover:shadow-2xl shadow-xl ${
                  course.isLive 
                    ? 'ring-4 ring-yellow-400/50' 
                    : 'ring-2 ring-blue-200/50'
                }`}
              >
                <div className={`bg-white border-2 ${course.isLive ? 'border-yellow-400 shadow-yellow-200/50' : 'border-gray-200'} rounded-3xl overflow-hidden h-full flex flex-col`}>
                  {/* Limited Time 50% OFF Corner Badge - Only for Live Course */}
                  {course.isLive && (
                    <div className="absolute top-0 right-0 z-20">
                      <div className="relative">
                        <div className="bg-gradient-to-br from-red-500 to-orange-500 text-white px-6 py-3 rounded-bl-3xl shadow-2xl">
                          <div className="flex flex-col items-center">
                            <span className="text-xs font-bold uppercase tracking-wider">Limited Time</span>
                            <span className="text-2xl font-black leading-tight">50% OFF</span>
                          </div>
                        </div>
                        <div className="absolute top-0 right-0 w-0 h-0 border-t-[20px] border-t-red-600 border-l-[20px] border-l-transparent"></div>
                      </div>
                    </div>
                  )}

                  {course.isLive && (
                    <div className="absolute inset-0 bg-gradient-to-r from-yellow-400/20 to-orange-400/20 animate-pulse"></div>
                  )}

                  <div className="h-48 overflow-hidden relative">
                    <img 
                      src={course.image} 
                      alt={course.title}
                      className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
                    />
                    <div className="absolute inset-0 bg-gradient-to-t from-gray-900/60 to-transparent"></div>
                  </div>

                  <div className="p-6 flex-1 flex flex-col">
                    {/* Status & Certification Badges - Above Title */}
                    <div className="flex items-center justify-between mb-4 pb-3 border-b border-gray-200">
                      <div className="flex items-center gap-2">
                        {course.isLive ? (
                          <div className="bg-gradient-to-r from-yellow-400 to-yellow-500 text-gray-900 px-3 py-1.5 rounded-full font-bold text-xs shadow-md flex items-center gap-1.5">
                            <Sparkles className="w-3 h-3" />
                            <span>Enrollment Open</span>
                          </div>
                        ) : (
                          <div className="bg-gradient-to-r from-blue-500 to-indigo-600 text-white px-3 py-1.5 rounded-full font-semibold text-xs shadow-md flex items-center gap-1.5">
                            <span>🚀</span>
                            <span>Launching Soon</span>
                          </div>
                        )}
                      </div>
                      <div className="bg-white px-2 py-1 rounded-md border border-blue-200">
                        <img 
                          src="https://customer-assets.emergentagent.com/job_ai-workshops/artifacts/sp9brvsf_exempler%20logo.webp" 
                          alt="Exemplar" 
                          className="h-5 object-contain"
                        />
                      </div>
                    </div>
                    
                    {/* Level & Duration Badges */}
                    <div className="flex items-center gap-2 mb-3">
                      <Badge className="bg-blue-100 text-blue-700 hover:bg-blue-100 text-sm px-3 py-1.5 font-bold">{course.level}</Badge>
                      <Badge className="bg-gray-100 text-gray-700 hover:bg-gray-100 text-sm px-3 py-1.5 font-bold">{course.duration}</Badge>
                    </div>
                    
                    {/* Course Title */}
                    <h3 className="text-xl font-bold text-gray-900 mb-3 leading-tight">{course.title}</h3>
                    
                    {/* Commencement Date for Live Course */}
                    {course.isLive && (
                      <div className="mb-3 flex items-center gap-2">
                        <Calendar className="w-4 h-4 text-blue-600" />
                        <p className="text-sm font-semibold text-gray-700">
                          <span className="text-gray-500">Commencing on:</span> <span className="text-blue-700">December 15, 2025</span>
                        </p>
                      </div>
                    )}
                    
                    <p className="text-gray-600 mb-6 leading-relaxed flex-1">{course.description}</p>
                    
                    {course.isLive ? (
                      <>
                        <button 
                          onClick={() => navigate('/courses/csdgpm')}
                          className="w-full bg-gradient-to-r from-blue-600 to-blue-700 hover:from-blue-700 hover:to-blue-800 text-white py-3 rounded-xl font-bold transition-all flex items-center justify-center gap-2 shadow-lg hover:shadow-xl"
                        >
                          View Details
                          <ArrowRight className="w-4 h-4" />
                        </button>
                        
                        {/* Ticker for Live Course */}
                        <div className="mt-4 overflow-hidden bg-gradient-to-r from-yellow-50 via-amber-50 to-yellow-50 rounded-lg border border-yellow-200 py-2.5">
                          <div className="flex animate-scroll whitespace-nowrap">
                            <span className="inline-flex items-center gap-2 text-sm font-bold text-gray-800 px-4">
                              ⭐ 100+ Students Enrolled
                            </span>
                            <span className="inline-flex items-center gap-2 text-sm font-bold text-gray-800 px-4">
                              🔥 Limited Seats Available
                            </span>
                            <span className="inline-flex items-center gap-2 text-sm font-bold text-gray-800 px-4">
                              📅 Next Batch: December 15, 2025
                            </span>
                            <span className="inline-flex items-center gap-2 text-sm font-bold text-gray-800 px-4">
                              🎓 Industry Expert Led
                            </span>
                            <span className="inline-flex items-center gap-2 text-sm font-bold text-gray-800 px-4">
                              ⭐ 100+ Students Enrolled
                            </span>
                            <span className="inline-flex items-center gap-2 text-sm font-bold text-gray-800 px-4">
                              🔥 Limited Seats Available
                            </span>
                          </div>
                        </div>
                      </>
                    ) : (
                      <button 
                        onClick={() => document.getElementById('enrollment-form')?.scrollIntoView({ behavior: 'smooth' })}
                        className="w-full border-2 border-blue-600 text-blue-600 hover:bg-blue-50 py-3 rounded-xl font-bold transition-all flex items-center justify-center gap-2 shadow-sm hover:shadow-md"
                      >
                        Enquire Now
                        <ArrowRight className="w-4 h-4" />
                      </button>
                    )}
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* SECTION 4: UPCOMING SHORT-TERM AI WORKSHOPS (NEW) */}
      <section id="workshops" className="relative py-32 px-6 bg-gradient-to-b from-blue-50 to-white overflow-hidden">
        {/* Curved Top Divider */}
        <div className="absolute top-0 left-0 right-0">
          <svg viewBox="0 0 1440 120" className="w-full h-24 fill-white">
            <path d="M0,64 C240,120 480,120 720,96 C960,72 1200,24 1440,48 L1440,0 L0,0 Z"></path>
          </svg>
        </div>

        {/* Floating Shapes */}
        <div className="absolute top-20 left-10 w-72 h-72 bg-yellow-200 rounded-full blur-3xl opacity-20"></div>
        <div className="absolute bottom-20 right-10 w-96 h-96 bg-blue-200 rounded-full blur-3xl opacity-20"></div>

        <div className="container mx-auto max-w-7xl relative z-10 pt-16">
          <div className="text-center mb-20">
            <h2 className="text-5xl md:text-6xl font-bold text-gray-900 mb-6">Upcoming Short-Term AI Workshops</h2>
            <p className="text-2xl text-gray-600 max-w-3xl mx-auto">Enhance your skills with fast, practical, career-ready micro courses</p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
            {workshops.map((workshop) => (
              <div 
                key={workshop.id}
                className="group bg-white rounded-3xl overflow-hidden shadow-lg hover:shadow-2xl transition-all duration-300 border-2 border-gray-100"
              >
                {/* Workshop Image */}
                <div className="relative h-48 overflow-hidden">
                  <img 
                    src={workshop.image}
                    alt={workshop.title}
                    className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
                  />
                  <div className={`absolute inset-0 bg-gradient-to-t ${workshop.color} opacity-40`}></div>
                  
                  {/* 50% OFF Corner Badge */}
                  <div className="absolute top-0 right-0 z-20">
                    <div className="relative">
                      <div className="bg-gradient-to-br from-red-500 to-orange-500 text-white px-5 py-2.5 rounded-bl-2xl shadow-xl">
                        <div className="flex flex-col items-center">
                          <span className="text-[10px] font-bold uppercase tracking-wide">Limited Time</span>
                          <span className="text-lg font-black leading-tight">50% OFF</span>
                        </div>
                      </div>
                      <div className="absolute top-0 right-0 w-0 h-0 border-t-[15px] border-t-red-600 border-l-[15px] border-l-transparent"></div>
                    </div>
                  </div>
                </div>

                {/* Content */}
                <div className="p-6">
                  <div className="flex items-center justify-between mb-3">
                    <Badge className="bg-blue-50 text-blue-700 text-xs">{workshop.duration}</Badge>
                  </div>
                  <h3 className="text-lg font-bold text-gray-900 mb-2 leading-tight">{workshop.title}</h3>
                  <p className="text-sm text-gray-600 mb-4 leading-relaxed">{workshop.description}</p>
                  
                  {/* Premium Price & Date Card */}
                  <div className="mb-4 bg-white rounded-xl border-2 border-gray-200 shadow-sm overflow-hidden">
                    {/* Price Section */}
                    <div className="p-4 bg-gradient-to-r from-blue-600 to-indigo-600 text-white text-center">
                      <p className="text-xs uppercase tracking-wide opacity-80 mb-2">Workshop Fee</p>
                      <div className="flex items-center justify-center gap-3 mb-2">
                        <span className="text-xl line-through opacity-70">$100</span>
                        <span className="text-4xl font-black">$50</span>
                      </div>
                      <p className="text-xs opacity-90 font-semibold">Save $50 Today</p>
                    </div>
                    
                    {/* Date Section */}
                    <div className="p-4 bg-gray-50 text-center">
                      <div className="flex items-center justify-center gap-2 mb-1">
                        <Calendar className="w-4 h-4 text-blue-600" />
                        <span className="text-xs font-semibold text-gray-600 uppercase tracking-wide">Next Batch</span>
                      </div>
                      <p className="text-base font-bold text-gray-900">{
                        workshop.id === 1 ? 'Nov 30, 2025' :
                        workshop.id === 2 ? 'Dec 6, 2025' :
                        workshop.id === 3 ? 'Dec 7, 2025' :
                        workshop.id === 4 ? 'Dec 13, 2025' :
                        workshop.id === 5 ? 'Dec 14, 2025' :
                        workshop.id === 6 ? 'Dec 20, 2025' :
                        workshop.id === 7 ? 'Dec 21, 2025' :
                        workshop.id === 8 ? 'Nov 29, 2025' :
                        workshop.id === 9 ? 'Nov 30, 2025' :
                        'Dec 13, 2025'
                      }</p>
                    </div>
                    
                    {/* Promo Banner */}
                    <div className="bg-gradient-to-r from-orange-500 to-red-500 text-white text-center py-2.5">
                      <p className="text-xs font-bold tracking-wide">🔥 50% OFF - LIMITED TIME</p>
                    </div>
                  </div>
                  
                  {/* Button */}
                  {workshop.id === 1 ? (
                    <button 
                      onClick={() => navigate('/workshops/photo-editing-ai')}
                      className="w-full bg-gradient-to-r from-orange-500 to-amber-500 hover:from-orange-600 hover:to-amber-600 text-white py-3 rounded-xl font-bold text-sm shadow-lg hover:shadow-xl transition-all"
                    >
                      View Workshop Details
                    </button>
                  ) : workshop.id === 2 ? (
                    <button 
                      onClick={() => navigate('/workshops/video-editing-reels')}
                      className="w-full bg-gradient-to-r from-purple-500 to-pink-500 hover:from-purple-600 hover:to-pink-600 text-white py-3 rounded-xl font-bold text-sm shadow-lg hover:shadow-xl transition-all"
                    >
                      View Workshop Details
                    </button>
                  ) : workshop.id === 3 ? (
                    <button 
                      onClick={() => navigate('/workshops/social-media-booster')}
                      className="w-full bg-gradient-to-r from-blue-500 to-cyan-500 hover:from-blue-600 hover:to-cyan-600 text-white py-3 rounded-xl font-bold text-sm shadow-lg hover:shadow-xl transition-all"
                    >
                      View Workshop Details
                    </button>
                  ) : workshop.id === 4 ? (
                    <button 
                      onClick={() => navigate('/workshops/linkedin-personal-branding')}
                      className="w-full bg-gradient-to-r from-blue-700 to-gray-800 hover:from-blue-800 hover:to-gray-900 text-white py-3 rounded-xl font-bold text-sm shadow-lg hover:shadow-xl transition-all"
                    >
                      View Workshop Details
                    </button>
                  ) : workshop.id === 5 ? (
                    <button 
                      onClick={() => navigate('/workshops/prompt-engineering')}
                      className="w-full bg-gradient-to-r from-emerald-500 to-purple-600 hover:from-emerald-600 hover:to-purple-700 text-white py-3 rounded-xl font-bold text-sm shadow-lg hover:shadow-xl transition-all"
                    >
                      View Workshop Details
                    </button>
                  ) : workshop.id === 6 ? (
                    <button 
                      onClick={() => navigate('/workshops/canva-design-secrets')}
                      className="w-full bg-gradient-to-r from-pink-500 to-orange-500 hover:from-pink-600 hover:to-orange-600 text-white py-3 rounded-xl font-bold text-sm shadow-lg hover:shadow-xl transition-all"
                    >
                      View Workshop Details
                    </button>
                  ) : workshop.id === 7 ? (
                    <button 
                      onClick={() => navigate('/workshops/freelancing-side-income')}
                      className="w-full bg-gradient-to-r from-orange-500 to-red-500 hover:from-orange-600 hover:to-red-600 text-white py-3 rounded-xl font-bold text-sm shadow-lg hover:shadow-xl transition-all"
                    >
                      View Workshop Details
                    </button>
                  ) : workshop.id === 8 ? (
                    <button 
                      onClick={() => navigate('/workshops/digital-ads-automation')}
                      className="w-full bg-gradient-to-r from-teal-500 to-cyan-500 hover:from-teal-600 hover:to-cyan-600 text-white py-3 rounded-xl font-bold text-sm shadow-lg hover:shadow-xl transition-all"
                    >
                      View Workshop Details
                    </button>
                  ) : workshop.id === 9 ? (
                    <button 
                      onClick={() => navigate('/workshops/content-creation-mastery')}
                      className="w-full bg-gradient-to-r from-purple-500 to-indigo-500 hover:from-purple-600 hover:to-indigo-600 text-white py-3 rounded-xl font-bold text-sm shadow-lg hover:shadow-xl transition-all"
                    >
                      View Workshop Details
                    </button>
                  ) : workshop.id === 10 ? (
                    <button 
                      onClick={() => navigate('/workshops/resume-portfolio-builder')}
                      className="w-full bg-gradient-to-r from-slate-600 to-blue-600 hover:from-slate-700 hover:to-blue-700 text-white py-3 rounded-xl font-bold text-sm shadow-lg hover:shadow-xl transition-all"
                    >
                      View Workshop Details
                    </button>
                  ) : (
                    <button 
                      disabled
                      className="w-full bg-gray-100 text-gray-400 py-3 rounded-xl font-bold text-sm cursor-not-allowed"
                    >
                      Coming Soon
                    </button>
                  )}
                </div>
              </div>
            ))}
          </div>

          {/* Additional Workshops Grid - Remaining 2 */}
          <div className="grid md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6 mt-6">
            {/* Filler for layout if needed */}
          </div>
        </div>
      </section>
      {/* SECTION 5: ABOUT AI SCHOOL FOR ALL */}
      <section id="about" className="py-32 px-6 bg-white">
        <div className="container mx-auto max-w-7xl">
          {/* Single Line Heading */}
          <h2 className="text-5xl md:text-6xl font-bold text-gray-900 mb-16 text-center">About AI School for All</h2>
          
          <div className="grid lg:grid-cols-2 gap-16 items-start">
            {/* Left: Image with Exemplar below */}
            <div>
              <div className="relative rounded-3xl overflow-hidden shadow-2xl mb-6">
                <img 
                  src="https://images.unsplash.com/photo-1524178232363-1fb2b075b655?w=800&h=600&fit=crop&q=80"
                  alt="Modern AI Learning Environment"
                  className="w-full h-[400px] object-cover"
                />
              </div>
              
              {/* Exemplar Certification under photo */}
              <div className="flex items-start gap-4">
                <a href="https://exemplarglobal.org" target="_blank" rel="noopener noreferrer" className="flex-shrink-0">
                  <img 
                    src="https://customer-assets.emergentagent.com/job_eduai-portal-6/artifacts/qvi4oewl_exempler%20logo.webp"
                    alt="Exemplar Global"
                    className="h-16 object-contain hover:opacity-80 transition-opacity"
                  />
                </a>
                <div>
                  <h3 className="font-bold text-gray-900 mb-1">Exemplar Global Certified</h3>
                  <p className="text-sm text-gray-600 leading-relaxed">Globally recognized accreditation for professional training programs ensuring highest quality standards.</p>
                  <p className="text-sm text-gray-600 mt-1">Programs designed and certified by Exemplar Global for worldwide recognition.</p>
                </div>
              </div>
            </div>

            {/* Right: Content */}
            <div>
              <p className="text-2xl text-gray-600 mb-6 leading-relaxed">
                We are a <span className="font-bold text-blue-600">globally recognized training institution</span> specializing in professional certifications for AI, Data Science, Digital Marketing, and Cloud Computing.
              </p>
              <p className="text-xl text-gray-600 mb-10 leading-relaxed">
                Since 2011, we've trained over 30,000 professionals worldwide, helping them achieve career transformation through industry-recognized certifications, practical skills, and expert mentorship.
              </p>

              {/* Statistics Row */}
              <div className="grid grid-cols-3 gap-6 mb-10">
                <div className="text-center p-6 bg-blue-50 rounded-2xl">
                  <div className="text-4xl font-bold text-blue-600 mb-2">14+</div>
                  <div className="text-sm text-gray-600 font-medium">Years Experience</div>
                </div>
                <div className="text-center p-6 bg-purple-50 rounded-2xl">
                  <div className="text-4xl font-bold text-purple-600 mb-2">30K+</div>
                  <div className="text-sm text-gray-600 font-medium">Students Trained</div>
                </div>
                <div className="text-center p-6 bg-green-50 rounded-2xl">
                  <div className="text-4xl font-bold text-green-600 mb-2">150+</div>
                  <div className="text-sm text-gray-600 font-medium">Hiring Partners</div>
                </div>
              </div>

              {/* Powered by Workshop Library - bottom right without border */}
              <div className="text-right">
                <a 
                  href="https://workshoplibrary.com" 
                  target="_blank" 
                  rel="noopener noreferrer"
                  className="inline-flex items-center gap-2 text-gray-600 hover:text-gray-900 transition-colors"
                >
                  <span className="text-sm font-semibold">Powered by</span>
                  <img 
                    src="https://customer-assets.emergentagent.com/job_eduai-portal-6/artifacts/zrncz5al_Untitled%20%281000%20x%20500%20px%29%20%281%29.jpg"
                    alt="Workshop Library"
                    className="h-6 object-contain"
                  />
                </a>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* SECTION 7: ADDITIONAL BENEFITS */}
      <section className="relative py-32 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-7xl">
          <div className="text-center mb-16">
            <h2 className="text-5xl md:text-6xl font-bold text-gray-900 mb-6">Additional Benefits</h2>
            <p className="text-2xl text-gray-600">Everything you need to succeed in your learning journey</p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-4 gap-6">
            {[
              {
                title: 'Hands-On Projects',
                description: 'Build real-world projects that showcase your skills to employers and clients.',
                color: 'from-blue-400 to-blue-500'
              },
              {
                title: 'State-of-the-Art Learning',
                description: 'Access latest curriculum updated with industry trends and cutting-edge tools.',
                color: 'from-purple-400 to-purple-500'
              },
              {
                title: 'Flexible Learning',
                description: 'Live sessions with recorded content available for review at your convenience.',
                color: 'from-green-400 to-green-500'
              },
              {
                title: 'Placement Support',
                description: 'Career guidance, interview prep, and connections to our hiring partner network.',
                color: 'from-orange-400 to-orange-500'
              }
            ].map((benefit, index) => (
              <div 
                key={index} 
                className="bg-white rounded-2xl p-6 shadow-lg hover:shadow-2xl transition-all transform hover:-translate-y-2 h-full"
                style={{ minHeight: '220px' }}
              >
                <div className={`h-1 w-16 rounded-full bg-gradient-to-r ${benefit.color} mb-6`}></div>
                <h3 className="text-xl font-bold text-gray-900 mb-3">{benefit.title}</h3>
                <p className="text-gray-600 leading-relaxed text-sm">{benefit.description}</p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* SECTION 6: WHY CHOOSE AI SCHOOL FOR ALL - NEW LAYOUT */}
      <section id="why-choose" className="py-32 px-6 bg-gradient-to-b from-white to-gray-50">
        <div className="container mx-auto max-w-7xl">
          {/* Heading */}
          <h2 className="text-5xl md:text-6xl font-bold text-gray-900 mb-20 text-center">Why Choose AI School for All?</h2>
          
          {/* Staggered Diagonal Grid Layout - Non-monotonous */}
          <div className="space-y-24">
            {/* Row 1: Image Left, Text Right */}
            <div className="grid lg:grid-cols-2 gap-12 items-center">
              <div className="relative">
                <div className="absolute -inset-4 bg-gradient-to-br from-blue-400/20 to-purple-400/20 rounded-3xl blur-2xl"></div>
                <img 
                  src="https://images.unsplash.com/photo-1522202176988-66273c2fd55f?w=800&h=600&fit=crop&q=80"
                  alt="Industry Expert Trainers"
                  className="relative rounded-3xl shadow-2xl w-full h-[400px] object-cover"
                />
              </div>
              <div className="space-y-6">
                <div className="inline-block px-4 py-2 bg-blue-100 rounded-full">
                  <span className="text-blue-700 font-bold text-sm">Expert Faculty</span>
                </div>
                <h3 className="text-4xl font-bold text-gray-900">Learn from Industry Leaders</h3>
                <p className="text-xl text-gray-600 leading-relaxed">
                  Our trainers are seasoned professionals with 10+ years of real-world experience. They bring practical insights from Fortune 500 companies and startups alike.
                </p>
                <ul className="space-y-3 text-gray-700">
                  <li className="flex items-start gap-3">
                    <CheckCircle2 className="w-6 h-6 text-green-500 flex-shrink-0 mt-1" />
                    <span>Working professionals from Google, Microsoft, Meta</span>
                  </li>
                  <li className="flex items-start gap-3">
                    <CheckCircle2 className="w-6 h-6 text-green-500 flex-shrink-0 mt-1" />
                    <span>Industry-recognized certifications and credentials</span>
                  </li>
                </ul>
              </div>
            </div>

            {/* Row 2: Text Left, Image Right (Reversed) */}
            <div className="grid lg:grid-cols-2 gap-12 items-center">
              <div className="space-y-6 lg:order-1">
                <div className="inline-block px-4 py-2 bg-purple-100 rounded-full">
                  <span className="text-purple-700 font-bold text-sm">Global Recognition</span>
                </div>
                <h3 className="text-4xl font-bold text-gray-900">Exemplar Global Certified Programs</h3>
                <p className="text-xl text-gray-600 leading-relaxed">
                  All our programs are accredited by Exemplar Global, ensuring worldwide recognition and the highest quality standards in professional training.
                </p>
                <ul className="space-y-3 text-gray-700">
                  <li className="flex items-start gap-3">
                    <CheckCircle2 className="w-6 h-6 text-green-500 flex-shrink-0 mt-1" />
                    <span>Internationally recognized certifications</span>
                  </li>
                  <li className="flex items-start gap-3">
                    <CheckCircle2 className="w-6 h-6 text-green-500 flex-shrink-0 mt-1" />
                    <span>Accepted by employers in 100+ countries</span>
                  </li>
                </ul>
              </div>
              <div className="relative lg:order-2">
                <div className="absolute -inset-4 bg-gradient-to-br from-purple-400/20 to-pink-400/20 rounded-3xl blur-2xl"></div>
                <img 
                  src="https://images.unsplash.com/photo-1523240795612-9a054b0db644?w=800&h=600&fit=crop&q=80"
                  alt="Global Certifications"
                  className="relative rounded-3xl shadow-2xl w-full h-[400px] object-cover"
                />
              </div>
            </div>

            {/* Row 3: Image Left, Text Right */}
            <div className="grid lg:grid-cols-2 gap-12 items-center">
              <div className="relative">
                <div className="absolute -inset-4 bg-gradient-to-br from-green-400/20 to-blue-400/20 rounded-3xl blur-2xl"></div>
                <img 
                  src="https://images.unsplash.com/photo-1531482615713-2afd69097998?w=800&h=600&fit=crop&q=80"
                  alt="Hands-on Learning"
                  className="relative rounded-3xl shadow-2xl w-full h-[400px] object-cover"
                />
              </div>
              <div className="space-y-6">
                <div className="inline-block px-4 py-2 bg-green-100 rounded-full">
                  <span className="text-green-700 font-bold text-sm">Practical Approach</span>
                </div>
                <h3 className="text-4xl font-bold text-gray-900">Hands-On Project-Based Learning</h3>
                <p className="text-xl text-gray-600 leading-relaxed">
                  Move beyond theory with real-world projects, case studies, and simulations. Build a portfolio that demonstrates your expertise to potential employers.
                </p>
                <ul className="space-y-3 text-gray-700">
                  <li className="flex items-start gap-3">
                    <CheckCircle2 className="w-6 h-6 text-green-500 flex-shrink-0 mt-1" />
                    <span>Live projects with real company scenarios</span>
                  </li>
                  <li className="flex items-start gap-3">
                    <CheckCircle2 className="w-6 h-6 text-green-500 flex-shrink-0 mt-1" />
                    <span>Build job-ready portfolio during training</span>
                  </li>
                </ul>
              </div>
            </div>
          </div>

          {/* CTA */}
          <div className="text-center mt-20">
            <button 
              onClick={() => document.getElementById('enrollment-form')?.scrollIntoView({ behavior: 'smooth' })}
              className="bg-gradient-to-r from-yellow-400 to-yellow-500 hover:from-yellow-500 hover:to-yellow-600 text-gray-900 px-10 py-5 rounded-2xl font-bold text-xl shadow-2xl hover:scale-105 transition-all inline-flex items-center gap-3"
            >
              Join Our Community
              <ArrowRight className="w-6 h-6" />
            </button>
          </div>
        </div>
      </section>

      {/* SECTION 8: WHAT ALUMNI SAY */}
      <section className="py-32 px-6 bg-white">
        <div className="container mx-auto max-w-7xl">
          <div className="text-center mb-16">
            <h2 className="text-5xl md:text-6xl font-bold text-gray-900 mb-6">What Alumni Say</h2>
            <p className="text-2xl text-gray-600">Success stories from our graduates</p>
          </div>

          <div className="grid md:grid-cols-3 gap-8">
            {testimonials.map((testimonial, index) => (
              <div 
                key={index}
                className="bg-white rounded-3xl p-8 shadow-lg hover:shadow-2xl transition-all duration-300 border border-gray-100"
              >
                <div className="flex gap-1 mb-4">
                  {[...Array(testimonial.rating)].map((_, i) => (
                    <Star key={i} className="w-5 h-5 fill-yellow-400 text-yellow-400" />
                  ))}
                </div>

                <p className="text-gray-700 mb-6 leading-relaxed italic text-lg">"{testimonial.quote}"</p>

                <div className="flex items-center gap-4">
                  <div className="w-12 h-12 rounded-full bg-gradient-to-r from-blue-500 to-purple-500 flex items-center justify-center text-white font-bold text-lg">
                    {testimonial.name.charAt(0)}
                  </div>
                  <div>
                    <p className="font-bold text-gray-900">{testimonial.name}</p>
                    <p className="text-sm text-gray-600">{testimonial.role}</p>
                    <p className="text-xs text-gray-500">{testimonial.company}</p>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* SECTION 9: FAQ (ENHANCED) */}
      <section id="faq" className="py-32 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-20">
            <h2 className="text-5xl md:text-6xl font-bold text-gray-900 mb-6">Frequently Asked Questions</h2>
            <p className="text-2xl text-gray-600">Everything you need to know about our programs</p>
          </div>

          <Accordion type="single" collapsible className="space-y-4">
            {faqs.map((faq, index) => (
              <AccordionItem 
                key={index} 
                value={`faq-${index}`}
                className="bg-white border-2 border-gray-200 rounded-2xl px-6 hover:border-blue-300 hover:shadow-lg transition-all overflow-hidden"
              >
                <AccordionTrigger className="hover:no-underline py-6">
                  <div className="flex items-center gap-4 text-left">
                    <div className="flex-shrink-0">{faq.icon}</div>
                    <span className="font-semibold text-lg text-gray-900">{faq.question}</span>
                  </div>
                </AccordionTrigger>
                <AccordionContent className="pb-6 text-gray-700 leading-relaxed text-base">
                  {faq.answer}
                </AccordionContent>
              </AccordionItem>
            ))}
          </Accordion>
        </div>
      </section>

      {/* SECTION 10: ENROLLMENT FORM (ENHANCED) */}
      <section id="enrollment-form" className="py-32 px-6 bg-gradient-to-b from-white to-blue-50 relative overflow-hidden">
        <div className="absolute inset-0 opacity-5">
          <div className="absolute inset-0" style={{ backgroundImage: 'radial-gradient(circle, #3b82f6 1px, transparent 1px)', backgroundSize: '30px 30px' }}></div>
        </div>

        <div className="container mx-auto max-w-4xl relative z-10">
          <div className="text-center mb-12">
            <div className="inline-flex flex-wrap items-center justify-center gap-6 bg-white px-8 py-4 rounded-2xl shadow-lg border border-gray-200 mb-10">
              <div className="flex items-center gap-2">
                <CheckCircle2 className="w-5 h-5 text-green-600" />
                <span className="font-semibold text-gray-900">Exemplar Global Accredited</span>
              </div>
              <div className="hidden sm:block w-px h-6 bg-gray-300"></div>
              <div className="flex items-center gap-2">
                <CheckCircle2 className="w-5 h-5 text-green-600" />
                <span className="font-semibold text-gray-900">30,000+ Students</span>
              </div>
              <div className="hidden sm:block w-px h-6 bg-gray-300"></div>
              <div className="flex items-center gap-2">
                <CheckCircle2 className="w-5 h-5 text-green-600" />
                <span className="font-semibold text-gray-900">14+ Years Excellence</span>
              </div>
            </div>

            <h2 className="text-5xl md:text-6xl font-bold text-gray-900 mb-6">Start Your Journey Today</h2>
            <p className="text-2xl text-gray-600">Enroll now and transform your career with globally recognized certification</p>
          </div>

          <div className="bg-white rounded-3xl p-8 md:p-12 shadow-2xl border border-gray-200">
            <EnrollmentForm />
          </div>
        </div>
      </section>

      {/* SECTION 11: FOOTER (ENHANCED) */}
      <footer className="bg-gradient-to-br from-gray-900 via-blue-900 to-gray-900 text-white py-16 px-6 relative overflow-hidden">
        <div className="absolute inset-0 opacity-5">
          <div className="absolute inset-0" style={{ backgroundImage: 'radial-gradient(circle, white 1px, transparent 1px)', backgroundSize: '30px 30px' }}></div>
        </div>

        <div className="absolute top-0 left-0 right-0 h-1 bg-gradient-to-r from-blue-500 via-purple-500 to-blue-500"></div>

        <div className="container mx-auto max-w-7xl relative z-10">
          <div className="grid md:grid-cols-4 gap-12 mb-12">
            <div className="md:col-span-2">
              <div className="flex items-center gap-3 mb-4">
                <div className="w-12 h-12 rounded-xl bg-gradient-to-r from-blue-500 to-purple-500 flex items-center justify-center">
                  <Sparkles className="w-7 h-7 text-white" />
                </div>
                <div>
                  <h3 className="font-bold text-xl">AI School for All</h3>
                  <p className="text-xs text-blue-200">by Workshop Library</p>
                </div>
              </div>
              <p className="text-gray-300 mb-6 leading-relaxed">
                Professional AI, Data Science, and Digital Marketing Education with globally recognized certifications. Making advanced technology education accessible to everyone since 2011.
              </p>

              <div className="flex items-center gap-3">
                <a href="#" className="w-10 h-10 rounded-lg bg-white/10 hover:bg-white/20 flex items-center justify-center transition-all">
                  <Linkedin className="w-5 h-5" />
                </a>
                <a href="#" className="w-10 h-10 rounded-lg bg-white/10 hover:bg-white/20 flex items-center justify-center transition-all">
                  <Twitter className="w-5 h-5" />
                </a>
                <a href="#" className="w-10 h-10 rounded-lg bg-white/10 hover:bg-white/20 flex items-center justify-center transition-all">
                  <Facebook className="w-5 h-5" />
                </a>
                <a href="#" className="w-10 h-10 rounded-lg bg-white/10 hover:bg-white/20 flex items-center justify-center transition-all">
                  <Instagram className="w-5 h-5" />
                </a>
              </div>
            </div>

            <div>
              <h4 className="font-bold text-lg mb-4">Quick Links</h4>
              <ul className="space-y-3 text-sm text-gray-300">
                <li><a href="#courses" className="hover:text-white transition-colors">Our Programs</a></li>
                <li><a href="/courses/csdgpm" className="hover:text-white transition-colors">CSDGPM Course</a></li>
                <li><a href="#enrollment-form" className="hover:text-white transition-colors">Enroll Now</a></li>
                <li><a href="#" className="hover:text-white transition-colors">About Us</a></li>
                <li><a href="#" className="hover:text-white transition-colors">FAQ</a></li>
              </ul>
            </div>

            <div>
              <h4 className="font-bold text-lg mb-4">Contact Us</h4>
              <ul className="space-y-3 text-sm text-gray-300">
                <li className="flex items-start gap-2">
                  <Mail className="w-4 h-4 mt-1 flex-shrink-0" />
                  <a href="mailto:info@aischoolforall.com" className="hover:text-white transition-colors">info@aischoolforall.com</a>
                </li>
                <li className="flex items-start gap-2">
                  <Phone className="w-4 h-4 mt-1 flex-shrink-0" />
                  <a href="tel:+15551234567" className="hover:text-white transition-colors">+1 (555) 123-4567</a>
                </li>
                <li className="flex items-start gap-2">
                  <MapPin className="w-4 h-4 mt-1 flex-shrink-0" />
                  <span>Global Online Programs<br />Accessible Worldwide</span>
                </li>
              </ul>
              <div className="mt-6 pt-6 border-t border-white/10">
                <ul className="space-y-2 text-xs text-gray-400">
                  <li><a href="#" className="hover:text-white transition-colors">Privacy Policy</a></li>
                  <li><a href="#" className="hover:text-white transition-colors">Terms & Conditions</a></li>
                  <li><a href="#" className="hover:text-white transition-colors">Refund Policy</a></li>
                </ul>
              </div>
            </div>
          </div>

          <div className="pt-8 border-t border-white/10">
            <div className="flex flex-col md:flex-row justify-between items-center gap-4">
              <p className="text-sm text-gray-400 text-center md:text-left">
                © 2025 AI School for All by Workshop Library. All rights reserved.
              </p>
              <div className="flex items-center gap-3">
                <Shield className="w-5 h-5 text-yellow-400" />
                <span className="text-sm text-gray-300 font-semibold">Exemplar Global Accredited</span>
              </div>
            </div>
          </div>
        </div>
      </footer>

      {/* Chatbot */}
      <Chatbot />

      {/* Custom Animations */}
      <style jsx>{`
        @keyframes scroll-slow {
          0% { transform: translateX(0); }
          100% { transform: translateX(-50%); }
        }
        .animate-scroll-slow {
          animation: scroll-slow 30s linear infinite;
        }
        @keyframes shine {
          0% { transform: translateX(-100%); }
          100% { transform: translateX(200%); }
        }
        .animate-shine {
          animation: shine 3s infinite;
        }
        @keyframes scroll {
          0% { transform: translateX(0); }
          100% { transform: translateX(-50%); }
        }
        .animate-scroll {
          animation: scroll 15s linear infinite;
        }
      `}</style>
    </div>
  );
};

export default LandingPage;
