import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '../components/ui/card';
import { Button } from '../components/ui/button';
import { Badge } from '../components/ui/badge';
import { Accordion, AccordionContent, AccordionItem, AccordionTrigger } from '../components/ui/accordion';
import { 
  CheckCircle2, Award, Users, BookOpen, Sparkles, TrendingUp, BarChart3, 
  Target, Globe, Clock, Calendar, Download, Star, Video, FileText, Shield, Phone, ArrowRight,
  Linkedin, Twitter, Facebook, Instagram, Mail, MapPin, ExternalLink, Play
} from 'lucide-react';
import CourseLeadForm from '../components/CourseLeadForm';
import Chatbot from '../components/Chatbot';
import { useNavigate } from 'react-router-dom';

const CourseCSDGPM = () => {
  const navigate = useNavigate();
  const [timeLeft, setTimeLeft] = useState({ days: 0, hours: 0, minutes: 0, seconds: 0 });

  // Countdown timer for cohort - December 15, 2025 12:00 AM EST
  useEffect(() => {
    const targetDate = new Date('2025-12-15T05:00:00Z').getTime(); // 12:00 AM EST = 5:00 AM UTC
    
    const timer = setInterval(() => {
      const now = new Date().getTime();
      const difference = targetDate - now;
      
      if (difference > 0) {
        setTimeLeft({
          days: Math.floor(difference / (1000 * 60 * 60 * 24)),
          hours: Math.floor((difference % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60)),
          minutes: Math.floor((difference % (1000 * 60 * 60)) / (1000 * 60)),
          seconds: Math.floor((difference % (1000 * 60)) / 1000)
        });
      }
    }, 1000);

    return () => clearInterval(timer);
  }, []);

  const scrollToForm = () => {
    const formElement = document.getElementById('lead-form');
    if (formElement) {
      formElement.scrollIntoView({ behavior: 'smooth' });
    }
  };

  const modules = [
    {
      title: "MODULE 1: Digital Growth Foundations",
      hours: "20 Hours",
      topics: ["Digital economy overview", "Growth mindset & AARRR framework", "ICP & consumer psychology", "Full-funnel strategy design", "Growth OKRs & metrics"],
      lab: "Lab: Create Digital Growth Map"
    },
    {
      title: "MODULE 2: Performance Marketing Mastery",
      hours: "35 Hours",
      topics: ["Google Ads mastery", "Meta Ads advanced", "LinkedIn Ads B2B", "Advanced targeting & audience building", "Scaling frameworks", "ROI modelling & attribution"],
      lab: "Lab: Launch real campaign"
    },
    {
      title: "MODULE 3: Content, Social & Brand Growth",
      hours: "25 Hours",
      topics: ["Content strategy framework", "AI content tools & automation", "Social media growth systems", "Email marketing & CRM", "30-day content sprint"],
      lab: "Lab: AI content creation workflow"
    },
    {
      title: "MODULE 4: Data, Analytics & Automation",
      hours: "30 Hours",
      topics: ["GA4 mastery", "Looker Studio dashboards", "CRO & experimentation", "Marketing automation", "Predictive analytics"],
      lab: "Lab: Fix a growth funnel"
    },
    {
      title: "MODULE 5: E-Commerce & D2C Growth",
      hours: "20 Hours",
      topics: ["Store optimization", "Marketplace strategy", "Customer lifecycle management", "Catalog ads", "D2C brand case studies"],
      lab: "Case Study: D2C brand growth"
    },
    {
      title: "MODULE 6: Strategy, Leadership & Capstone",
      hours: "20 Hours",
      topics: ["Digital strategy framework", "Team leadership & management", "Specialist certification exam", "Capstone project presentation"],
      lab: "Capstone: End-to-end growth plan"
    }
  ];

  const keyTakeaways = [
    "Master paid media strategy across Google Ads, Meta Ads, and LinkedIn Ads",
    "Build data-driven growth frameworks using GA4 and advanced analytics",
    "Create automated marketing workflows and CRM systems",
    "Develop comprehensive content and social media strategies",
    "Execute e-commerce and D2C growth campaigns",
    "Lead digital transformation initiatives with strategic planning",
    "Build a professional portfolio with real-world projects",
    "Earn Exemplar Global accredited specialist certification"
  ];

  const industryExperts = [
    {
      name: "Sarah Mitchell",
      title: "Chief Marketing Officer",
      company: "Fortune 500 Tech Company",
      expertise: "Performance Marketing Leadership",
      image: "https://images.unsplash.com/photo-1573496359142-b8d87734a5a2?w=400&h=400&fit=crop"
    },
    {
      name: "David Chen",
      title: "Senior Data Scientist",
      company: "Leading Analytics Firm",
      expertise: "Marketing Analytics & AI",
      image: "https://images.unsplash.com/photo-1556157382-97eda2d62296?w=400&h=400&fit=crop"
    },
    {
      name: "Priya Kapoor",
      title: "Marketplace Expert",
      company: "E-commerce Unicorn",
      expertise: "D2C & Marketplace Growth",
      image: "https://images.unsplash.com/photo-1580489944761-15a19d654956?w=400&h=400&fit=crop"
    },
    {
      name: "Michael Rodriguez",
      title: "Growth Director",
      company: "SaaS Scale-up",
      expertise: "Growth Frameworks & Strategy",
      image: "https://images.unsplash.com/photo-1472099645785-5658abf4ff4e?w=400&h=400&fit=crop"
    },
    {
      name: "Emma Thompson",
      title: "Agency Director",
      company: "Global Marketing Agency",
      expertise: "Paid Media & Content Strategy",
      image: "https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=400&h=400&fit=crop"
    }
  ];

  const learningOutcomes = [
    "Master full-funnel digital strategy",
    "Run high-performing paid campaigns",
    "Build analytics dashboards",
    "Create automated marketing workflows",
    "Optimize content/social growth",
    "Build e-commerce performance systems",
    "Deliver strategic growth plans",
    "Become job-ready with specialist portfolio",
    "Earn globally recognized certification"
  ];

  const testimonials = [
    {
      name: "Sarah Johnson",
      role: "Performance Marketing Manager",
      company: "Tech Startup",
      rating: 5,
      text: "This certification transformed my career. The practical approach and industry insights were invaluable."
    },
    {
      name: "Michael Chen",
      role: "Digital Marketing Director",
      company: "E-commerce Brand",
      rating: 5,
      text: "Best investment in my professional development. The curriculum is comprehensive and immediately applicable."
    },
    {
      name: "Priya Sharma",
      role: "Growth Marketing Lead",
      company: "SaaS Company",
      rating: 5,
      text: "The specialist certification opened doors to senior roles. Highly recommend for career advancement."
    },
    {
      name: "David Williams",
      role: "Marketing Consultant",
      company: "Freelance",
      rating: 5,
      text: "Exemplar Global accreditation adds real credibility. Clients recognize and value this certification."
    }
  ];

  const faqs = [
    {
      question: "Who is this certification for?",
      answer: "This specialist-level certification is designed for marketing professionals, digital marketers, performance marketers, business owners, and anyone looking to master digital growth and performance marketing at an advanced level."
    },
    {
      question: "What is the weekly time commitment?",
      answer: "Expect to commit 6-8 hours per week including live sessions, practical labs, assignments, and self-study. The program is designed to fit working professionals' schedules."
    },
    {
      question: "What skills will I master?",
      answer: "You'll master paid advertising (Google, Meta, LinkedIn), analytics (GA4), marketing automation, content strategy, e-commerce growth, data-driven decision making, and strategic planning."
    },
    {
      question: "What happens if I miss a session?",
      answer: "All live sessions are recorded and available for playback. You can catch up at your convenience. However, live participation is encouraged for maximum learning."
    },
    {
      question: "Is this globally recognized?",
      answer: "Yes! This certification is accredited by Exemplar Global, making it internationally recognized and valued by employers worldwide."
    },
    {
      question: "What are the trainer qualifications?",
      answer: "Our trainers are senior industry professionals with 10+ years of experience, currently working in leadership roles at leading companies and agencies."
    },
    {
      question: "What tools will I learn?",
      answer: "Google Ads, Meta Ads Manager, LinkedIn Campaign Manager, GA4, Looker Studio, marketing automation tools, CRM platforms, AI content tools, and more."
    },
    {
      question: "Are the sessions recorded?",
      answer: "Yes, all live sessions are recorded and available for the duration of the program plus 6 months after completion."
    },
    {
      question: "How do assessments work?",
      answer: "You'll have module-wise assignments, practical labs, case study projects, and a final capstone project. Plus a specialist certification exam at the end."
    },
    {
      question: "What is the refund policy?",
      answer: "We offer a 7-day money-back guarantee from the program start date if you're not satisfied with the quality of training."
    }
  ];

  return (
    <div className="min-h-screen bg-white">
      {/* Navigation Menu Strip - Same as Homepage */}
      <nav className="fixed top-4 left-1/2 transform -translate-x-1/2 z-50 w-[95%] max-w-7xl">
        <div className="bg-black/30 backdrop-blur-lg rounded-full border border-white/20 px-8 py-3 shadow-2xl">
          <div className="flex items-center justify-between">
            {/* Logo/Brand */}
            <button onClick={() => navigate('/')} className="text-white font-bold text-lg hover:text-yellow-400 transition-colors">
              AI School for All
            </button>
            
            {/* Navigation Links */}
            <div className="hidden md:flex items-center gap-6">
              <button onClick={() => navigate('/#home')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Home</button>
              <button onClick={() => navigate('/#about')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">About</button>
              <button onClick={() => navigate('/#courses')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Programs</button>
              <button onClick={() => navigate('/#workshops')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Workshops</button>
              <button onClick={() => navigate('/#testimonials')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">Testimonials</button>
              <button onClick={() => navigate('/#faq')} className="text-white/90 hover:text-yellow-400 transition-colors font-medium text-sm">FAQ</button>
              <button 
                onClick={scrollToForm}
                className="bg-yellow-400 hover:bg-yellow-500 text-gray-900 px-6 py-2 rounded-full font-bold transition-all text-sm"
              >
                Enroll Now
              </button>
            </div>
          </div>
        </div>
      </nav>

      {/* Breadcrumb Navigation */}
      <div className="bg-gray-50 border-b">
        <div className="container mx-auto px-6 py-4">
          <div className="flex items-center gap-1 text-sm text-gray-600">
            <button onClick={() => navigate('/')} className="hover:text-yellow-500 transition-colors">Home</button>
            <span>/</span>
            <button onClick={() => navigate('/#courses')} className="hover:text-yellow-500 transition-colors">Courses</button>
            <span>/</span>
            <span className="text-yellow-500 font-semibold">CSDGPM</span>
          </div>
        </div>
      </div>

      {/* Premium Hero Section - Full Width Background */}
      <section className="relative min-h-screen px-6 overflow-hidden flex items-center pt-16">
        {/* Full Width Background Image with Dark Overlay */}
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1460925895917-afdab827c52f?w=1920&h=800&fit=crop&q=80"
            alt="Digital Marketing Analytics"
            className="w-full h-full object-cover"
          />
          {/* Dark Gradient Overlay for Text Readability */}
          <div className="absolute inset-0 bg-gradient-to-r from-gray-900/95 via-gray-900/85 to-gray-900/75"></div>
        </div>
        
        <div className="container mx-auto relative z-10 max-w-6xl py-12">
          {/* Main Title */}
          <div className="text-center">
            <h1 className="text-4xl md:text-6xl font-black text-white mb-5 leading-tight drop-shadow-2xl">
              Certified Specialist in<br />
              <span className="text-transparent bg-clip-text bg-gradient-to-r from-yellow-300 via-yellow-400 to-yellow-500">
                Digital Growth & Performance Marketing
              </span>
            </h1>

            {/* Authority Lines - Beneath the course title */}
            <div className="flex flex-wrap items-center justify-center gap-6 mb-5">
              {/* Line 1: Powered by + Workshop Library logo */}
              <div className="flex items-center gap-2">
                <span className="text-white font-bold text-lg">Powered by</span>
                <img 
                  src="https://customer-assets.emergentagent.com/job_neontech-redesign/artifacts/7vc16rtd_Untitled%20%281000%20x%20500%20px%29%20%281%29.jpg"
                  alt="Workshop Library"
                  className="h-12 object-contain bg-white rounded-lg px-2 py-1"
                />
              </div>

              {/* Line 2: Globally Certified by + Exemplar logo */}
              <div className="flex items-center gap-2">
                <span className="text-white font-bold text-lg">Globally Certified by</span>
                <img 
                  src="https://customer-assets.emergentagent.com/job_neontech-redesign/artifacts/ky6oqo3g_exempler%20logo.webp"
                  alt="Exemplar Global"
                  className="h-12 object-contain bg-white rounded-lg px-2 py-1"
                />
              </div>
            </div>
            
            {/* Subtitle */}
            <p className="text-lg md:text-xl text-gray-200 mb-6 max-w-5xl mx-auto leading-relaxed font-medium">
              A 6-Month Specialist-Level AI-Driven Performance Marketing Certification with 150 Hours of Hands-On Learning.
            </p>

            {/* Highlight Row */}
            <div className="flex flex-wrap items-center justify-center gap-2 mb-5">
              <div className="bg-white/10 backdrop-blur-md px-4 py-2 rounded-full border border-white/30">
                <span className="text-white font-bold text-sm">⭐ Specialist Level</span>
              </div>
              <div className="bg-white/10 backdrop-blur-md px-4 py-2 rounded-full border border-white/30">
                <span className="text-white font-bold text-sm">💼 100% Expert Led</span>
              </div>
              <div className="bg-white/10 backdrop-blur-md px-4 py-2 rounded-full border border-white/30">
                <span className="text-white font-bold text-sm">🛠 Live + Labs + Projects</span>
              </div>
              <div className="bg-white/10 backdrop-blur-md px-4 py-2 rounded-full border border-white/30">
                <span className="text-white font-bold text-sm">⏰ 150 Hours Training</span>
              </div>
              <div className="bg-white/10 backdrop-blur-md px-4 py-2 rounded-full border border-white/30">
                <span className="text-white font-bold text-sm">📊 Real Case Studies</span>
              </div>
            </div>

            {/* CTA Button */}
            <div className="flex flex-wrap gap-5 justify-center">
              <button 
                onClick={scrollToForm} 
                className="bg-gradient-to-r from-yellow-400 to-yellow-500 hover:from-yellow-500 hover:to-yellow-600 text-gray-900 px-8 py-4 rounded-2xl font-black text-base shadow-2xl hover:scale-105 transition-all"
              >
                Enroll Now
                <ArrowRight className="w-5 h-5 inline ml-2" />
              </button>
            </div>
          </div>
        </div>
      </section>
      {/* Course Snapshot - Professional Stats Bar */}
      <section className="py-16 px-6 bg-gradient-to-b from-gray-100 to-white">
        <div className="container mx-auto max-w-6xl">
          <div className="bg-gradient-to-r from-gray-600 to-gray-500 rounded-2xl shadow-xl overflow-hidden mb-16">
            <div className="grid md:grid-cols-4 divide-x divide-gray-400">
              {/* Duration */}
              <div className="p-8 text-center group hover:bg-gray-500/30 transition-all">
                <div className="flex items-center justify-center gap-3 mb-3">
                  <Clock className="w-6 h-6 text-yellow-400" />
                  <p className="text-gray-200 text-sm font-medium uppercase tracking-wider">Duration</p>
                </div>
                <h3 className="text-4xl font-bold text-white">6 Months</h3>
              </div>

              {/* Training Hours */}
              <div className="p-8 text-center group hover:bg-gray-500/30 transition-all">
                <div className="flex items-center justify-center gap-3 mb-3">
                  <Video className="w-6 h-6 text-yellow-400" />
                  <p className="text-gray-200 text-sm font-medium uppercase tracking-wider">Training</p>
                </div>
                <h3 className="text-4xl font-bold text-white">150 Hours</h3>
              </div>

              {/* Format */}
              <div className="p-8 text-center group hover:bg-gray-500/30 transition-all">
                <div className="flex items-center justify-center gap-3 mb-3">
                  <Users className="w-6 h-6 text-yellow-400" />
                  <p className="text-gray-200 text-sm font-medium uppercase tracking-wider">Format</p>
                </div>
                <h3 className="text-4xl font-bold text-white">Live + Labs</h3>
              </div>

              {/* Accreditation */}
              <div className="p-8 text-center group hover:bg-gray-500/30 transition-all">
                <div className="flex items-center justify-center gap-3 mb-3">
                  <Award className="w-6 h-6 text-yellow-400" />
                  <p className="text-gray-200 text-sm font-medium uppercase tracking-wider">Accredited</p>
                </div>
                <h3 className="text-4xl font-bold text-white">Exemplar Global</h3>
              </div>
            </div>
          </div>

          {/* What You'll Master - Moved here after 4 cards */}
          <div className="bg-transparent rounded-3xl p-14 border-2 border-gray-600 shadow-lg">
            <h3 className="text-4xl font-bold mb-12 text-center text-gray-900">What You'll Master</h3>
            <div className="grid md:grid-cols-2 gap-6 max-w-6xl mx-auto">
              {keyTakeaways.map((takeaway, index) => (
                <div key={index} className="flex items-start gap-5 bg-gray-50 rounded-xl p-6 hover:bg-gray-100 transition-all border border-gray-200">
                  <div className="w-7 h-7 rounded-full bg-yellow-400 flex items-center justify-center flex-shrink-0 mt-1">
                    <CheckCircle2 className="w-5 h-5 text-gray-900" />
                  </div>
                  <p className="text-gray-800 text-lg leading-relaxed">{takeaway}</p>
                </div>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* Certification Section */}
      <section className="py-20 px-6 bg-white relative overflow-hidden">
        {/* Background image */}
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1557804506-669a67965ba0?w=1920&h=1080&fit=crop&q=80"
            alt="Professional background"
            className="w-full h-full object-cover opacity-10"
          />
        </div>
        <div className="container mx-auto max-w-5xl relative z-10">
          <div className="bg-gradient-to-br from-gray-500 to-gray-600 rounded-3xl p-12 text-white text-center relative overflow-hidden border-2 border-yellow-400/50 shadow-2xl">
            <div className="absolute top-0 right-0 w-64 h-64 bg-yellow-400/10 rounded-full blur-3xl"></div>
            <div className="relative z-10">
              <Award className="w-24 h-24 mx-auto mb-6 text-yellow-400" />
              <h2 className="text-4xl font-bold mb-6 text-yellow-400">Exemplar Global Accredited Certification</h2>
              <p className="text-xl mb-4 text-white max-w-3xl mx-auto">
                This program is officially accredited by Exemplar Global.
              </p>
              <p className="text-lg text-gray-100 max-w-3xl mx-auto">
                Upon completion, learners receive the <span className="font-bold text-yellow-300">CSDGPM – Certified Specialist in Digital Growth & Performance Marketing</span> credential recognized worldwide.
              </p>
            </div>
          </div>
        </div>
      </section>

      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-5xl">
          <div className="text-center mb-12">
            <Calendar className="w-20 h-20 text-yellow-500 mx-auto mb-6" />
            <h2 className="text-5xl font-bold text-gray-900 mb-6">Next Cohort Starting Soon</h2>
            <div className="w-24 h-1 bg-yellow-400 mx-auto"></div>
          </div>

          <div className="bg-gradient-to-br from-gray-600 to-gray-500 rounded-3xl p-12 shadow-2xl border-2 border-yellow-400/50">
            <div className="grid md:grid-cols-2 gap-8 mb-10">
              <div className="text-center border-r border-yellow-400/30">
                <p className="text-gray-200 font-semibold mb-2 uppercase text-sm tracking-wide">Cohort Starts</p>
                <p className="text-4xl font-bold text-yellow-400 mb-1">January 2026</p>
              </div>
              <div className="text-center">
                <p className="text-gray-200 font-semibold mb-2 uppercase text-sm tracking-wide">Application Deadline</p>
                <p className="text-4xl font-bold text-yellow-400 mb-1">15 Dec 2025</p>
              </div>
            </div>

            {/* Countdown Timer */}
            <div className="bg-gradient-to-r from-gray-700 to-gray-600 rounded-2xl p-8 mb-8 border-2 border-yellow-400/30">
              <p className="text-center text-white font-semibold mb-4 text-lg">Application Closes In:</p>
              <div className="grid grid-cols-4 gap-4">
                <div className="text-center">
                  <div className="bg-white/20 backdrop-blur-sm rounded-xl p-4 shadow-lg mb-2 border-2 border-yellow-400/50">
                    <p className="text-4xl font-bold text-yellow-400">{timeLeft.days}</p>
                  </div>
                  <p className="text-sm text-gray-200 font-medium">Days</p>
                </div>
                <div className="text-center">
                  <div className="bg-white/20 backdrop-blur-sm rounded-xl p-4 shadow-lg mb-2 border-2 border-yellow-400/50">
                    <p className="text-4xl font-bold text-yellow-400">{timeLeft.hours}</p>
                  </div>
                  <p className="text-sm text-gray-200 font-medium">Hours</p>
                </div>
                <div className="text-center">
                  <div className="bg-white/20 backdrop-blur-sm rounded-xl p-4 shadow-lg mb-2 border-2 border-yellow-400/50">
                    <p className="text-4xl font-bold text-yellow-400">{timeLeft.minutes}</p>
                  </div>
                  <p className="text-sm text-gray-200 font-medium">Minutes</p>
                </div>
                <div className="text-center">
                  <div className="bg-white/20 backdrop-blur-sm rounded-xl p-4 shadow-lg mb-2 border-2 border-yellow-400/50">
                    <p className="text-4xl font-bold text-yellow-400">{timeLeft.seconds}</p>
                  </div>
                  <p className="text-sm text-gray-200 font-medium">Seconds</p>
                </div>
              </div>
            </div>

            <div className="grid md:grid-cols-3 gap-6 mb-8">
              <div className="text-center p-4 bg-white/10 backdrop-blur-sm rounded-xl border-2 border-yellow-400/30">
                <Video className="w-8 h-8 text-yellow-400 mx-auto mb-2" />
                <p className="font-semibold text-white">Online Live</p>
              </div>
              <div className="text-center p-4 bg-white/10 backdrop-blur-sm rounded-xl border-2 border-yellow-400/30">
                <Users className="w-8 h-8 text-yellow-400 mx-auto mb-2" />
                <p className="font-semibold text-white">40 Seats Only</p>
              </div>
              <div className="text-center p-4 bg-white/10 backdrop-blur-sm rounded-xl border-2 border-yellow-400/30">
                <Globe className="w-8 h-8 text-yellow-400 mx-auto mb-2" />
                <p className="font-semibold text-white">Global Access</p>
              </div>
            </div>

            <div className="text-center">
              <button 
                onClick={scrollToForm}
                className="bg-gradient-to-r from-yellow-400 to-yellow-500 hover:from-yellow-500 hover:to-yellow-600 text-gray-900 px-10 py-5 rounded-2xl font-bold text-xl shadow-2xl hover:scale-105 transition-all inline-flex items-center gap-3"
              >
                Reserve Your Seat Now
                <ArrowRight className="w-6 h-6" />
              </button>
            </div>
          </div>
        </div>
      </section>
      {/* Premium Course Overview - Executive Education Style */}
      <section className="py-16 px-6 bg-white">
        <div className="container mx-auto max-w-7xl">
          {/* Three Prominent Badges */}
          <div className="flex flex-wrap justify-center items-center gap-4 mb-10">
            <div className="bg-gradient-to-r from-gray-600 to-gray-700 px-6 py-3 rounded-xl shadow-lg hover:shadow-xl transition-all">
              <p className="text-sm font-bold text-white">
                ✓ Assured Google Certification
              </p>
            </div>
            <div className="bg-gradient-to-r from-gray-600 to-gray-700 px-6 py-3 rounded-xl shadow-lg hover:shadow-xl transition-all">
              <p className="text-sm font-bold text-white">
                ✓ Assured Exemplar Global Certification
              </p>
            </div>
            <div className="bg-gradient-to-r from-gray-600 to-gray-700 px-6 py-3 rounded-xl shadow-lg hover:shadow-xl transition-all">
              <p className="text-sm font-bold text-white">
                ✓ Powered by Workshop Library
              </p>
            </div>
          </div>

          {/* Section Title */}
          <div className="text-center mb-10">
            <h2 className="text-5xl font-bold mb-4 text-gray-900">Course Overview</h2>
            <div className="w-32 h-0.5 bg-gradient-to-r from-transparent via-yellow-400 to-transparent mx-auto"></div>
          </div>
          
          {/* Two-Column Split Layout */}
          <div className="grid lg:grid-cols-2 gap-12 items-start mb-10">
            {/* Left Side - Premium Image */}
            <div className="relative">
              <div className="relative rounded-2xl overflow-hidden border-4 border-white shadow-lg">
                <img 
                  src="https://images.unsplash.com/photo-1522071820081-009f0129c71c?w=800&h=600&fit=crop&q=80"
                  alt="Executive Learning Environment"
                  className="w-full h-[550px] object-cover"
                />
                
                {/* Premium Stats Bar */}
                <div className="absolute bottom-8 left-8 right-8 bg-white/98 backdrop-blur-md rounded-xl p-6 shadow-2xl border border-gray-100">
                  <div className="flex items-center justify-between gap-6">
                    <div className="text-center flex-1">
                      <div className="text-4xl font-bold text-yellow-500 mb-2">93%</div>
                      <div className="text-sm text-gray-600 font-medium">Career Growth</div>
                    </div>
                    <div className="w-px h-14 bg-gray-200"></div>
                    <div className="text-center flex-1">
                      <div className="text-4xl font-bold text-yellow-500 mb-2">$40K+</div>
                      <div className="text-sm text-gray-600 font-medium">Avg Salary Increase</div>
                    </div>
                    <div className="w-px h-14 bg-gray-200"></div>
                    <div className="text-center flex-1">
                      <div className="text-4xl font-bold text-yellow-500 mb-2">500+</div>
                      <div className="text-sm text-gray-600 font-medium">Alumni Success</div>
                    </div>
                  </div>
                </div>
              </div>
            </div>

            {/* Right Side - Executive Content */}
            <div className="space-y-10">
              <div className="space-y-6">
                <p className="text-xl leading-loose text-gray-700">
                  This <span className="font-semibold text-gray-900">globally recognized specialist program</span> prepares participants to become digital growth and performance marketing leaders.
                </p>
                
                {/* Bullet Points with Icons */}
                <div className="space-y-5 pt-4">
                  <div className="flex items-start gap-4">
                    <div className="w-2 h-2 rounded-full bg-gray-600 mt-3 flex-shrink-0"></div>
                    <p className="text-lg leading-relaxed text-gray-700">
                      <span className="font-semibold text-gray-900">Comprehensive Curriculum:</span> Integrates paid media mastery, analytics, content systems, and data-driven strategy
                    </p>
                  </div>
                  <div className="flex items-start gap-4">
                    <div className="w-2 h-2 rounded-full bg-gray-600 mt-3 flex-shrink-0"></div>
                    <p className="text-lg leading-relaxed text-gray-700">
                      <span className="font-semibold text-gray-900">Industry Practitioners:</span> Delivered by senior professionals from leading global organizations
                    </p>
                  </div>
                  <div className="flex items-start gap-4">
                    <div className="w-2 h-2 rounded-full bg-gray-600 mt-3 flex-shrink-0"></div>
                    <p className="text-lg leading-relaxed text-gray-700">
                      <span className="font-semibold text-gray-900">Hands-On Learning:</span> Structured live sessions, practical labs, assessments, and capstone project
                    </p>
                  </div>
                  <div className="flex items-start gap-4">
                    <div className="w-2 h-2 rounded-full bg-gray-600 mt-3 flex-shrink-0"></div>
                    <p className="text-lg leading-relaxed text-gray-700">
                      <span className="font-semibold text-gray-900">Career Advancement:</span> Build a specialist portfolio and earn globally recognized credentials
                    </p>
                  </div>
                </div>
              </div>
            </div>
          </div>

          {/* Premium Dual Certification Block - Centered & Full Width */}
          <div className="max-w-5xl mx-auto">
            <div className="bg-white rounded-2xl p-12 shadow-lg border border-gray-100">
              <h3 className="text-3xl font-bold text-gray-900 mb-10 text-center">Dual Certification</h3>
              <div className="flex items-center justify-center gap-16">
                <div className="flex flex-col items-center">
                  <div className="w-32 h-32 bg-gradient-to-br from-gray-100 to-white rounded-xl flex items-center justify-center mb-5 shadow-md border border-gray-200 p-4">
                    <Shield className="w-20 h-20 text-yellow-500" />
                  </div>
                  <p className="text-lg font-bold text-gray-900">Exemplar Global</p>
                  <p className="text-sm text-gray-500 mt-1">Accreditation</p>
                </div>
                
                <div className="text-4xl font-light text-gray-300">+</div>
                
                <div className="flex flex-col items-center">
                  <div className="w-32 h-32 bg-gradient-to-br from-red-50 to-yellow-50 rounded-xl flex items-center justify-center mb-5 shadow-md border border-yellow-100 p-4">
                    <Award className="w-20 h-20 text-yellow-600" />
                  </div>
                  <p className="text-lg font-bold text-gray-900">Google Ads</p>
                  <p className="text-sm text-gray-500 mt-1">Certification Ready</p>
                </div>
              </div>
              <p className="text-center text-base text-gray-600 mt-10 leading-relaxed max-w-3xl mx-auto">
                Earn globally recognized certifications that employers trust and value
              </p>
            </div>
          </div>
        </div>
      </section>

      {/* Skills You Will Gain - Clean Simple List */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-12">
            <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-4">Skills You Will Gain</h2>
            <p className="text-lg text-gray-600 max-w-3xl mx-auto">
              Master the most in-demand digital marketing and growth skills
            </p>
          </div>

          {/* Skills Simple Grid */}
          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-x-8 gap-y-4">
            <div className="flex items-center gap-3 py-2">
              <TrendingUp className="w-5 h-5 text-yellow-500 flex-shrink-0" />
              <span className="text-gray-700 font-medium">SEO</span>
            </div>
            
            <div className="flex items-center gap-3 py-2">
              <Users className="w-5 h-5 text-yellow-500 flex-shrink-0" />
              <span className="text-gray-700 font-medium">Social Media Marketing</span>
            </div>
            
            <div className="flex items-center gap-3 py-2">
              <Linkedin className="w-5 h-5 text-yellow-500 flex-shrink-0" />
              <span className="text-gray-700 font-medium">LinkedIn Marketing</span>
            </div>
            
            <div className="flex items-center gap-3 py-2">
              <Sparkles className="w-5 h-5 text-yellow-500 flex-shrink-0" />
              <span className="text-gray-700 font-medium">Canva Designing</span>
            </div>
            
            <div className="flex items-center gap-3 py-2">
              <Facebook className="w-5 h-5 text-yellow-500 flex-shrink-0" />
              <span className="text-gray-700 font-medium">FB/Instagram Ads</span>
            </div>
            
            <div className="flex items-center gap-3 py-2">
              <Target className="w-5 h-5 text-yellow-500 flex-shrink-0" />
              <span className="text-gray-700 font-medium">Google Ads</span>
            </div>
            
            <div className="flex items-center gap-3 py-2">
              <Mail className="w-5 h-5 text-yellow-500 flex-shrink-0" />
              <span className="text-gray-700 font-medium">Email Marketing</span>
            </div>
            
            <div className="flex items-center gap-3 py-2">
              <FileText className="w-5 h-5 text-yellow-500 flex-shrink-0" />
              <span className="text-gray-700 font-medium">Content Writing</span>
            </div>
            
            <div className="flex items-center gap-3 py-2">
              <MapPin className="w-5 h-5 text-yellow-500 flex-shrink-0" />
              <span className="text-gray-700 font-medium">Local SEO</span>
            </div>
            
            <div className="flex items-center gap-3 py-2">
              <TrendingUp className="w-5 h-5 text-yellow-500 flex-shrink-0" />
              <span className="text-gray-700 font-medium">Growth Hacking</span>
            </div>
            
            <div className="flex items-center gap-3 py-2">
              <BarChart3 className="w-5 h-5 text-yellow-500 flex-shrink-0" />
              <span className="text-gray-700 font-medium">Analytics & Reporting</span>
            </div>
            
            <div className="flex items-center gap-3 py-2">
              <Sparkles className="w-5 h-5 text-yellow-500 flex-shrink-0" />
              <span className="text-gray-700 font-medium">AI Marketing Automation</span>
            </div>
          </div>
        </div>
      </section>

      {/* Premium Accreditation Section - Executive Style */}
      <section className="py-24 px-6 bg-gradient-to-b from-[#EEF4FF] to-[#DCE9FF] relative overflow-hidden">
        {/* Background Image with Light Opacity */}
        <div className="absolute inset-0 opacity-[0.08]">
          <img 
            src="https://images.unsplash.com/photo-1517245386807-bb43f82c33c4?w=1600&h=900&fit=crop&q=80"
            alt="Professional Background"
            className="w-full h-full object-cover"
          />
        </div>

        {/* Faint Circular Abstract Patterns */}
        <div className="absolute inset-0 opacity-[0.12]">
          <div className="absolute top-20 right-20 w-96 h-96 border-2 border-gray-400 rounded-full"></div>
          <div className="absolute bottom-20 left-20 w-80 h-80 border-2 border-gray-400 rounded-full"></div>
          <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 w-[600px] h-[600px] border border-gray-300 rounded-full"></div>
        </div>
        
        <div className="container mx-auto max-w-6xl relative z-10">
          <div className="text-center mb-16">
            <div className="inline-block p-5 bg-white/60 backdrop-blur-sm rounded-2xl shadow-sm mb-8">
              <Shield className="w-20 h-20 text-yellow-500 mx-auto" />
            </div>
            <h2 className="text-5xl font-bold text-gray-900 mb-6 leading-tight">
              Officially Accredited by<br />Exemplar Global
            </h2>
            <div className="w-32 h-0.5 bg-gradient-to-r from-transparent via-yellow-400 to-transparent mx-auto mb-8"></div>
            <p className="text-xl text-gray-700 leading-loose max-w-3xl mx-auto">
              This program meets international standards for specialist-level competency development, professional assessment, and skills validation. Graduates receive the globally recognized <span className="font-bold text-gray-900">CSDGPM Specialist Certification</span>.
            </p>
          </div>

          <div className="grid md:grid-cols-3 gap-8 mb-12">
            <div className="bg-white/80 backdrop-blur-md rounded-2xl p-8 shadow-sm border border-white/50 hover:shadow-md transition-all">
              <Award className="w-14 h-14 text-yellow-500 mx-auto mb-5" />
              <p className="text-gray-900 font-semibold text-center text-lg">Globally Recognized</p>
              <p className="text-gray-600 text-sm text-center mt-2">Valued by employers worldwide</p>
            </div>
            <div className="bg-white/80 backdrop-blur-md rounded-2xl p-8 shadow-sm border border-white/50 hover:shadow-md transition-all">
              <Shield className="w-14 h-14 text-yellow-500 mx-auto mb-5" />
              <p className="text-gray-900 font-semibold text-center text-lg">Quality Assured</p>
              <p className="text-gray-600 text-sm text-center mt-2">Meets international standards</p>
            </div>
            <div className="bg-white/80 backdrop-blur-md rounded-2xl p-8 shadow-sm border border-white/50 hover:shadow-md transition-all">
              <FileText className="w-14 h-14 text-yellow-500 mx-auto mb-5" />
              <p className="text-gray-900 font-semibold text-center text-lg">Verified Standards</p>
              <p className="text-gray-600 text-sm text-center mt-2">Rigorous assessment criteria</p>
            </div>
          </div>

          <div className="text-center">
            <button 
              onClick={scrollToForm}
              className="bg-gray-600 hover:bg-gray-700 text-white px-10 py-4 rounded-xl font-bold text-lg shadow-lg hover:shadow-xl transition-all inline-flex items-center gap-3"
            >
              <Download className="w-5 h-5" />
              Download Sample Certificate
            </button>
          </div>
        </div>
      </section>

      {/* Lead Form - Mid Page */}
      <section id="lead-form" className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-3xl">
          <div className="text-center mb-8">
            <h2 className="text-4xl font-bold mb-4 text-gray-900">Get Complete Course Information</h2>
            <p className="text-lg text-gray-600">Fill in your details to receive the full curriculum and brochure</p>
          </div>
          <CourseLeadForm courseName="CSDGPM" formTitle="Request Course Brochure" />
        </div>
      </section>

      {/* Professional Modules Section */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-16">
            <h2 className="text-5xl font-bold mb-6 text-gray-900">Detailed Curriculum</h2>
            <div className="w-24 h-1 bg-gradient-to-r from-gray-600 to-gray-400 mx-auto mb-6"></div>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">
              6 comprehensive modules covering every aspect of digital growth and performance marketing
            </p>
          </div>

          <Accordion type="single" collapsible className="space-y-6">
            {modules.map((module, index) => (
              <AccordionItem 
                key={index} 
                value={`module-${index}`}
                className="bg-gradient-to-br from-white to-gray-100 border-2 border-gray-300 rounded-3xl overflow-hidden hover:border-yellow-400 hover:shadow-2xl transition-all"
              >
                <AccordionTrigger className="text-left hover:no-underline py-8 px-8">
                  <div className="flex items-center gap-6 w-full">
                    <div className="w-16 h-16 rounded-2xl bg-gradient-to-br from-gray-600 to-gray-700 flex items-center justify-center text-white font-bold text-2xl flex-shrink-0 shadow-lg">
                      {index + 1}
                    </div>
                    <div className="flex-1">
                      <h3 className="font-bold text-2xl mb-2 text-gray-900">{module.title}</h3>
                      <div className="flex items-center gap-4 text-sm">
                        <span className="flex items-center gap-1 text-gray-600">
                          <Clock className="w-4 h-4" />
                          {module.hours}
                        </span>
                      </div>
                    </div>
                  </div>
                </AccordionTrigger>
                <AccordionContent className="pb-8 pt-4 px-8">
                  <div className="pl-20">
                    <h4 className="font-bold text-lg mb-4 text-yellow-500 flex items-center gap-2">
                      <BookOpen className="w-5 h-5" />
                      Topics Covered:
                    </h4>
                    <div className="grid md:grid-cols-2 gap-4 mb-6">
                      {module.topics.map((topic, idx) => (
                        <div key={idx} className="flex items-start gap-3 bg-white p-3 rounded-xl border border-gray-200">
                          <CheckCircle2 className="w-5 h-5 text-yellow-500 flex-shrink-0 mt-0.5" />
                          <span className="text-gray-700">{topic}</span>
                        </div>
                      ))}
                    </div>
                    <div className="bg-gradient-to-r from-gray-600 to-gray-700 p-6 rounded-2xl text-white shadow-lg">
                      <div className="flex items-center gap-3">
                        <Target className="w-6 h-6" />
                        <p className="font-semibold text-lg">{module.lab}</p>
                      </div>
                    </div>
                  </div>
                </AccordionContent>
              </AccordionItem>
            ))}
          </Accordion>
        </div>
      </section>

      {/* Industry Expert Masterclasses - With Professional Images */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-100 to-white">
        <div className="container mx-auto max-w-7xl">
          <div className="text-center mb-12">
            <h2 className="text-5xl font-bold mb-6 text-gray-900">Industry Expert Masterclasses Included</h2>
            <div className="w-24 h-1 bg-gradient-to-r from-gray-600 to-gray-400 mx-auto mb-6"></div>
            <p className="text-lg text-gray-600 max-w-4xl mx-auto mb-4">
              Learn directly from senior professionals who have worked with top global companies across performance marketing, analytics, e-commerce, and digital transformation.
            </p>
            <p className="text-sm text-gray-500 italic max-w-3xl mx-auto">
              Our expert pool includes professionals with experience across Fortune 500 companies, leading digital agencies, high-growth SaaS brands, and global e-commerce firms.
            </p>
          </div>

          <div className="grid md:grid-cols-5 gap-6 mb-10">
            {/* Card 1: Performance Marketing Leadership */}
            <div className="bg-white rounded-3xl overflow-hidden border-2 border-gray-200 hover:border-yellow-400 hover:shadow-2xl transition-all group">
              <div className="aspect-square overflow-hidden relative">
                <img 
                  src="https://images.unsplash.com/photo-1460925895917-afdab827c52f?w=400&h=400&fit=crop&q=80"
                  alt="Performance Marketing"
                  className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-gray-800/80 to-transparent"></div>
              </div>
              <div className="p-5">
                <div className="bg-gray-600 text-white px-3 py-1 rounded-full text-xs font-bold mb-3 inline-block">
                  Leadership & Paid Media
                </div>
                <h4 className="font-bold text-lg text-gray-900 mb-2">Performance Marketing Leadership</h4>
                <p className="text-sm text-gray-600">Delivered by CMOs & Marketing Directors</p>
              </div>
            </div>

            {/* Card 2: Marketing Analytics & AI */}
            <div className="bg-white rounded-3xl overflow-hidden border-2 border-gray-200 hover:border-yellow-400 hover:shadow-2xl transition-all group">
              <div className="aspect-square overflow-hidden relative">
                <img 
                  src="https://images.unsplash.com/photo-1551288049-bebda4e38f71?w=400&h=400&fit=crop&q=80"
                  alt="Analytics AI"
                  className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-gray-800/80 to-transparent"></div>
              </div>
              <div className="p-5">
                <div className="bg-gray-600 text-white px-3 py-1 rounded-full text-xs font-bold mb-3 inline-block">
                  Analytics & AI
                </div>
                <h4 className="font-bold text-lg text-gray-900 mb-2">Marketing Analytics & AI</h4>
                <p className="text-sm text-gray-600">Delivered by Senior Data Scientists</p>
              </div>
            </div>

            {/* Card 3: D2C & Marketplace Growth */}
            <div className="bg-white rounded-3xl overflow-hidden border-2 border-gray-200 hover:border-yellow-400 hover:shadow-2xl transition-all group">
              <div className="aspect-square overflow-hidden relative">
                <img 
                  src="https://images.unsplash.com/photo-1556742502-ec7c0e9f34b1?w=400&h=400&fit=crop&q=80"
                  alt="E-commerce Growth"
                  className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-gray-800/80 to-transparent"></div>
              </div>
              <div className="p-5">
                <div className="bg-gray-600 text-white px-3 py-1 rounded-full text-xs font-bold mb-3 inline-block">
                  E-Commerce Strategy
                </div>
                <h4 className="font-bold text-lg text-gray-900 mb-2">D2C & Marketplace Growth</h4>
                <p className="text-sm text-gray-600">Delivered by E-commerce & Marketplace Specialists</p>
              </div>
            </div>

            {/* Card 4: Growth Frameworks */}
            <div className="bg-white rounded-3xl overflow-hidden border-2 border-gray-200 hover:border-yellow-400 hover:shadow-2xl transition-all group">
              <div className="aspect-square overflow-hidden relative">
                <img 
                  src="https://images.unsplash.com/photo-1542744173-8e7e53415bb0?w=400&h=400&fit=crop&q=80"
                  alt="Growth Strategy"
                  className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-gray-800/80 to-transparent"></div>
              </div>
              <div className="p-5">
                <div className="bg-gray-600 text-white px-3 py-1 rounded-full text-xs font-bold mb-3 inline-block">
                  Growth Strategy
                </div>
                <h4 className="font-bold text-lg text-gray-900 mb-2">Growth Frameworks & Scaling</h4>
                <p className="text-sm text-gray-600">Delivered by Growth Directors & SaaS Experts</p>
              </div>
            </div>

            {/* Card 5: Paid Media & Content */}
            <div className="bg-white rounded-3xl overflow-hidden border-2 border-gray-200 hover:border-yellow-400 hover:shadow-2xl transition-all group">
              <div className="aspect-square overflow-hidden relative">
                <img 
                  src="https://images.unsplash.com/photo-1557804506-669a67965ba0?w=400&h=400&fit=crop&q=80"
                  alt="Content Strategy"
                  className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-gray-800/80 to-transparent"></div>
              </div>
              <div className="p-5">
                <div className="bg-gray-600 text-white px-3 py-1 rounded-full text-xs font-bold mb-3 inline-block">
                  Paid Media + Content
                </div>
                <h4 className="font-bold text-lg text-gray-900 mb-2">Paid Media & Content Strategy</h4>
                <p className="text-sm text-gray-600">Delivered by Agency & Creative Directors</p>
              </div>
            </div>
          </div>

          <div className="text-center mt-12">
            <button 
              onClick={scrollToForm}
              className="bg-gradient-to-r from-yellow-400 to-yellow-500 hover:from-yellow-500 hover:to-yellow-600 text-gray-900 px-12 py-5 rounded-2xl font-bold text-xl shadow-2xl hover:scale-105 transition-all inline-flex items-center gap-3"
            >
              <Phone className="w-6 h-6" />
              Connect With Us
              <ArrowRight className="w-6 h-6" />
            </button>
            <p className="text-xs text-gray-500 italic max-w-2xl mx-auto mt-6">
              *Images shown are representative. Actual experts vary by cohort and include senior professionals from global organizations.
            </p>
          </div>
        </div>
      </section>

      {/* Redesigned Learning Outcomes - Premium Bullet Grid */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-7xl">
          <div className="text-center mb-16">
            <h2 className="text-5xl font-bold mb-6 text-gray-900">Learning Outcomes</h2>
            <div className="w-24 h-1 bg-gradient-to-r from-gray-600 to-gray-400 mx-auto mb-6"></div>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">
              What you'll achieve by completing this specialist certification
            </p>
          </div>

          <div className="grid md:grid-cols-3 gap-x-12 gap-y-10">
            {/* Row 1 */}
            <div className="flex items-start gap-5 group">
              <div className="w-16 h-16 rounded-full bg-gradient-to-br from-gray-1000 to-gray-600 flex items-center justify-center flex-shrink-0 shadow-lg group-hover:scale-110 transition-transform">
                <Target className="w-8 h-8 text-white" />
              </div>
              <div className="flex-1">
                <h3 className="font-bold text-xl text-gray-900 mb-2">Master full-funnel digital strategy</h3>
                <p className="text-sm text-gray-600 leading-relaxed">Design and deploy end-to-end growth systems.</p>
              </div>
            </div>

            <div className="flex items-start gap-5 group">
              <div className="w-16 h-16 rounded-full bg-gradient-to-br from-purple-500 to-purple-600 flex items-center justify-center flex-shrink-0 shadow-lg group-hover:scale-110 transition-transform">
                <TrendingUp className="w-8 h-8 text-white" />
              </div>
              <div className="flex-1">
                <h3 className="font-bold text-xl text-gray-900 mb-2">Run high-performing paid campaigns</h3>
                <p className="text-sm text-gray-600 leading-relaxed">Meta, Google, LinkedIn & multi-channel funnels.</p>
              </div>
            </div>

            <div className="flex items-start gap-5 group">
              <div className="w-16 h-16 rounded-full bg-gradient-to-br from-green-500 to-green-600 flex items-center justify-center flex-shrink-0 shadow-lg group-hover:scale-110 transition-transform">
                <BarChart3 className="w-8 h-8 text-white" />
              </div>
              <div className="flex-1">
                <h3 className="font-bold text-xl text-gray-900 mb-2">Build analytics dashboards</h3>
                <p className="text-sm text-gray-600 leading-relaxed">GA4, Looker Studio & attribution.</p>
              </div>
            </div>

            {/* Row 2 */}
            <div className="flex items-start gap-5 group">
              <div className="w-16 h-16 rounded-full bg-gradient-to-br from-orange-500 to-orange-600 flex items-center justify-center flex-shrink-0 shadow-lg group-hover:scale-110 transition-transform">
                <Sparkles className="w-8 h-8 text-white" />
              </div>
              <div className="flex-1">
                <h3 className="font-bold text-xl text-gray-900 mb-2">Create automated marketing workflows</h3>
                <p className="text-sm text-gray-600 leading-relaxed">AI, Make.com, Zapier & CRM logic.</p>
              </div>
            </div>

            <div className="flex items-start gap-5 group">
              <div className="w-16 h-16 rounded-full bg-gradient-to-br from-teal-500 to-teal-600 flex items-center justify-center flex-shrink-0 shadow-lg group-hover:scale-110 transition-transform">
                <Users className="w-8 h-8 text-white" />
              </div>
              <div className="flex-1">
                <h3 className="font-bold text-xl text-gray-900 mb-2">Optimize content/social growth</h3>
                <p className="text-sm text-gray-600 leading-relaxed">Scale content & engagement systems.</p>
              </div>
            </div>

            <div className="flex items-start gap-5 group">
              <div className="w-16 h-16 rounded-full bg-gradient-to-br from-pink-500 to-pink-600 flex items-center justify-center flex-shrink-0 shadow-lg group-hover:scale-110 transition-transform">
                <Globe className="w-8 h-8 text-white" />
              </div>
              <div className="flex-1">
                <h3 className="font-bold text-xl text-gray-900 mb-2">Build e-commerce performance systems</h3>
                <p className="text-sm text-gray-600 leading-relaxed">D2C funnels, catalog ads, upsell paths.</p>
              </div>
            </div>

            {/* Row 3 */}
            <div className="flex items-start gap-5 group">
              <div className="w-16 h-16 rounded-full bg-gradient-to-br from-indigo-500 to-indigo-600 flex items-center justify-center flex-shrink-0 shadow-lg group-hover:scale-110 transition-transform">
                <FileText className="w-8 h-8 text-white" />
              </div>
              <div className="flex-1">
                <h3 className="font-bold text-xl text-gray-900 mb-2">Deliver strategic growth plans</h3>
                <p className="text-sm text-gray-600 leading-relaxed">Annual plans, OKRs & scaling frameworks.</p>
              </div>
            </div>

            <div className="flex items-start gap-5 group">
              <div className="w-16 h-16 rounded-full bg-gradient-to-br from-cyan-500 to-cyan-600 flex items-center justify-center flex-shrink-0 shadow-lg group-hover:scale-110 transition-transform">
                <BookOpen className="w-8 h-8 text-white" />
              </div>
              <div className="flex-1">
                <h3 className="font-bold text-xl text-gray-900 mb-2">Become job-ready with specialist portfolio</h3>
                <p className="text-sm text-gray-600 leading-relaxed">Campaigns, dashboards & capstone.</p>
              </div>
            </div>

            <div className="flex items-start gap-5 group">
              <div className="w-16 h-16 rounded-full bg-gradient-to-br from-yellow-500 to-yellow-600 flex items-center justify-center flex-shrink-0 shadow-lg group-hover:scale-110 transition-transform">
                <Award className="w-8 h-8 text-white" />
              </div>
              <div className="flex-1">
                <h3 className="font-bold text-xl text-gray-900 mb-2">Earn globally recognized certification</h3>
                <p className="text-sm text-gray-600 leading-relaxed">Exemplar Global accredited specialist credential.</p>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Premium Pricing Section - Redesigned */}
      <section className="py-20 px-6 bg-gradient-to-b from-gray-50 to-white">
        <div className="container mx-auto max-w-7xl">
          <div className="text-center mb-16">
            <h2 className="text-5xl font-bold mb-6 text-gray-900">Choose Your Learning Path</h2>
            <div className="w-24 h-1 bg-gradient-to-r from-gray-600 to-gray-400 mx-auto mb-6"></div>
            <p className="text-xl text-gray-600 max-w-3xl mx-auto">
              Select the package that best fits your career goals and learning style
            </p>
          </div>

          <div className="grid md:grid-cols-3 gap-8 items-center">
            {/* Standard Plan */}
            <div className="bg-gradient-to-br from-white to-gray-50 rounded-3xl border-2 border-gray-200 hover:border-gray-400 hover:shadow-2xl transition-all duration-300 p-8 h-full">
              <div className="flex items-center gap-3 mb-6">
                <div className="w-12 h-12 rounded-full bg-gradient-to-br from-gray-1000 to-gray-600 flex items-center justify-center shadow-lg">
                  <CheckCircle2 className="w-7 h-7 text-white" />
                </div>
                <div>
                  <h3 className="text-3xl font-bold text-gray-900">Standard</h3>
                </div>
              </div>
              <p className="text-gray-600 mb-4">Complete certification program</p>
              
              <div className="mb-6">
                <div className="flex items-baseline gap-2">
                  <span className="text-4xl font-bold text-gray-900">$499</span>
                  <span className="text-gray-500">/ ₹19,999</span>
                </div>
              </div>

              <ul className="space-y-3 mb-8">
                <li className="flex items-start gap-3">
                  <CheckCircle2 className="w-5 h-5 text-yellow-500 flex-shrink-0 mt-0.5" />
                  <span className="text-gray-700">6 months full access</span>
                </li>
                <li className="flex items-start gap-3">
                  <CheckCircle2 className="w-5 h-5 text-yellow-500 flex-shrink-0 mt-0.5" />
                  <span className="text-gray-700">All live sessions</span>
                </li>
                <li className="flex items-start gap-3">
                  <CheckCircle2 className="w-5 h-5 text-yellow-500 flex-shrink-0 mt-0.5" />
                  <span className="text-gray-700">All labs & projects</span>
                </li>
                <li className="flex items-start gap-3">
                  <CheckCircle2 className="w-5 h-5 text-yellow-500 flex-shrink-0 mt-0.5" />
                  <span className="text-gray-700">Certificate</span>
                </li>
              </ul>
              <button 
                onClick={scrollToForm}
                className="w-full bg-gradient-to-r from-gray-600 to-gray-700 hover:from-gray-700 hover:to-gray-700 text-white py-4 rounded-xl font-bold text-lg shadow-lg hover:shadow-xl hover:scale-105 transition-all duration-300"
              >
                Enroll Now
              </button>
            </div>

            {/* Pro Plan - Most Popular & Visually Dominant */}
            <div className="bg-gradient-to-br from-gray-600 via-gray-700 to-gray-700 rounded-3xl border-2 border-yellow-400 shadow-2xl transform md:scale-110 p-8 relative">
              <div className="absolute -top-4 left-1/2 transform -translate-x-1/2 z-10">
                <div className="bg-gradient-to-r from-yellow-400 to-yellow-500 text-gray-900 px-8 py-2 rounded-full font-bold text-sm shadow-xl">
                  ⭐ MOST POPULAR
                </div>
              </div>
              
              <div className="flex items-center gap-3 mb-6">
                <div className="w-12 h-12 rounded-full bg-gradient-to-br from-green-400 to-green-500 flex items-center justify-center shadow-lg">
                  <CheckCircle2 className="w-7 h-7 text-white" />
                </div>
                <div>
                  <h3 className="text-3xl font-bold text-white">Pro</h3>
                </div>
              </div>
              <p className="text-gray-200 mb-4">Enhanced learning experience</p>
              
              <div className="mb-6">
                <div className="flex items-baseline gap-2">
                  <span className="text-4xl font-bold text-white">$799</span>
                  <span className="text-gray-300">/ ₹34,999</span>
                </div>
              </div>

              <ul className="space-y-3 mb-8">
                <li className="flex items-start gap-3">
                  <CheckCircle2 className="w-5 h-5 text-yellow-300 flex-shrink-0 mt-0.5" />
                  <span className="text-white font-medium">Everything in Standard</span>
                </li>
                <li className="flex items-start gap-3">
                  <CheckCircle2 className="w-5 h-5 text-yellow-300 flex-shrink-0 mt-0.5" />
                  <span className="text-white font-medium">Mentor sessions</span>
                </li>
                <li className="flex items-start gap-3">
                  <CheckCircle2 className="w-5 h-5 text-yellow-300 flex-shrink-0 mt-0.5" />
                  <span className="text-white font-medium">Extra exam prep</span>
                </li>
                <li className="flex items-start gap-3">
                  <CheckCircle2 className="w-5 h-5 text-yellow-300 flex-shrink-0 mt-0.5" />
                  <span className="text-white font-medium">Priority support</span>
                </li>
              </ul>
              <button 
                onClick={scrollToForm}
                className="w-full bg-white hover:bg-gray-100 text-gray-900 py-4 rounded-xl font-bold text-lg shadow-xl hover:shadow-2xl hover:scale-105 transition-all duration-300"
              >
                Enroll in Pro
              </button>
            </div>

            {/* Premium Plan */}
            <div className="bg-gradient-to-br from-white to-amber-50 rounded-3xl border-2 border-amber-200 hover:border-amber-400 hover:shadow-2xl transition-all duration-300 p-8 h-full">
              <div className="flex items-center gap-3 mb-6">
                <div className="w-12 h-12 rounded-full bg-gradient-to-br from-amber-500 to-yellow-600 flex items-center justify-center shadow-lg">
                  <Award className="w-7 h-7 text-white" />
                </div>
                <div>
                  <h3 className="text-3xl font-bold text-gray-900">Premium</h3>
                </div>
              </div>
              <p className="text-gray-600 mb-4">Ultimate career transformation</p>
              
              <div className="mb-6">
                <div className="flex items-baseline gap-2">
                  <span className="text-4xl font-bold text-gray-900">$1299</span>
                  <span className="text-gray-500">/ ₹54,999</span>
                </div>
              </div>

              <ul className="space-y-3 mb-8">
                <li className="flex items-start gap-3">
                  <CheckCircle2 className="w-5 h-5 text-amber-600 flex-shrink-0 mt-0.5" />
                  <span className="text-gray-700">Everything in Pro</span>
                </li>
                <li className="flex items-start gap-3">
                  <CheckCircle2 className="w-5 h-5 text-amber-600 flex-shrink-0 mt-0.5" />
                  <span className="text-gray-700">1:1 coaching</span>
                </li>
                <li className="flex items-start gap-3">
                  <CheckCircle2 className="w-5 h-5 text-amber-600 flex-shrink-0 mt-0.5" />
                  <span className="text-gray-700">Portfolio review</span>
                </li>
                <li className="flex items-start gap-3">
                  <CheckCircle2 className="w-5 h-5 text-amber-600 flex-shrink-0 mt-0.5" />
                  <span className="text-gray-700">Career preparation</span>
                </li>
              </ul>
              <button 
                onClick={scrollToForm}
                className="w-full bg-gradient-to-r from-amber-500 to-yellow-600 hover:from-amber-600 hover:to-yellow-700 text-white py-4 rounded-xl font-bold text-lg shadow-lg hover:shadow-xl hover:scale-105 transition-all duration-300"
              >
                Join Premium
              </button>
            </div>
          </div>
        </div>
      </section>

      {/* Testimonials */}
      <section className="py-20 px-6">
        <div className="container mx-auto max-w-6xl">
          <div className="text-center mb-12">
            <h2 className="text-4xl font-bold mb-4" style={{ color: 'var(--text-primary)' }}>What Our Alumni Say</h2>
          </div>

          <div className="grid md:grid-cols-2 gap-8">
            {testimonials.map((testimonial, index) => (
              <Card key={index} className="border-2 border-gray-200 hover:border-gray-400 transition-all">
                <CardContent className="p-8">
                  <div className="flex gap-1 mb-4">
                    {[...Array(testimonial.rating)].map((_, i) => (
                      <Star key={i} className="w-5 h-5 fill-yellow-400 text-yellow-400" />
                    ))}
                  </div>
                  <p className="text-gray-700 mb-6 italic">"{testimonial.text}"</p>
                  <div className="border-t pt-4">
                    <p className="font-bold">{testimonial.name}</p>
                    <p className="text-sm text-gray-600">{testimonial.role}</p>
                    <p className="text-sm text-gray-500">{testimonial.company}</p>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      </section>

      {/* Upcoming Cohort with Countdown */}

      {/* FAQ Section */}
      <section className="py-20 px-6 bg-white">
        <div className="container mx-auto max-w-4xl">
          <div className="text-center mb-12">
            <h2 className="text-4xl font-bold mb-4" style={{ color: 'var(--text-primary)' }}>Frequently Asked Questions</h2>
          </div>

          <Accordion type="single" collapsible className="space-y-4">
            {faqs.map((faq, index) => (
              <AccordionItem 
                key={index} 
                value={`faq-${index}`}
                className="bg-gray-50 border-2 border-gray-200 rounded-2xl px-8 hover:border-gray-400 transition-all"
              >
                <AccordionTrigger className="text-left hover:no-underline py-6">
                  <span className="font-semibold text-lg">{faq.question}</span>
                </AccordionTrigger>
                <AccordionContent className="pb-6 text-gray-700">
                  {faq.answer}
                </AccordionContent>
              </AccordionItem>
            ))}
          </Accordion>
        </div>
      </section>

      {/* Bottom Lead Form */}
      <section className="py-20 px-6 bg-gradient-to-br from-gray-600 to-gray-500">
        <div className="container mx-auto max-w-2xl">
          <div className="text-center mb-8">
            <h2 className="text-4xl font-bold text-white mb-4">Ready to Begin Your Journey?</h2>
            <p className="text-xl text-gray-100">Get started with CSDGPM certification today</p>
          </div>
          <CourseLeadForm courseName="CSDGPM" formTitle="Start Your Enrollment" />
        </div>
      </section>

      {/* Enhanced Professional Footer */}
      <footer className="py-16 px-6 bg-gradient-to-r from-gray-900 via-gray-800 to-gray-900 text-white relative overflow-hidden">
        {/* Background image like hero */}
        <div className="absolute inset-0 z-0">
          <img 
            src="https://images.unsplash.com/photo-1451187580459-43490279c0fa?w=1920&h=1080&fit=crop&q=80"
            alt="Footer background"
            className="w-full h-full object-cover opacity-5"
          />
        </div>
        <div className="relative z-10">
        <div className="container mx-auto max-w-7xl">
          <div className="grid md:grid-cols-5 gap-10 mb-12">
            {/* About Column */}
            <div className="md:col-span-2">
              <div className="flex items-center gap-3 mb-4">
                <div className="w-12 h-12 rounded-xl bg-gradient-to-br from-gray-700 to-gray-600 border-2 border-yellow-400/50 flex items-center justify-center">
                  <Sparkles className="w-7 h-7 text-yellow-400" />
                </div>
                <div>
                  <h3 className="font-bold text-xl">AI School for All</h3>
                  <p className="text-xs text-gray-400">by Workshop Library</p>
                </div>
              </div>
              <p className="text-gray-300 text-sm leading-relaxed mb-4">
                Professional AI and Digital Marketing Education with globally recognized certifications. Making advanced technology education accessible to everyone.
              </p>
              <div className="flex items-center gap-3">
                <a href="#" className="w-10 h-10 rounded-lg bg-white/10 hover:bg-white/20 flex items-center justify-center transition-all">
                  <Linkedin className="w-5 h-5" />
                </a>
                <a href="#" className="w-10 h-10 rounded-lg bg-white/10 hover:bg-white/20 flex items-center justify-center transition-all">
                  <Twitter className="w-5 h-5" />
                </a>
                <a href="#" className="w-10 h-10 rounded-lg bg-white/10 hover:bg-white/20 flex items-center justify-center transition-all">
                  <Facebook className="w-5 h-5" />
                </a>
                <a href="#" className="w-10 h-10 rounded-lg bg-white/10 hover:bg-white/20 flex items-center justify-center transition-all">
                  <Instagram className="w-5 h-5" />
                </a>
              </div>
            </div>

            {/* Courses Column */}
            <div>
              <h4 className="font-bold text-lg mb-4">Courses</h4>
              <ul className="space-y-3 text-sm text-gray-300">
                <li><button onClick={() => navigate('/')} className="hover:text-white transition-colors">All Courses</button></li>
                <li><button onClick={() => navigate('/#courses')} className="hover:text-white transition-colors">Machine Learning</button></li>
                <li><button onClick={() => navigate('/#courses')} className="hover:text-white transition-colors">Data Analytics</button></li>
                <li><button onClick={scrollToForm} className="hover:text-white transition-colors">Digital Marketing</button></li>
                <li><button onClick={() => navigate('/#courses')} className="hover:text-white transition-colors">Cloud Computing</button></li>
              </ul>
            </div>

            {/* Resources Column */}
            <div>
              <h4 className="font-bold text-lg mb-4">Resources</h4>
              <ul className="space-y-3 text-sm text-gray-300">
                <li><button onClick={scrollToForm} className="hover:text-white transition-colors">Download Curriculum</button></li>
                <li><button onClick={scrollToForm} className="hover:text-white transition-colors">Sample Certificate</button></li>
                <li><button onClick={scrollToForm} className="hover:text-white transition-colors">Prospectus</button></li>
                <li><button onClick={scrollToForm} className="hover:text-white transition-colors">FAQ</button></li>
                <li><button onClick={scrollToForm} className="hover:text-white transition-colors">Enrollment Guide</button></li>
              </ul>
            </div>

            {/* Contact & Policies Column */}
            <div>
              <h4 className="font-bold text-lg mb-4">Contact & Legal</h4>
              <ul className="space-y-3 text-sm text-gray-300">
                <li className="flex items-center gap-2">
                  <Mail className="w-4 h-4" />
                  <a href="mailto:info@aischoolforall.com" className="hover:text-white transition-colors">info@aischoolforall.com</a>
                </li>
                <li className="flex items-center gap-2">
                  <Phone className="w-4 h-4" />
                  <a href="tel:+15551234567" className="hover:text-white transition-colors">+1 (555) 123-4567</a>
                </li>
                <li className="flex items-start gap-2">
                  <MapPin className="w-4 h-4 mt-1" />
                  <span>Global Online Programs</span>
                </li>
              </ul>
              <div className="mt-6 pt-6 border-t border-white/10">
                <ul className="space-y-2 text-xs text-gray-400">
                  <li><a href="#" className="hover:text-white transition-colors">Privacy Policy</a></li>
                  <li><a href="#" className="hover:text-white transition-colors">Terms & Conditions</a></li>
                  <li><a href="#" className="hover:text-white transition-colors">Refund Policy</a></li>
                </ul>
              </div>
            </div>
          </div>

          <div className="pt-8 border-t border-white/10">
            <div className="flex flex-col md:flex-row justify-between items-center gap-4">
              <p className="text-sm text-gray-400 text-center md:text-left">
                © 2025 AI School for All by Workshop Library. All rights reserved.
              </p>
              <div className="flex items-center gap-3">
                <Shield className="w-5 h-5 text-yellow-400" />
                <span className="text-sm text-gray-300 font-semibold">Exemplar Global Accredited</span>
              </div>
            </div>
            <p className="text-xs text-gray-500 text-center mt-4">
              CSDGPM - Certified Specialist in Digital Growth & Performance Marketing | Professional Education & Training
            </p>
          </div>
        </div>
        </div>
      </footer>

      {/* Chatbot */}
      <Chatbot />
    </div>
  );
};

export default CourseCSDGPM;
