import React, { useState, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Textarea } from './ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select';
import { toast } from 'sonner';
import { Loader2 } from 'lucide-react';
import axios from 'axios';
import { trackFormSubmission, trackConversion, storeUtmParams, getStoredUtmParams } from '../utils/analytics';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

const EnrollmentForm = () => {
  const [formData, setFormData] = useState({
    name: '',
    email: '',
    phone: '',
    city: '',
    course: '',
    education: '',
    experience: '',
    message: ''
  });
  const [loading, setLoading] = useState(false);

  // Store UTM parameters when component mounts
  useEffect(() => {
    storeUtmParams();
  }, []);

  const courses = [
    'Certified Specialist in Digital Growth & Performance Marketing (CSDGPM)',
    'Certified Professional in Applied Machine Learning (CPAML)',
    'Certified Specialist in Machine Learning & Automation (CSMLA)',
    'Professional Certificate in Business & Data Analytics (PCBDA)',
    'Advanced Diploma in Applied Data Science & Predictive Modeling (ADADSPM)',
    'Certified Cloud Computing Specialist (CCCS)',
    'Professional Certificate in Digital Marketing Strategies (PCDMS)',
    'Workshop: Photo Editing with AI',
    'Workshop: Video Production & Editing with AI',
    'Workshop: AI-Powered Social Media Booster',
    'Workshop: LinkedIn Automation using AI',
    'Workshop: Prompt Engineering for Beginners',
    'Workshop: Canva + AI Design Secrets',
    'Workshop: AI Tools for Freelancers',
    'Workshop: Digital Ads Auto-Optimization with AI',
    'Workshop: Content Creation Mastery using AI',
    'Workshop: AI-Powered Resume + Portfolio Builder'
  ];

  const handleChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleCourseChange = (value) => {
    setFormData(prev => ({
      ...prev,
      course: value
    }));
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    
    if (!formData.name || !formData.email || !formData.phone || !formData.city || !formData.course) {
      toast.error('Please fill in all required fields');
      return;
    }

    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(formData.email)) {
      toast.error('Please enter a valid email address');
      return;
    }

    const phoneRegex = /^[0-9]{10}$/;
    if (!phoneRegex.test(formData.phone.replace(/\D/g, ''))) {
      toast.error('Please enter a valid 10-digit phone number');
      return;
    }

    setLoading(true);

    try {
      // Get stored UTM parameters
      const utmParams = getStoredUtmParams();
      
      // Combine form data with UTM parameters
      const dataToSubmit = {
        ...formData,
        ...utmParams
      };

      // Track form submission in Google Analytics
      trackFormSubmission('enrollment_form', {
        course: formData.course,
        ...utmParams
      });

      // Submit to backend
      const response = await axios.post(`${API}/enrollments`, dataToSubmit);
      
      if (response.status === 200) {
        // Track conversion in Google Analytics
        trackConversion(formData.course, 50); // $50 course value for tracking
        
        toast.success('Enrollment request submitted successfully! We will contact you within 24 hours.');
        
        setFormData({
          name: '',
          email: '',
          phone: '',
          city: '',
          course: '',
          education: '',
          experience: '',
          message: ''
        });
      }
    } catch (error) {
      console.error('Enrollment submission error:', error);
      toast.error('Failed to submit enrollment. Please try again or contact us directly.');
    } finally {
      setLoading(false);
    }
  };

  return (
    <Card className="border-0 shadow-none">
      <CardHeader className="pb-6">
        <CardTitle className="heading-3 text-3xl text-center mb-2">Enroll Now</CardTitle>
        <CardDescription className="body-medium text-center">
          Fill in your details and we&apos;ll get back to you within 24 hours
        </CardDescription>
      </CardHeader>
      <CardContent className="px-8 pb-8">
        <form onSubmit={handleSubmit} className="space-y-6">
          <div className="space-y-2">
            <Label htmlFor="name">Full Name *</Label>
            <Input
              id="name"
              name="name"
              type="text"
              placeholder="Enter your full name"
              value={formData.name}
              onChange={handleChange}
              required
              className="border-gray-300 h-12 rounded-xl"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="email">Email Address *</Label>
            <Input
              id="email"
              name="email"
              type="email"
              placeholder="your.email@example.com"
              value={formData.email}
              onChange={handleChange}
              required
              className="border-gray-300 h-12 rounded-xl"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="phone">Phone Number *</Label>
            <Input
              id="phone"
              name="phone"
              type="tel"
              placeholder="10-digit mobile number"
              value={formData.phone}
              onChange={handleChange}
              required
              className="border-gray-300 h-12 rounded-xl"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="city">City *</Label>
            <Input
              id="city"
              name="city"
              type="text"
              placeholder="Enter your city"
              value={formData.city}
              onChange={handleChange}
              required
              className="border-gray-300 h-12 rounded-xl"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="course">Select Course *</Label>
            <Select value={formData.course} onValueChange={handleCourseChange} required>
              <SelectTrigger className="border-gray-300 h-12 rounded-xl">
                <SelectValue placeholder="Choose a course" />
              </SelectTrigger>
              <SelectContent>
                {courses.map((course) => (
                  <SelectItem key={course} value={course}>
                    {course}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div className="space-y-2">
            <Label htmlFor="education">Educational Background</Label>
            <Input
              id="education"
              name="education"
              type="text"
              placeholder="e.g., B.Tech, MBA, 12th Pass"
              value={formData.education}
              onChange={handleChange}
              className="border-gray-300 h-12 rounded-xl"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="experience">Work Experience (if any)</Label>
            <Input
              id="experience"
              name="experience"
              type="text"
              placeholder="e.g., 2 years in Marketing"
              value={formData.experience}
              onChange={handleChange}
              className="border-gray-300 h-12 rounded-xl"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="message">Additional Message (Optional)</Label>
            <Textarea
              id="message"
              name="message"
              placeholder="Any questions or specific requirements?"
              value={formData.message}
              onChange={handleChange}
              rows={4}
              className="border-gray-300 h-12 rounded-xl"
            />
          </div>

          <Button 
            type="submit" 
            className="w-full btn-primary h-12 rounded-xl text-lg font-semibold hover:shadow-lg hover:shadow-blue-500/30"
            disabled={loading}
          >
            {loading ? (
              <>
                <Loader2 className="mr-2 h-4 w-4 animate-spin" />
                Submitting...
              </>
            ) : (
              'Submit Enrollment Request'
            )}
          </Button>
        </form>
      </CardContent>
    </Card>
  );
};

export default EnrollmentForm;
