import React, { useState } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { Textarea } from './ui/textarea';
import { toast } from 'sonner';
import { Loader2, Download, Phone } from 'lucide-react';
import axios from 'axios';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

const CourseLeadForm = ({ courseName = "CSDGPM", formTitle = "Get Course Brochure" }) => {
  const [formData, setFormData] = useState({
    name: '',
    email: '',
    phone: '',
    country: '',
    course: courseName,
    message: ''
  });
  const [loading, setLoading] = useState(false);

  const handleChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleSubmit = async (e, actionType) => {
    e.preventDefault();
    
    if (!formData.name || !formData.email || !formData.phone || !formData.country) {
      toast.error('Please fill in all required fields');
      return;
    }

    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(formData.email)) {
      toast.error('Please enter a valid email address');
      return;
    }

    setLoading(true);

    try {
      const response = await axios.post(`${API}/course-leads`, {
        ...formData,
        lead_type: actionType === 'brochure' ? 'brochure_request' : 'call_request'
      });
      
      if (response.status === 200) {
        if (actionType === 'brochure') {
          toast.success('Brochure request submitted! We will email you shortly.');
        } else {
          toast.success('Call request submitted! Our team will contact you within 24 hours.');
        }
        
        setFormData({
          name: '',
          email: '',
          phone: '',
          country: '',
          course: courseName,
          message: ''
        });
      }
    } catch (error) {
      console.error('Lead submission error:', error);
      toast.error('Failed to submit request. Please try again.');
    } finally {
      setLoading(false);
    }
  };

  return (
    <Card className="border-2 border-yellow-400 shadow-xl">
      <CardHeader className="bg-gradient-to-r from-gray-100 to-gray-200 pb-6">
        <CardTitle className="heading-3 text-2xl text-center text-gray-900">{formTitle}</CardTitle>
        <CardDescription className="body-medium text-center text-gray-700">
          Fill in your details and we'll get back to you with complete course information
        </CardDescription>
      </CardHeader>
      <CardContent className="px-8 pb-8 pt-6">
        <form className="space-y-5">
          <div className="space-y-2">
            <Label htmlFor="name" className="text-gray-700 font-medium">Full Name *</Label>
            <Input
              id="name"
              name="name"
              type="text"
              placeholder="Enter your full name"
              value={formData.name}
              onChange={handleChange}
              required
              className="border-gray-300 h-12 rounded-xl focus:border-yellow-400 focus:ring-yellow-400"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="email" className="text-gray-700 font-medium">Email Address *</Label>
            <Input
              id="email"
              name="email"
              type="email"
              placeholder="your.email@example.com"
              value={formData.email}
              onChange={handleChange}
              required
              className="border-gray-300 h-12 rounded-xl focus:border-yellow-400 focus:ring-yellow-400"
            />
          </div>

          <div className="grid grid-cols-2 gap-4">
            <div className="space-y-2">
              <Label htmlFor="phone" className="text-gray-700 font-medium">Phone Number *</Label>
              <Input
                id="phone"
                name="phone"
                type="tel"
                placeholder="+1234567890"
                value={formData.phone}
                onChange={handleChange}
                required
                className="border-gray-300 h-12 rounded-xl focus:border-yellow-400 focus:ring-yellow-400"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="country" className="text-gray-700 font-medium">Country *</Label>
              <Input
                id="country"
                name="country"
                type="text"
                placeholder="Your country"
                value={formData.country}
                onChange={handleChange}
                required
                className="border-gray-300 h-12 rounded-xl focus:border-yellow-400 focus:ring-yellow-400"
              />
            </div>
          </div>

          <div className="space-y-2">
            <Label htmlFor="message" className="text-gray-700 font-medium">Message (Optional)</Label>
            <Textarea
              id="message"
              name="message"
              placeholder="Any questions or specific requirements?"
              value={formData.message}
              onChange={handleChange}
              rows={3}
              className="border-gray-300 rounded-xl focus:border-yellow-400 focus:ring-yellow-400"
            />
          </div>

          <div className="flex gap-3 pt-2">
            <Button 
              type="button"
              onClick={(e) => handleSubmit(e, 'brochure')}
              className="flex-1 bg-gradient-to-r from-yellow-400 to-yellow-500 hover:from-yellow-500 hover:to-yellow-600 text-gray-900 h-12 rounded-xl text-base font-semibold shadow-lg hover:shadow-xl transition-all"
              disabled={loading}
            >
              {loading ? (
                <>
                  <Loader2 className="mr-2 h-5 w-5 animate-spin" />
                  Submitting...
                </>
              ) : (
                <>
                  <Download className="mr-2 h-5 w-5" />
                  Get Brochure
                </>
              )}
            </Button>

            <Button 
              type="button"
              onClick={(e) => handleSubmit(e, 'call')}
              className="flex-1 bg-white border-2 border-gray-600 text-gray-800 hover:bg-gray-100 h-12 rounded-xl text-base font-semibold shadow-lg hover:shadow-xl transition-all"
              disabled={loading}
            >
              {loading ? (
                <>
                  <Loader2 className="mr-2 h-5 w-5 animate-spin" />
                  Submitting...
                </>
              ) : (
                <>
                  <Phone className="mr-2 h-5 w-5" />
                  Request a Call
                </>
              )}
            </Button>
          </div>
        </form>
      </CardContent>
    </Card>
  );
};

export default CourseLeadForm;
