import React, { useState } from 'react';
import { X, MessageCircle, Send } from 'lucide-react';
import axios from 'axios';

const Chatbot = () => {
  const [isOpen, setIsOpen] = useState(false);
  const [messages, setMessages] = useState([
    { 
      type: 'bot', 
      text: 'Hello! 👋 Welcome to AI School for All. Thank you for visiting us today!\n\nI\'m here to assist you with information about our CSDGPM certification program and workshops.\n\nTo get started and provide you with personalized assistance, may I kindly ask for your name?' 
    }
  ]);
  const [currentInput, setCurrentInput] = useState('');
  const [userInfo, setUserInfo] = useState({
    name: '',
    city: '',
    email: '',
    phone: '',
    course: 'CSDGPM'
  });
  const [step, setStep] = useState(0); // 0: name, 1: city, 2: email, 3: phone, 4: questions

  const questions = [
    "May I kindly ask for your name?",
    "Thank you! Could you please share which city you're from?",
    "Thank you for sharing! May I have your email address to send you relevant information?",
    "Much appreciated! Lastly, could you please provide your phone number so we can reach out to you?",
    "Thank you so much for your time and information! 🙏 I truly appreciate it.\n\nNow, how may I assist you today? Feel free to ask me about:\n• Course curriculum and modules\n• Certification details and benefits\n• Pricing and enrollment options\n• Program duration and schedule\n• Career outcomes and success stories"
  ];

  const courseInfo = {
    curriculum: "The CSDGPM program covers 6 comprehensive modules: Digital Growth Foundations, Performance Marketing Mastery, Content & Social Growth, Data & Analytics, E-Commerce Growth, and Strategy & Leadership. Each module includes live sessions, practical labs, and real-world projects.",
    
    certification: "You'll earn dual certification: Exemplar Global accredited CSDGPM Specialist Certification and Google Ads Certification readiness. Both are globally recognized and valued by employers worldwide.",
    
    pricing: "We offer three plans:\n• Standard: $499/₹19,999 - Complete certification program\n• Pro: $799/₹34,999 - Enhanced with mentor sessions (Most Popular)\n• Premium: $1299/₹54,999 - Ultimate with 1:1 coaching",
    
    duration: "The program is 6 months long with 150 hours of training. It includes live sessions, labs, and a capstone project. Weekly commitment is 6-8 hours.",
    
    schedule: "Next cohort starts January 2026. Application deadline is December 15, 2025. All sessions are live + recorded for flexibility.",
    
    career: "93% of our alumni achieve career growth, with an average salary increase of $40K+. You'll build a professional portfolio with real campaigns and strategies."
  };

  const handleSendMessage = async () => {
    if (!currentInput.trim()) return;

    const userMessage = currentInput.trim();
    setMessages(prev => [...prev, { type: 'user', text: userMessage }]);
    setCurrentInput('');

    // Process based on current step
    if (step === 0) {
      // Collect name
      setUserInfo(prev => ({ ...prev, name: userMessage }));
      setMessages(prev => [...prev, { 
        type: 'bot', 
        text: `It's wonderful to meet you, ${userMessage}! 😊\n\n${questions[1]}` 
      }]);
      setStep(1);
    } else if (step === 1) {
      // Collect city
      setUserInfo(prev => ({ ...prev, city: userMessage }));
      setMessages(prev => [...prev, { 
        type: 'bot', 
        text: `Thank you so much, ${userInfo.name}! It's great to connect with someone from ${userMessage}. 🌍\n\n${questions[2]}` 
      }]);
      setStep(2);
    } else if (step === 2) {
      // Collect email
      const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
      if (!emailRegex.test(userMessage)) {
        setMessages(prev => [...prev, { 
          type: 'bot', 
          text: 'I appreciate you sharing that, but it seems the email format might not be correct. Could you please provide a valid email address? (e.g., yourname@example.com)' 
        }]);
        return;
      }
      setUserInfo(prev => ({ ...prev, email: userMessage }));
      setMessages(prev => [...prev, { 
        type: 'bot', 
        text: `Perfect, thank you! ✅\n\n${questions[3]}` 
      }]);
      setStep(3);
    } else if (step === 3) {
      // Collect phone
      const phoneRegex = /^\+?[\d\s\-()]{10,}$/;
      if (!phoneRegex.test(userMessage)) {
        setMessages(prev => [...prev, { 
          type: 'bot', 
          text: 'Thank you for sharing! However, it seems the phone number format might not be correct. Could you please provide a valid phone number with at least 10 digits?' 
        }]);
        return;
      }
      setUserInfo(prev => ({ ...prev, phone: userMessage }));
      
      // Submit lead to backend
      try {
        await axios.post(`${process.env.REACT_APP_BACKEND_URL}/api/course-leads`, {
          name: userInfo.name,
          city: userInfo.city,
          email: userInfo.email,
          phone: userMessage,
          course: 'CSDGPM'
        });
      } catch (error) {
        console.error('Error submitting lead:', error);
      }

      setMessages(prev => [...prev, { 
        type: 'bot', 
        text: `Excellent, ${userInfo.name}! 🎉 Your information has been securely saved.\n\n${questions[4]}` 
      }]);
      setStep(4);
    } else {
      // Answer questions about the course
      const input = userMessage.toLowerCase();
      let response = '';

      // Check for "thanks" or conversation end
      if (input.includes('thank') || input.includes('thanks') || input.includes('thx') || input === 'bye' || input === 'goodbye' || input.includes('no more') || input.includes('that\'s all') || input.includes('that is all')) {
        response = "You're most welcome! 😊 It was truly a pleasure assisting you today.\n\nThank you so much for your interest in AI School for All. Our dedicated team will reach out to you within 24 hours with more details.\n\nIf you have any urgent questions in the meantime, please feel free to:\n📞 Call us: +1 (555) 123-4567\n📧 Email us: info@aischoolforall.com\n\nWishing you a wonderful day ahead! 🌟";
        setMessages(prev => [...prev, { type: 'bot', text: response }]);
        return;
      }

      // Check for human/agent request
      if (input.includes('human') || input.includes('agent') || input.includes('real person') || input.includes('speak to someone') || input.includes('talk to') || input.includes('connect with')) {
        response = "I completely understand! 👋 Sometimes it's great to speak with a team member directly.\n\nSince we already have your details, our team will be delighted to connect with you within 24 hours.\n\n📞 For urgent requirements, please don't hesitate to:\n• Call us: +1 (555) 123-4567\n• Email us: info@aischoolforall.com\n\nWe truly appreciate your interest and look forward to speaking with you soon! 🙏";
        setMessages(prev => [...prev, { type: 'bot', text: response }]);
        return;
      }

      if (input.includes('curriculum') || input.includes('module') || input.includes('content') || input.includes('syllabus')) {
        response = courseInfo.curriculum;
      } else if (input.includes('certif') || input.includes('credential') || input.includes('accredit')) {
        response = courseInfo.certification;
      } else if (input.includes('price') || input.includes('cost') || input.includes('fee') || input.includes('plan')) {
        response = courseInfo.pricing;
      } else if (input.includes('duration') || input.includes('how long') || input.includes('time')) {
        response = courseInfo.duration;
      } else if (input.includes('schedule') || input.includes('start') || input.includes('date') || input.includes('cohort')) {
        response = courseInfo.schedule;
      } else if (input.includes('career') || input.includes('job') || input.includes('salary') || input.includes('outcome')) {
        response = courseInfo.career;
      } else if (input.includes('enroll') || input.includes('apply') || input.includes('register')) {
        response = "That's wonderful! I'm so glad you're interested in joining us! 🎉\n\nTo enroll, simply scroll up and click any 'Enroll Now' button on this page. Our team will personally contact you within 24 hours using the details you've shared.\n\nYou're also welcome to reach out directly:\n📧 info@aischoolforall.com\n📞 +1 (555) 123-4567\n\nThank you for choosing AI School for All! We're excited to help you achieve your goals.";
      } else {
        response = "Thank you for your question! I'd be absolutely delighted to help you with that. 😊\n\nYou can ask me about:\n• Course curriculum and modules\n• Certification details and recognition\n• Pricing and payment plans\n• Program duration and schedule\n• Career outcomes and success stories\n• Enrollment process\n• Speaking with our team\n\nWhat would you like to know more about?";
      }

      setMessages(prev => [...prev, { type: 'bot', text: response }]);
    }
  };

  const handleKeyPress = (e) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      handleSendMessage();
    }
  };

  return (
    <div className="fixed bottom-6 right-6 z-50">
      {/* Chat Button */}
      {!isOpen && (
        <button
          onClick={() => setIsOpen(true)}
          className="bg-gradient-to-r from-blue-600 to-blue-700 hover:from-blue-700 hover:to-blue-800 text-white rounded-full p-4 shadow-2xl hover:shadow-3xl transition-all duration-300 hover:scale-110 flex items-center gap-2"
        >
          <MessageCircle className="w-6 h-6" />
          <span className="font-semibold pr-2">Chat with us</span>
        </button>
      )}

      {/* Chat Window */}
      {isOpen && (
        <div className="bg-white rounded-2xl shadow-2xl w-96 h-[600px] flex flex-col border-2 border-gray-200">
          {/* Header */}
          <div className="bg-gradient-to-r from-blue-600 to-blue-700 text-white p-4 rounded-t-2xl flex items-center justify-between">
            <div className="flex items-center gap-3">
              <div className="w-10 h-10 rounded-full bg-white/20 flex items-center justify-center">
                <MessageCircle className="w-6 h-6" />
              </div>
              <div>
                <h3 className="font-bold text-lg">AI School Assistant</h3>
                <p className="text-xs text-blue-100">Ask me anything about CSDGPM</p>
              </div>
            </div>
            <button
              onClick={() => setIsOpen(false)}
              className="hover:bg-white/20 rounded-full p-1 transition-all"
            >
              <X className="w-5 h-5" />
            </button>
          </div>

          {/* Messages */}
          <div className="flex-1 overflow-y-auto p-4 space-y-4 bg-gray-50">
            {messages.map((message, index) => (
              <div
                key={index}
                className={`flex ${message.type === 'user' ? 'justify-end' : 'justify-start'}`}
              >
                <div
                  className={`max-w-[80%] rounded-2xl p-3 ${
                    message.type === 'user'
                      ? 'bg-blue-600 text-white'
                      : 'bg-white text-gray-800 border border-gray-200'
                  }`}
                >
                  <p className="text-sm whitespace-pre-line leading-relaxed">{message.text}</p>
                </div>
              </div>
            ))}
          </div>

          {/* Input */}
          <div className="p-4 border-t border-gray-200 bg-white rounded-b-2xl">
            <div className="flex gap-2">
              <input
                type="text"
                value={currentInput}
                onChange={(e) => setCurrentInput(e.target.value)}
                onKeyPress={handleKeyPress}
                placeholder="Type your message..."
                className="flex-1 px-4 py-2 border border-gray-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              />
              <button
                onClick={handleSendMessage}
                disabled={!currentInput.trim()}
                className="bg-blue-600 hover:bg-blue-700 disabled:bg-gray-300 text-white rounded-xl px-4 py-2 transition-all flex items-center gap-2"
              >
                <Send className="w-4 h-4" />
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default Chatbot;
