import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from './ui/card';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { Label } from './ui/label';
import { toast } from 'sonner';
import { Loader2, CheckCircle2, Tag } from 'lucide-react';
import axios from 'axios';
import { trackFormSubmission } from '../utils/analytics';

const BACKEND_URL = process.env.REACT_APP_BACKEND_URL;
const API = `${BACKEND_URL}/api`;

const BookingForm = ({ onClose }) => {
  const [formData, setFormData] = useState({
    name: '',
    email: '',
    phone: '',
    city: '',
    promoCode: ''
  });
  const [loading, setLoading] = useState(false);
  const [promoValidated, setPromoValidated] = useState(false);
  const [discount, setDiscount] = useState(0);

  const handleChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const validatePromoCode = () => {
    const code = formData.promoCode.trim().toUpperCase();
    if (code === 'BLACKFRIDAY50') {
      setPromoValidated(true);
      setDiscount(50);
      toast.success('🎉 Promo code applied! You get 50% OFF');
    } else {
      setPromoValidated(false);
      setDiscount(0);
      toast.error('Invalid promo code. Please try again.');
    }
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    
    // Validation
    if (!formData.name || !formData.email || !formData.phone || !formData.city) {
      toast.error('Please fill in all required fields');
      return;
    }

    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(formData.email)) {
      toast.error('Please enter a valid email address');
      return;
    }

    const phoneRegex = /^[0-9]{10}$/;
    if (!phoneRegex.test(formData.phone.replace(/\D/g, ''))) {
      toast.error('Please enter a valid 10-digit phone number');
      return;
    }

    if (!promoValidated) {
      toast.error('Please apply the promo code BLACKFRIDAY50 to continue');
      return;
    }

    setLoading(true);

    try {
      // Track booking submission
      trackFormSubmission('booking_form', {
        promo_code: formData.promoCode,
        discount_applied: discount
      });

      // Submit booking to backend
      const response = await axios.post(`${API}/bookings`, {
        ...formData,
        promoCode: formData.promoCode.toUpperCase(),
        originalPrice: 20,
        discountPercent: discount,
        finalPrice: 20 * (1 - discount / 100),
        bookingDate: new Date().toISOString()
      });
      
      if (response.status === 200) {
        toast.success('🎉 Booking Confirmed! Check your email for details.');
        
        // Reset form
        setFormData({
          name: '',
          email: '',
          phone: '',
          city: '',
          promoCode: ''
        });
        setPromoValidated(false);
        setDiscount(0);

        // Close modal after 2 seconds
        setTimeout(() => {
          if (onClose) onClose();
        }, 2000);
      }
    } catch (error) {
      console.error('Booking submission error:', error);
      toast.error('Failed to complete booking. Please try again or contact support.');
    } finally {
      setLoading(false);
    }
  };

  const originalPrice = 20;
  const finalPrice = originalPrice * (1 - discount / 100);

  return (
    <Card className="border-2 border-yellow-400 shadow-2xl">
      <CardHeader className="bg-gradient-to-r from-yellow-400 to-orange-500 text-gray-900">
        <CardTitle className="text-2xl text-center font-black">
          🔥 Reserve Your Seat - Limited Offer!
        </CardTitle>
        <p className="text-center text-sm font-semibold mt-2">
          Pay just ${originalPrice} to lock in 50% OFF on the full course
        </p>
      </CardHeader>
      <CardContent className="p-6">
        <form onSubmit={handleSubmit} className="space-y-4">
          <div className="space-y-2">
            <Label htmlFor="name">Full Name *</Label>
            <Input
              id="name"
              name="name"
              type="text"
              placeholder="Enter your full name"
              value={formData.name}
              onChange={handleChange}
              required
              className="border-gray-300 h-12 rounded-xl"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="email">Email Address *</Label>
            <Input
              id="email"
              name="email"
              type="email"
              placeholder="your.email@example.com"
              value={formData.email}
              onChange={handleChange}
              required
              className="border-gray-300 h-12 rounded-xl"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="phone">Phone Number *</Label>
            <Input
              id="phone"
              name="phone"
              type="tel"
              placeholder="10-digit mobile number"
              value={formData.phone}
              onChange={handleChange}
              required
              className="border-gray-300 h-12 rounded-xl"
            />
          </div>

          <div className="space-y-2">
            <Label htmlFor="city">City *</Label>
            <Input
              id="city"
              name="city"
              type="text"
              placeholder="Enter your city"
              value={formData.city}
              onChange={handleChange}
              required
              className="border-gray-300 h-12 rounded-xl"
            />
          </div>

          {/* Promo Code Section */}
          <div className="space-y-2">
            <Label htmlFor="promoCode">Promo Code *</Label>
            <div className="flex gap-2">
              <Input
                id="promoCode"
                name="promoCode"
                type="text"
                placeholder="Enter BLACKFRIDAY50"
                value={formData.promoCode}
                onChange={handleChange}
                required
                className="border-gray-300 h-12 rounded-xl flex-1"
                disabled={promoValidated}
              />
              <Button
                type="button"
                onClick={validatePromoCode}
                disabled={promoValidated || !formData.promoCode}
                className="h-12 px-6 rounded-xl bg-blue-600 hover:bg-blue-700"
              >
                {promoValidated ? (
                  <>
                    <CheckCircle2 className="w-5 h-5 mr-2" />
                    Applied
                  </>
                ) : (
                  <>
                    <Tag className="w-5 h-5 mr-2" />
                    Apply
                  </>
                )}
              </Button>
            </div>
            {promoValidated && (
              <p className="text-sm text-green-600 font-semibold flex items-center gap-2">
                <CheckCircle2 className="w-4 h-4" />
                Promo code applied successfully! You save ${originalPrice * (discount / 100)}
              </p>
            )}
          </div>

          {/* Price Summary */}
          <div className="bg-gradient-to-br from-blue-50 to-purple-50 rounded-xl p-4 border-2 border-blue-200">
            <div className="flex justify-between items-center mb-2">
              <span className="text-gray-700 font-semibold">Original Price:</span>
              <span className={`font-bold ${discount > 0 ? 'line-through text-gray-500' : 'text-gray-900 text-xl'}`}>
                ${originalPrice}
              </span>
            </div>
            {discount > 0 && (
              <>
                <div className="flex justify-between items-center mb-2">
                  <span className="text-green-600 font-semibold">Discount ({discount}%):</span>
                  <span className="text-green-600 font-bold">-${(originalPrice * (discount / 100)).toFixed(2)}</span>
                </div>
                <div className="border-t border-gray-300 pt-2 flex justify-between items-center">
                  <span className="text-gray-900 font-black text-lg">Final Price:</span>
                  <span className="text-blue-600 font-black text-2xl">${finalPrice.toFixed(2)}</span>
                </div>
              </>
            )}
          </div>

          <Button 
            type="submit" 
            className="w-full bg-gradient-to-r from-yellow-400 to-orange-500 hover:from-yellow-500 hover:to-orange-600 text-gray-900 h-14 rounded-xl text-lg font-black hover:shadow-xl transition-all"
            disabled={loading || !promoValidated}
          >
            {loading ? (
              <>
                <Loader2 className="mr-2 h-5 w-5 animate-spin" />
                Processing...
              </>
            ) : (
              <>
                🔒 Reserve Seat for ${finalPrice.toFixed(2)}
              </>
            )}
          </Button>

          <p className="text-xs text-gray-600 text-center">
            💳 Secure Booking • 📧 Instant Confirmation • 🔒 100% Money-Back Guarantee
          </p>
        </form>
      </CardContent>
    </Card>
  );
};

export default BookingForm;
