#!/usr/bin/env python3
"""
Backend Testing Suite for Google Analytics Implementation
Tests Google Analytics integration, UTM parameter handling, and enrollment form functionality
"""

import requests
import json
import time
import sys
from datetime import datetime
import uuid

# Configuration
BACKEND_URL = "https://enrollment-booster.preview.emergentagent.com"
API_BASE = f"{BACKEND_URL}/api"

class Colors:
    GREEN = '\033[92m'
    RED = '\033[91m'
    YELLOW = '\033[93m'
    BLUE = '\033[94m'
    ENDC = '\033[0m'
    BOLD = '\033[1m'

def print_test_header(test_name):
    print(f"\n{Colors.BLUE}{Colors.BOLD}{'='*60}{Colors.ENDC}")
    print(f"{Colors.BLUE}{Colors.BOLD}Testing: {test_name}{Colors.ENDC}")
    print(f"{Colors.BLUE}{Colors.BOLD}{'='*60}{Colors.ENDC}")

def print_success(message):
    print(f"{Colors.GREEN}✅ {message}{Colors.ENDC}")

def print_error(message):
    print(f"{Colors.RED}❌ {message}{Colors.ENDC}")

def print_warning(message):
    print(f"{Colors.YELLOW}⚠️  {message}{Colors.ENDC}")

def print_info(message):
    print(f"{Colors.BLUE}ℹ️  {message}{Colors.ENDC}")

def test_backend_connectivity():
    """Test basic backend connectivity"""
    print_test_header("Backend Connectivity")
    
    try:
        response = requests.get(f"{API_BASE}/", timeout=10)
        if response.status_code == 200:
            print_success("Backend is accessible")
            print_info(f"Response: {response.json()}")
            return True
        else:
            print_error(f"Backend returned status code: {response.status_code}")
            return False
    except requests.exceptions.RequestException as e:
        print_error(f"Failed to connect to backend: {str(e)}")
        return False

def test_enrollment_api_basic():
    """Test basic enrollment API functionality"""
    print_test_header("Basic Enrollment API")
    
    # Test data without UTM parameters
    test_enrollment = {
        "name": "John Analytics Test",
        "email": "john.analytics@testdomain.com",
        "phone": "9876543210",
        "city": "Mumbai",
        "course": "Certified Specialist in Digital Growth & Performance Marketing (CSDGPM)",
        "education": "B.Tech Computer Science",
        "experience": "2 years in Digital Marketing",
        "message": "Testing enrollment API without UTM parameters"
    }
    
    try:
        response = requests.post(f"{API_BASE}/enrollments", json=test_enrollment, timeout=10)
        
        if response.status_code == 200:
            print_success("Enrollment API is working")
            enrollment_data = response.json()
            print_info(f"Created enrollment ID: {enrollment_data.get('id')}")
            return True
        else:
            print_error(f"Enrollment API failed with status: {response.status_code}")
            print_error(f"Response: {response.text}")
            return False
            
    except requests.exceptions.RequestException as e:
        print_error(f"Failed to test enrollment API: {str(e)}")
        return False

def test_enrollment_with_utm_parameters():
    """Test enrollment API with UTM parameters"""
    print_test_header("Enrollment API with UTM Parameters")
    
    # Test data with UTM parameters (simulating what frontend would send)
    test_enrollment_utm = {
        "name": "Sarah UTM Test",
        "email": "sarah.utm@testdomain.com", 
        "phone": "9876543211",
        "city": "Delhi",
        "course": "Workshop: AI-Powered Social Media Booster",
        "education": "MBA Marketing",
        "experience": "3 years in Social Media",
        "message": "Testing enrollment with UTM tracking",
        # UTM parameters that would be captured by frontend
        "utm_source": "google",
        "utm_medium": "cpc",
        "utm_campaign": "ai_workshops_2025",
        "utm_content": "social_media_ad",
        "utm_term": "ai social media course"
    }
    
    try:
        response = requests.post(f"{API_BASE}/enrollments", json=test_enrollment_utm, timeout=10)
        
        if response.status_code == 200:
            print_success("Enrollment API accepts UTM parameters")
            enrollment_data = response.json()
            print_info(f"Created enrollment ID: {enrollment_data.get('id')}")
            
            # Verify UTM parameters are stored (if backend stores them)
            if any(key.startswith('utm_') for key in enrollment_data.keys()):
                print_success("UTM parameters are being stored in backend")
                for key, value in enrollment_data.items():
                    if key.startswith('utm_'):
                        print_info(f"  {key}: {value}")
            else:
                print_warning("UTM parameters not found in response (may be stored separately)")
            
            return True
        else:
            print_error(f"Enrollment with UTM failed with status: {response.status_code}")
            print_error(f"Response: {response.text}")
            return False
            
    except requests.exceptions.RequestException as e:
        print_error(f"Failed to test UTM enrollment: {str(e)}")
        return False

def test_enrollment_retrieval():
    """Test enrollment retrieval to verify data persistence"""
    print_test_header("Enrollment Data Retrieval")
    
    try:
        response = requests.get(f"{API_BASE}/enrollments", timeout=10)
        
        if response.status_code == 200:
            enrollments = response.json()
            print_success(f"Retrieved {len(enrollments)} enrollments")
            
            # Check if our test enrollments exist
            test_enrollments = [e for e in enrollments if 'test' in e.get('name', '').lower()]
            if test_enrollments:
                print_success(f"Found {len(test_enrollments)} test enrollments")
                
                # Check for UTM data in recent enrollments
                utm_enrollments = [e for e in enrollments if any(key.startswith('utm_') for key in e.keys())]
                if utm_enrollments:
                    print_success(f"Found {len(utm_enrollments)} enrollments with UTM data")
                else:
                    print_warning("No enrollments found with UTM parameters")
            
            return True
        else:
            print_error(f"Failed to retrieve enrollments: {response.status_code}")
            return False
            
    except requests.exceptions.RequestException as e:
        print_error(f"Failed to retrieve enrollments: {str(e)}")
        return False

def test_course_leads_api():
    """Test course leads API (alternative enrollment endpoint)"""
    print_test_header("Course Leads API")
    
    test_lead = {
        "name": "Mike Analytics Lead",
        "email": "mike.lead@testdomain.com",
        "phone": "9876543212",
        "city": "Bangalore",
        "course": "Certified Professional in Applied Machine Learning (CPAML)",
        "message": "Testing course leads API for analytics integration"
    }
    
    try:
        response = requests.post(f"{API_BASE}/course-leads", json=test_lead, timeout=10)
        
        if response.status_code == 200:
            print_success("Course leads API is working")
            lead_data = response.json()
            print_info(f"Created lead ID: {lead_data.get('id')}")
            return True
        else:
            print_error(f"Course leads API failed with status: {response.status_code}")
            print_error(f"Response: {response.text}")
            return False
            
    except requests.exceptions.RequestException as e:
        print_error(f"Failed to test course leads API: {str(e)}")
        return False

def test_email_notification():
    """Test email notification system"""
    print_test_header("Email Notification System")
    
    try:
        response = requests.post(f"{API_BASE}/test-email", timeout=15)
        
        if response.status_code == 200:
            result = response.json()
            if result.get('success'):
                print_success("Email notification system is working")
                print_info(f"Message: {result.get('message')}")
            else:
                print_warning("Email system configured but may not be sending")
                print_info(f"Message: {result.get('message')}")
            return True
        else:
            print_error(f"Email test failed with status: {response.status_code}")
            return False
            
    except requests.exceptions.RequestException as e:
        print_error(f"Failed to test email system: {str(e)}")
        return False

def test_cors_configuration():
    """Test CORS configuration for frontend integration"""
    print_test_header("CORS Configuration")
    
    # Test preflight request
    headers = {
        'Origin': 'https://enrollment-booster.preview.emergentagent.com',
        'Access-Control-Request-Method': 'POST',
        'Access-Control-Request-Headers': 'Content-Type'
    }
    
    try:
        response = requests.options(f"{API_BASE}/enrollments", headers=headers, timeout=10)
        
        if response.status_code in [200, 204]:
            print_success("CORS preflight request successful")
            
            # Check CORS headers
            cors_headers = {
                'Access-Control-Allow-Origin': response.headers.get('Access-Control-Allow-Origin'),
                'Access-Control-Allow-Methods': response.headers.get('Access-Control-Allow-Methods'),
                'Access-Control-Allow-Headers': response.headers.get('Access-Control-Allow-Headers')
            }
            
            for header, value in cors_headers.items():
                if value:
                    print_info(f"{header}: {value}")
                else:
                    print_warning(f"{header}: Not set")
            
            return True
        else:
            print_error(f"CORS preflight failed with status: {response.status_code}")
            return False
            
    except requests.exceptions.RequestException as e:
        print_error(f"Failed to test CORS: {str(e)}")
        return False

def run_comprehensive_backend_tests():
    """Run all backend tests for Google Analytics integration"""
    print(f"{Colors.BOLD}{Colors.BLUE}")
    print("=" * 80)
    print("GOOGLE ANALYTICS BACKEND INTEGRATION TESTING")
    print("=" * 80)
    print(f"{Colors.ENDC}")
    
    test_results = {}
    
    # Run all tests
    tests = [
        ("Backend Connectivity", test_backend_connectivity),
        ("Basic Enrollment API", test_enrollment_api_basic),
        ("UTM Parameter Handling", test_enrollment_with_utm_parameters),
        ("Data Retrieval", test_enrollment_retrieval),
        ("Course Leads API", test_course_leads_api),
        ("Email Notifications", test_email_notification),
        ("CORS Configuration", test_cors_configuration)
    ]
    
    for test_name, test_func in tests:
        try:
            result = test_func()
            test_results[test_name] = result
            time.sleep(1)  # Brief pause between tests
        except Exception as e:
            print_error(f"Test '{test_name}' crashed: {str(e)}")
            test_results[test_name] = False
    
    # Summary
    print(f"\n{Colors.BOLD}{Colors.BLUE}{'='*60}{Colors.ENDC}")
    print(f"{Colors.BOLD}{Colors.BLUE}TEST SUMMARY{Colors.ENDC}")
    print(f"{Colors.BOLD}{Colors.BLUE}{'='*60}{Colors.ENDC}")
    
    passed = sum(1 for result in test_results.values() if result)
    total = len(test_results)
    
    for test_name, result in test_results.items():
        status = "PASS" if result else "FAIL"
        color = Colors.GREEN if result else Colors.RED
        print(f"{color}{status:6} | {test_name}{Colors.ENDC}")
    
    print(f"\n{Colors.BOLD}Overall: {passed}/{total} tests passed{Colors.ENDC}")
    
    if passed == total:
        print(f"{Colors.GREEN}{Colors.BOLD}🎉 ALL TESTS PASSED - Backend ready for Google Analytics integration!{Colors.ENDC}")
        return True
    else:
        print(f"{Colors.RED}{Colors.BOLD}❌ {total - passed} tests failed - Issues need to be resolved{Colors.ENDC}")
        return False

if __name__ == "__main__":
    success = run_comprehensive_backend_tests()
    sys.exit(0 if success else 1)