from fastapi import FastAPI, APIRouter, HTTPException
from dotenv import load_dotenv
from starlette.middleware.cors import CORSMiddleware
from motor.motor_asyncio import AsyncIOMotorClient
import os
import logging
from pathlib import Path
from pydantic import BaseModel, Field, ConfigDict, EmailStr
from typing import List, Optional
import uuid
from datetime import datetime, timezone
from models import Enrollment, EnrollmentCreate
from course_models import CourseLead, CourseLeadCreate
from booking_models import Booking, BookingCreate
import smtplib
from email.mime.text import MIMEText
from email.mime.multipart import MIMEMultipart
from sendgrid import SendGridAPIClient
from sendgrid.helpers.mail import Mail


ROOT_DIR = Path(__file__).parent
load_dotenv(ROOT_DIR / '.env')

# MongoDB connection with error handling
mongo_url = os.environ.get('MONGO_URL', 'mongodb://localhost:27017')
db_name = os.environ.get('DB_NAME', 'test_database')

# Configure logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

try:
    client = AsyncIOMotorClient(
        mongo_url,
        serverSelectionTimeoutMS=5000,  # 5 second timeout
        connectTimeoutMS=5000,
        socketTimeoutMS=5000
    )
    db = client[db_name]
    logger.info(f"MongoDB client initialized with database: {db_name}")
except Exception as e:
    logger.error(f"Failed to initialize MongoDB client: {str(e)}")
    # Create client anyway, will be tested on startup
    client = AsyncIOMotorClient(mongo_url)
    db = client[db_name]

# Create the main app without a prefix
app = FastAPI()

# Create a router with the /api prefix
api_router = APIRouter(prefix="/api")


# Define Models
class StatusCheck(BaseModel):
    model_config = ConfigDict(extra="ignore")  # Ignore MongoDB's _id field
    
    id: str = Field(default_factory=lambda: str(uuid.uuid4()))
    client_name: str
    timestamp: datetime = Field(default_factory=lambda: datetime.now(timezone.utc))

class StatusCheckCreate(BaseModel):
    client_name: str

# Email notification function
async def send_email_notification(subject: str, body: str, to_email: str = "shrey@workshoplibrary.com"):
    """Send email notification for form submissions using SendGrid"""
    try:
        # Check if SendGrid API key is configured
        sendgrid_api_key = os.environ.get('SENDGRID_API_KEY', '')
        from_email = os.environ.get('SENDGRID_FROM_EMAIL', 'shrey@workshoplibrary.com')
        
        if sendgrid_api_key:
            # Use SendGrid API
            logger.info(f"📧 Sending email via SendGrid to {to_email}...")
            message = Mail(
                from_email=from_email,
                to_emails=to_email,
                subject=subject,
                plain_text_content=body
            )
            
            sg = SendGridAPIClient(sendgrid_api_key)
            response = sg.send(message)
            
            logger.info(f"✅ Email sent successfully via SendGrid to {to_email} (Status: {response.status_code})")
            return True
        else:
            # Fallback to SMTP if SendGrid not configured
            logger.info(f"📧 SendGrid not configured, trying SMTP...")
            
            msg = MIMEMultipart()
            msg['From'] = os.environ.get('SMTP_FROM_EMAIL', 'noreply@aischoolforall.com')
            msg['To'] = to_email
            msg['Subject'] = subject
            msg.attach(MIMEText(body, 'plain'))
            
            smtp_host = os.environ.get('SMTP_HOST', 'smtp.gmail.com')
            smtp_port = int(os.environ.get('SMTP_PORT', '587'))
            smtp_username = os.environ.get('SMTP_USERNAME', '')
            smtp_password = os.environ.get('SMTP_PASSWORD', '')
            
            if not smtp_username or not smtp_password:
                logger.info(f"⚠️  Email not configured. Logging only:")
                logger.info(f"To: {to_email}\nSubject: {subject}\nBody:\n{body}")
                return True
            
            with smtplib.SMTP(smtp_host, smtp_port) as server:
                server.starttls()
                server.login(smtp_username, smtp_password)
                server.send_message(msg)
            
            logger.info(f"✅ Email sent successfully via SMTP to {to_email}")
            return True
            
    except Exception as e:
        logger.error(f"❌ Error sending email: {str(e)}")
        logger.info(f"Email Content:\nTo: {to_email}\nSubject: {subject}\n\n{body}")
        return False

# Add your routes to the router instead of directly to app
@api_router.get("/")
async def root():
    return {"message": "Hello World"}

@api_router.post("/test-email")
async def test_email():
    """Test email notification functionality"""
    subject = "Test Email from AI School for All"
    body = """
This is a test email to verify email notification setup.

If you receive this, email notifications are working correctly!

Test Details:
- From: AI School Backend
- To: madhurkhera03@gmail.com
- Time: Automated test

Best regards,
AI School System
    """
    success = await send_email_notification(subject, body)
    return {
        "success": success,
        "message": "Email sent successfully" if success else "Email logged (SMTP not configured)",
        "note": "Check backend logs or your email inbox"
    }

@api_router.post("/status", response_model=StatusCheck)
async def create_status_check(input: StatusCheckCreate):
    status_dict = input.model_dump()
    status_obj = StatusCheck(**status_dict)
    
    # Convert to dict and serialize datetime to ISO string for MongoDB
    doc = status_obj.model_dump()
    doc['timestamp'] = doc['timestamp'].isoformat()
    
    _ = await db.status_checks.insert_one(doc)
    return status_obj

@api_router.get("/status", response_model=List[StatusCheck])
async def get_status_checks():
    # Exclude MongoDB's _id field from the query results
    status_checks = await db.status_checks.find({}, {"_id": 0}).to_list(1000)
    
    # Convert ISO string timestamps back to datetime objects
    for check in status_checks:
        if isinstance(check['timestamp'], str):
            check['timestamp'] = datetime.fromisoformat(check['timestamp'])
    
    return status_checks

# Enrollment endpoints
@api_router.post("/enrollments", response_model=Enrollment)
async def create_enrollment(enrollment_data: EnrollmentCreate):
    """Create a new enrollment request"""
    try:
        enrollment = Enrollment(**enrollment_data.model_dump())
        
        # Convert to dict and serialize datetime for MongoDB
        doc = enrollment.model_dump()
        doc['created_at'] = doc['created_at'].isoformat()
        
        await db.enrollments.insert_one(doc)
        
        # Send email notification
        email_subject = f"New Enrollment: {enrollment.course}"
        email_body = f"""
New Enrollment Received:

Name: {enrollment.name}
Email: {enrollment.email}
Phone: {enrollment.phone}
Course: {enrollment.course}
Timestamp: {enrollment.created_at}

Please follow up with the student.
        """
        await send_email_notification(email_subject, email_body)
        
        logger.info(f"New enrollment created: {enrollment.name} for course {enrollment.course}")
        return enrollment
    except Exception as e:
        logger.error(f"Error creating enrollment: {str(e)}")
        raise HTTPException(status_code=500, detail="Failed to create enrollment")

@api_router.get("/enrollments", response_model=List[Enrollment])
async def get_enrollments():
    """Get all enrollment requests (admin endpoint)"""
    try:
        enrollments = await db.enrollments.find({}, {"_id": 0}).to_list(1000)
        
        # Convert ISO string timestamps back to datetime objects
        for enrollment in enrollments:
            if isinstance(enrollment.get('created_at'), str):
                enrollment['created_at'] = datetime.fromisoformat(enrollment['created_at'])
        
        return enrollments
    except Exception as e:
        logger.error(f"Error fetching enrollments: {str(e)}")
        raise HTTPException(status_code=500, detail="Failed to fetch enrollments")

@api_router.get("/enrollments/{enrollment_id}", response_model=Enrollment)
async def get_enrollment(enrollment_id: str):
    """Get a specific enrollment by ID"""
    enrollment = await db.enrollments.find_one({"id": enrollment_id}, {"_id": 0})
    
    if not enrollment:
        raise HTTPException(status_code=404, detail="Enrollment not found")
    
    if isinstance(enrollment.get('created_at'), str):
        enrollment['created_at'] = datetime.fromisoformat(enrollment['created_at'])
    
    return enrollment

# Course Lead endpoints
@api_router.post("/course-leads", response_model=CourseLead)
async def create_course_lead(lead_data: CourseLeadCreate):
    """Create a new course inquiry lead"""
    try:
        lead = CourseLead(**lead_data.model_dump())
        
        doc = lead.model_dump()
        doc['created_at'] = doc['created_at'].isoformat()
        
        await db.course_leads.insert_one(doc)
        
        # Send email notification
        email_subject = f"New Course Inquiry: {lead.course}"
        email_body = f"""
New Course Inquiry Received:

Name: {lead.name}
Email: {lead.email}
Phone: {lead.phone}
City: {lead.city}
Course: {lead.course}
Message: {lead.message if lead.message else 'N/A'}
Timestamp: {lead.created_at}

Please follow up with the prospective student within 24 hours.
        """
        await send_email_notification(email_subject, email_body)
        
        logger.info(f"New course lead: {lead.name} for {lead.course}")
        return lead
    except Exception as e:
        logger.error(f"Error creating course lead: {str(e)}")
        raise HTTPException(status_code=500, detail="Failed to submit inquiry")

@api_router.get("/course-leads", response_model=List[CourseLead])
async def get_course_leads():
    """Get all course leads (admin endpoint)"""
    try:
        leads = await db.course_leads.find({}, {"_id": 0}).to_list(1000)
        
        for lead in leads:
            if isinstance(lead.get('created_at'), str):
                lead['created_at'] = datetime.fromisoformat(lead['created_at'])
        
        return leads
    except Exception as e:
        logger.error(f"Error fetching course leads: {str(e)}")
        raise HTTPException(status_code=500, detail="Failed to fetch leads")

@api_router.post("/bookings", response_model=Booking)
async def create_booking(booking_input: BookingCreate):
    """Create a new slot booking with promo code validation"""
    try:
        # Validate promo code
        valid_promo_codes = {"BLACKFRIDAY50": 50}
        promo_code_upper = booking_input.promoCode.upper()
        
        if promo_code_upper not in valid_promo_codes:
            raise HTTPException(status_code=400, detail="Invalid promo code")
        
        # Verify discount calculation
        expected_discount = valid_promo_codes[promo_code_upper]
        if booking_input.discountPercent != expected_discount:
            raise HTTPException(status_code=400, detail="Discount mismatch")
        
        # Create booking object
        booking_dict = booking_input.model_dump()
        booking_dict['bookingDate'] = datetime.fromisoformat(booking_input.bookingDate)
        booking_obj = Booking(**booking_dict)
        
        # Save to database
        doc = booking_obj.model_dump()
        doc['bookingDate'] = doc['bookingDate'].isoformat()
        doc['created_at'] = doc['created_at'].isoformat()
        
        await db.bookings.insert_one(doc)
        
        # Send confirmation email to student
        student_email_subject = "🎉 Seat Reserved! Your CSDGPM Course Booking Confirmation"
        student_email_body = f"""
Dear {booking_obj.name},

Congratulations! Your seat has been successfully reserved for the Certified Specialist in Digital Growth & Performance Marketing (CSDGPM) program.

BOOKING DETAILS:
================
Name: {booking_obj.name}
Email: {booking_obj.email}
Phone: {booking_obj.phone}
City: {booking_obj.city}

PAYMENT SUMMARY:
================
Original Price: ${booking_obj.originalPrice}
Promo Code Applied: {booking_obj.promoCode}
Discount: {booking_obj.discountPercent}% OFF
Amount Paid: ${booking_obj.finalPrice}

WHAT'S NEXT?
============
✅ Your seat is now confirmed for the next batch starting December 15, 2025
✅ You've locked in the 50% OFF discount on the full course ($299 instead of $599)
✅ We'll send you course joining details 7 days before the start date
✅ Access to exclusive pre-course materials will be shared via email

PROGRAM HIGHLIGHTS:
==================
• 6-Month comprehensive training program
• Exemplar Global Certification (globally recognized)
• 200+ hours of live + recorded sessions
• Hands-on projects and real-world case studies
• 100% Placement Assistance with resume building & mock interviews
• Lifetime community access

Need Help?
If you have any questions, feel free to reach out to us at support@aischoolforall.com

We're excited to have you join our community of 30,000+ successful digital marketers!

Best regards,
AI School for All Team
Powered by Workshop Library

---
This is an automated confirmation email. Please do not reply to this email.
        """
        await send_email_notification(student_email_subject, student_email_body, booking_obj.email)
        
        # Send notification to admin
        admin_email_subject = f"🎯 New Booking: {booking_obj.name} - CSDGPM Course"
        admin_email_body = f"""
New slot booking received for CSDGPM Course!

STUDENT DETAILS:
================
Name: {booking_obj.name}
Email: {booking_obj.email}
Phone: {booking_obj.phone}
City: {booking_obj.city}

BOOKING DETAILS:
================
Booking ID: {booking_obj.id}
Promo Code: {booking_obj.promoCode}
Amount Paid: ${booking_obj.finalPrice}
Booking Date: {booking_obj.bookingDate}
Status: {booking_obj.status}

Please ensure the student receives all pre-course materials and batch joining details.
        """
        await send_email_notification(admin_email_subject, admin_email_body)
        
        logger.info(f"New booking: {booking_obj.name} - ${booking_obj.finalPrice} paid")
        return booking_obj
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error creating booking: {str(e)}")
        raise HTTPException(status_code=500, detail="Failed to process booking")

@api_router.get("/bookings", response_model=List[Booking])
async def get_bookings():
    """Get all bookings (admin endpoint)"""
    try:
        bookings = await db.bookings.find({}, {"_id": 0}).to_list(1000)
        
        for booking in bookings:
            if isinstance(booking.get('bookingDate'), str):
                booking['bookingDate'] = datetime.fromisoformat(booking['bookingDate'])
            if isinstance(booking.get('created_at'), str):
                booking['created_at'] = datetime.fromisoformat(booking['created_at'])
        
        return bookings
    except Exception as e:
        logger.error(f"Error fetching bookings: {str(e)}")
        raise HTTPException(status_code=500, detail="Failed to fetch bookings")

# Include the router in the main app
app.include_router(api_router)

app.add_middleware(
    CORSMiddleware,
    allow_credentials=True,
    allow_origins=os.environ.get('CORS_ORIGINS', '*').split(','),
    allow_methods=["*"],
    allow_headers=["*"],
)

@app.on_event("startup")
async def startup_db_client():
    """Test MongoDB connection on startup"""
    try:
        await client.admin.command('ping')
        logger.info("✅ MongoDB connection successful")
        logger.info(f"Connected to database: {db_name}")
    except Exception as e:
        logger.error(f"❌ MongoDB connection failed: {str(e)}")
        logger.error("Application will continue but database operations will fail")

@app.on_event("shutdown")
async def shutdown_db_client():
    """Close MongoDB connection on shutdown"""
    try:
        client.close()
        logger.info("MongoDB connection closed")
    except Exception as e:
        logger.error(f"Error closing MongoDB connection: {str(e)}")